/*
 * Copyright 2024 the original author or authors.
 * <p>
 * Licensed under the Moderne Source Available License (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * <p>
 * https://docs.moderne.io/licensing/moderne-source-available-license
 * <p>
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.openrewrite.staticanalysis;

import org.jspecify.annotations.Nullable;
import org.openrewrite.*;
import org.openrewrite.java.JavaIsoVisitor;
import org.openrewrite.java.style.Checkstyle;
import org.openrewrite.java.style.HiddenFieldStyle;
import org.openrewrite.java.tree.J;
import org.openrewrite.java.tree.JavaSourceFile;

import java.util.Set;

import static java.util.Collections.singleton;
import static java.util.Objects.requireNonNull;

@Incubating(since = "7.6.0")
public class HiddenField extends Recipe {
    @Override
    public String getDisplayName() {
        return "Hidden field";
    }

    @Override
    public String getDescription() {
        return "Refactor local variables or parameters which shadow a field defined in the same class.";
    }

    @Override
    public Set<String> getTags() {
        return singleton("RSPEC-S1117");
    }

    @Override
    public TreeVisitor<?, ExecutionContext> getVisitor() {
        return new HiddenFieldFromCompilationUnitStyle();
    }

    private static class HiddenFieldFromCompilationUnitStyle extends JavaIsoVisitor<ExecutionContext> {
        @Override
        public J visit(@Nullable Tree tree, ExecutionContext ctx) {
            if (tree instanceof JavaSourceFile) {
                JavaSourceFile cu = (JavaSourceFile) requireNonNull(tree);
                HiddenFieldStyle style = cu.getStyle(HiddenFieldStyle.class);
                if (style == null) {
                    style = Checkstyle.hiddenFieldStyle();
                }
                return new HiddenFieldVisitor<>(style).visit(cu, ctx);
            }
            return (J) tree;
        }
    }
}
