/*
 * Copyright 2020 the original author or authors.
 * <p>
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * <p>
 * https://www.apache.org/licenses/LICENSE-2.0
 * <p>
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.openrewrite.java.search

import org.assertj.core.api.Assertions.assertThat
import org.junit.jupiter.api.Test
import org.openrewrite.Issue
import org.openrewrite.java.JavaParser
import org.openrewrite.java.JavaRecipeTest

interface FindAnnotationsTest : JavaRecipeTest {
    companion object {
        const val foo = """
            package com.netflix.foo;
            public @interface Foo {
                String bar();
                String baz();
            }
        """
    }

    @Issue("https://github.com/openrewrite/rewrite/issues/357")
    @Test
    fun matchesClassArgument(jp: JavaParser.Builder<*, *>) = assertChanged(
        jp.classpath("junit-jupiter-api").build(),
        recipe = FindAnnotations("@org.junit.jupiter.api.extension.ExtendWith(org.openrewrite.MyExtension.class)"),
        before = """
            import org.junit.jupiter.api.extension.ExtendWith;
            import org.openrewrite.MyExtension;
            @ExtendWith(MyExtension.class) public class A {}
            @ExtendWith({MyExtension.class}) class B {}
            @ExtendWith(value = MyExtension.class) class C {}
            @ExtendWith(value = {MyExtension.class}) class D {}
        """,
        after = """
            import org.junit.jupiter.api.extension.ExtendWith;
            import org.openrewrite.MyExtension;
            /*~~>*/@ExtendWith(MyExtension.class) public class A {}
            /*~~>*/@ExtendWith({MyExtension.class}) class B {}
            /*~~>*/@ExtendWith(value = MyExtension.class) class C {}
            /*~~>*/@ExtendWith(value = {MyExtension.class}) class D {}
        """,
        dependsOn = arrayOf(
            """
                package org.openrewrite;
                import org.junit.jupiter.api.extension.Extension;
                public class MyExtension implements Extension {}
            """
        )
    )

    @Test
    fun matchesSimpleFullyQualifiedAnnotation(jp: JavaParser) = assertChanged(
        jp,
        recipe = FindAnnotations("@java.lang.Deprecated"),
        before = "@Deprecated public class A {}",
        after = "/*~~>*/@Deprecated public class A {}"
    )

    @Test
    fun matchesAnnotationOnMethod(jp: JavaParser) = assertChanged(
        jp,
        recipe = FindAnnotations("@java.lang.Deprecated"),
        before = """
            public class A {
                @Deprecated
                public void foo() {}
            }
        """,
        after = """
            public class A {
                /*~~>*/@Deprecated
                public void foo() {}
            }
        """
    )

    @Test
    fun matchesAnnotationOnField(jp: JavaParser) = assertChanged(
        jp,
        recipe = FindAnnotations("@java.lang.Deprecated"),
        before = """
            public class A {
                @Deprecated String s;
            }
        """,
        after = """
            public class A {
                /*~~>*/@Deprecated String s;
            }
        """
    )

    @Test
    fun doesNotMatchNotFullyQualifiedAnnotations(jp: JavaParser) = assertUnchanged(
        jp,
        recipe = FindAnnotations("@Deprecated"),
        before = "@Deprecated public class A {}"
    )

    @Test
    fun matchesSingleAnnotationParameter(jp: JavaParser) = assertChanged(
        jp,
        recipe = FindAnnotations("""@java.lang.SuppressWarnings("deprecation")"""),
        before = "@SuppressWarnings(\"deprecation\") public class A {}",
        after = "/*~~>*/@SuppressWarnings(\"deprecation\") public class A {}"
    )

    @Test
    fun doesNotMatchDifferentSingleAnnotationParameter(jp: JavaParser) = assertUnchanged(
        jp,
        recipe = FindAnnotations("""@java.lang.SuppressWarnings("foo")"""),
        before = "@SuppressWarnings(\"deprecation\") public class A {}"
    )

    @Test
    fun matchesNamedParameters(jp: JavaParser) = assertChanged(
        jp,
        recipe = FindAnnotations("""@com.netflix.foo.Foo(bar="quux",baz="bar")"""),
        before = """
            import com.netflix.foo.Foo;
            @Foo(bar="quux", baz="bar")
            public class A {}
        """,
        after = """
            import com.netflix.foo.Foo;
            /*~~>*/@Foo(bar="quux", baz="bar")
            public class A {}
        """,
        dependsOn = arrayOf(foo)
    )

    @Test
    fun doesNotMatchDifferentNamedParameters(jp: JavaParser) = assertUnchanged(
        jp,
        recipe = FindAnnotations("""@com.netflix.foo.Foo(bar="qux",baz="baz")"""),
        before = """
            import com.netflix.foo.Foo;
            @Foo(bar="quux", baz="bar")
            public class A {} 
        """,
        dependsOn = arrayOf(foo)
    )

    @Test
    fun matchesPartialNamedParameters(jp: JavaParser) = assertChanged(
        jp,
        recipe = FindAnnotations("""@com.netflix.foo.Foo(baz="bar")"""),
        before = """
            import com.netflix.foo.Foo;
            @Foo(bar="quux", baz="bar")
            public class A {}
        """,
        after = """
            import com.netflix.foo.Foo;
            /*~~>*/@Foo(bar="quux", baz="bar")
            public class A {}
        """,
        dependsOn = arrayOf(foo)
    )

    @Issue("https://github.com/openrewrite/rewrite/issues/358")
    @Test
    fun matchesNamedParametersRegardlessOfOrder(jp: JavaParser) = assertChanged(
        jp,
        recipe = FindAnnotations("""@com.netflix.foo.Foo(baz="bar",bar="quux")"""),
        before = """
            import com.netflix.foo.Foo;
            @Foo(bar="quux", baz="bar")
            public class A {}
        """,
        after = """
            import com.netflix.foo.Foo;
            /*~~>*/@Foo(bar="quux", baz="bar")
            public class A {}
        """,
        dependsOn = arrayOf(foo)
    )

    @Suppress("NULLABILITY_MISMATCH_BASED_ON_JAVA_ANNOTATIONS")
    @Test
    fun checkValidation() {
        var recipe = FindAnnotations(null)
        var valid = recipe.validate()
        assertThat(valid.isValid).isFalse()
        assertThat(valid.failures()).hasSize(1)
        assertThat(valid.failures()[0].property).isEqualTo("annotationPattern")

        recipe = FindAnnotations("@com.netflix.foo.Foo(baz=\"bar\",bar=\"quux\")")
        valid = recipe.validate()
        assertThat(valid.isValid).isTrue()
    }

    @Issue("https://github.com/openrewrite/rewrite/issues/394")
    @Test
    fun findAnnotationWithClassTypeArgument(jp: JavaParser) {
        val fooClass = jp.parse("""
            package com.foo;
            
            import java.lang.annotation.ElementType;
            import java.lang.annotation.Inherited;
            import java.lang.annotation.Retention;
            import java.lang.annotation.RetentionPolicy;
            import java.lang.annotation.Target;
            
            @Retention(RetentionPolicy.RUNTIME)
            @Target({ElementType.TYPE})
            @Inherited
            public @interface Example { 
                Class<?> value();
            }
        """,
        """
            package com.foo;
            
            @Example(Foo.class)
            public class Foo {}
        """).find { it.classes.first().simpleName == "Foo" }!!

        val maybeExample = FindAnnotations.find(fooClass, "@com.foo.Example(com.foo.Foo.class)")
        assertThat(maybeExample).hasSize(1)
    }
}
