/*
 * Copyright 2020 the original author or authors.
 * <p>
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * <p>
 * https://www.apache.org/licenses/LICENSE-2.0
 * <p>
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.openrewrite.java.security

import org.junit.jupiter.api.Test
import org.openrewrite.Recipe
import org.openrewrite.java.JavaParser
import org.openrewrite.java.JavaRecipeTest

interface XmlParserXXEVulnerabilityTest : JavaRecipeTest {
    override val recipe: Recipe
        get() = XmlParserXXEVulnerability()

    @Test
    fun factoryIsNotVulnerable(jp: JavaParser) = assertUnchanged(
        before = """
            import javax.xml.stream.XMLInputFactory;
            import javax.xml.stream.XMLStreamReader;
            import java.io.InputStream;
            public class MyXmlReader {
                public void parseXML(InputStream input) {
                    XMLInputFactory factory = XMLInputFactory.newFactory();
                    factory.setProperty(XMLInputFactory.IS_SUPPORTING_EXTERNAL_ENTITIES, false);
                    factory.setProperty(XMLInputFactory.SUPPORT_DTD, false);
                    XMLStreamReader reader = factory.createXMLStreamReader(input);
                }
            }
        """
    )

    @Test
    fun factoryIsNotVulnerableClassBlockInitialization(jp: JavaParser) = assertUnchanged(
        before = """
            import javax.xml.stream.XMLInputFactory;
            import javax.xml.stream.XMLStreamReader;
            import java.io.InputStream;
            public class MyXmlReader {
                XMLInputFactory factory = XMLInputFactory.newFactory();
                {
                    factory.setProperty(XMLInputFactory.IS_SUPPORTING_EXTERNAL_ENTITIES, false);
                    factory.setProperty(XMLInputFactory.SUPPORT_DTD, false);
                }
                public void parseXML(InputStream input) {
                    XMLStreamReader reader = factory.createXMLStreamReader(input);
                }
            }
        """
    )

    @Test
    fun factoryIsVulnerableWithMethodInitialization() = assertChanged(
        before = """
            import javax.xml.stream.XMLInputFactory;
            import javax.xml.stream.XMLStreamReader;
            import java.io.InputStream;
            public class MyXmlReader {
                public void parseXML(InputStream input) {
                    XMLInputFactory f = XMLInputFactory.newFactory();
                    XMLStreamReader reader = f.createXMLStreamReader(input);
                }
            }
        """,
        after = """
            import javax.xml.stream.XMLInputFactory;
            import javax.xml.stream.XMLStreamReader;
            import java.io.InputStream;
            public class MyXmlReader {
                public void parseXML(InputStream input) {
                    XMLInputFactory f = XMLInputFactory.newFactory();
                    f.setProperty(XMLInputFactory.IS_SUPPORTING_EXTERNAL_ENTITIES, false);
                    f.setProperty(XMLInputFactory.SUPPORT_DTD, false);
                    XMLStreamReader reader = f.createXMLStreamReader(input);
                }
            }
        """
    )

    @Test
    fun factoryConstructorInitialization() = assertChanged(
        before = """
            import javax.xml.stream.XMLInputFactory;
            public class MyXmlReader {
                private XMLInputFactory f;
                public MyXmlReader() {
                    f = XMLInputFactory.newFactory();
                    f.setProperty(XMLInputFactory.IS_SUPPORTING_EXTERNAL_ENTITIES, false);
                }
            }
        """,
        after = """
            import javax.xml.stream.XMLInputFactory;
            public class MyXmlReader {
                private XMLInputFactory f;
                public MyXmlReader() {
                    f = XMLInputFactory.newFactory();
                    f.setProperty(XMLInputFactory.IS_SUPPORTING_EXTERNAL_ENTITIES, false);
                    f.setProperty(XMLInputFactory.SUPPORT_DTD, false);
                }
            }
        """
    )

    @Test
    fun factoryIsVulnerableWithClassBlockInitialization() = assertChanged(
        before = """
            import javax.xml.stream.XMLInputFactory;
            import javax.xml.stream.XMLStreamReader;
            import java.io.InputStream;
            public class MyXmlReader {
                XMLInputFactory factory = XMLInputFactory.newFactory();
                public void parseXML(InputStream input) {
                    XMLStreamReader reader = factory.createXMLStreamReader(input);
                }
            }
        """,
        after = """
            import javax.xml.stream.XMLInputFactory;
            import javax.xml.stream.XMLStreamReader;
            import java.io.InputStream;
            public class MyXmlReader {
                XMLInputFactory factory = XMLInputFactory.newFactory();
            
                {
                    factory.setProperty(XMLInputFactory.IS_SUPPORTING_EXTERNAL_ENTITIES, false);
                    factory.setProperty(XMLInputFactory.SUPPORT_DTD, false);
                }
                public void parseXML(InputStream input) {
                    XMLStreamReader reader = factory.createXMLStreamReader(input);
                }
            }
        """
    )

    @Test
    fun factoryNeedsDtdWithClassBlockInitialization() = assertChanged(
        before = """
            import javax.xml.stream.XMLInputFactory;
            import javax.xml.stream.XMLStreamReader;
            import java.io.InputStream;
            public class MyXmlReader {
                XMLInputFactory factory = XMLInputFactory.newFactory();
                
                {
                    factory.setProperty(XMLInputFactory.IS_SUPPORTING_EXTERNAL_ENTITIES, false);
                }
                public void parseXML(InputStream input) {
                    XMLStreamReader reader = factory.createXMLStreamReader(input);
                }
            }
        """,
        after = """
            import javax.xml.stream.XMLInputFactory;
            import javax.xml.stream.XMLStreamReader;
            import java.io.InputStream;
            public class MyXmlReader {
                XMLInputFactory factory = XMLInputFactory.newFactory();
                
                {
                    factory.setProperty(XMLInputFactory.IS_SUPPORTING_EXTERNAL_ENTITIES, false);
                    factory.setProperty(XMLInputFactory.SUPPORT_DTD, false);
                }
                public void parseXML(InputStream input) {
                    XMLStreamReader reader = factory.createXMLStreamReader(input);
                }
            }
        """
    )

    @Test
    fun factoryExternalWithClassBlockInitialization() = assertChanged(
        before = """
            import javax.xml.stream.XMLInputFactory;
            import javax.xml.stream.XMLStreamReader;
            import java.io.InputStream;
            public class MyXmlReader {
                XMLInputFactory factory = XMLInputFactory.newFactory();
                
                {
                    factory.setProperty(XMLInputFactory.SUPPORT_DTD, false);
                }
                public void parseXML(InputStream input) {
                    XMLStreamReader reader = factory.createXMLStreamReader(input);
                }
            }
        """,
        after = """
            import javax.xml.stream.XMLInputFactory;
            import javax.xml.stream.XMLStreamReader;
            import java.io.InputStream;
            public class MyXmlReader {
                XMLInputFactory factory = XMLInputFactory.newFactory();
                
                {
                    factory.setProperty(XMLInputFactory.SUPPORT_DTD, false);
                    factory.setProperty(XMLInputFactory.IS_SUPPORTING_EXTERNAL_ENTITIES, false);
                }
                public void parseXML(InputStream input) {
                    XMLStreamReader reader = factory.createXMLStreamReader(input);
                }
            }
        """
    )
}
