/*
 * OpenURP, Agile University Resource Planning Solution.
 *
 * Copyright © 2014, The OpenURP Software.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful.
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.openurp.app.security

import java.net.{URL, URLEncoder}

import org.beangle.commons.collection.Collections
import org.beangle.commons.lang.Strings
import org.beangle.commons.net.http.HttpUtils.{getText, getData}
import org.openurp.app.{Urp, UrpApp}
import org.openurp.app.util.JSON
import org.beangle.security.Securities
/**
 * @author chaostone
 */
object RemoteService {

  def getResource(resourceName: String): Option[Resource] = {
    val url = Urp.api + "/platform/security/func/" + UrpApp.name +
      "/resources/info.json?name=" +
      URLEncoder.encode(resourceName, "utf-8")
    val script = getText(url).orNull
    val r = JSON.parse(script).asInstanceOf[Map[String, _]]
    if (r.contains("id")) {
      Some(Resource(r("id").asInstanceOf[Number].intValue, r("scope").toString, r("roles").asInstanceOf[Seq[Number]].map(_.intValue).toSet))
    } else {
      None
    }
  }

  def roots: Option[Set[String]] = {
    val url = Urp.api + "/platform/user/roots.json?app=" + UrpApp.name
    getText(url) map { s =>
      val resources = Collections.newSet[String]
      resources ++= JSON.parse(s).asInstanceOf[Iterable[String]]
      resources.toSet
    }
  }

  def resources: collection.Map[String, Resource] = {
    val url = Urp.api + "/platform/security/func/" + UrpApp.name + "/resources.json"
    val resources = Collections.newMap[String, Resource]
    val resourceJsons = JSON.parse(getText(url).orNull).asInstanceOf[Iterable[Map[String, _]]]
    resourceJsons.map { r =>
      resources += (r("name").toString ->
        Resource(r("id").asInstanceOf[Number].intValue, r("scope").toString, r("roles").asInstanceOf[Iterable[Number]].map(_.intValue).toSet))
    }
    resources
  }

  def getRolePermissions(roleId: Int): Set[Int] = {
    val url = Urp.api + "/platform/security/func/" + UrpApp.name + "/permissions/role/" + roleId + ".json"
    val resources = new collection.mutable.HashSet[Int]
    resources ++= JSON.parse(getText(url).orNull).asInstanceOf[Iterable[Number]].map(n => n.intValue)
    resources.toSet
  }

  def getMenusJson(): String = {
    getText(Urp.api + "/platform/security/func/" + UrpApp.name + "/menus/user/" + Securities.user + ".json").orNull
  }

  def getDomainMenusJson(): String = {
    getText(Urp.api + "/platform/security/func/" + UrpApp.name + "/menus/user/" + Securities.user + ".json?forDomain=1").orNull
  }

  def getAppsJson(): String = {
    getText(Urp.api + "/platform/user/apps/" + Securities.user + ".json").orNull
  }

  def getOrg(): Urp.Org = {
    val json = getText(Urp.api + "/platform/config/orgs.json").orNull
    val data = JSON.parse(json).asInstanceOf[Map[String, Any]]
    val org = new Urp.Org
    data.get("id") foreach (e => org.id = e.asInstanceOf[Number].intValue)
    data.get("code") foreach (e => org.code = e.toString)
    data.get("name") foreach (e => org.name = e.toString)
    data.get("shortName") foreach (e => org.shortName = e.toString)
    data.get("logoUrl") foreach (e => org.logoUrl = e.toString)
    data.get("wwwUrl") foreach (e => org.wwwUrl = e.toString)
    org
  }
}

object Resource {
  def apply(id: Int, scope: String, roles: Set[Int]): Resource = {
    val scopes = Map("Private" -> 2, "Protected" -> 1, "Public" -> 0)
    new Resource(id, scopes(scope), roles)
  }
}

class Resource(val id: Int, val scope: Int, val roles: Set[Int]) extends Serializable {

  def matches(authorities: collection.Set[Int]): Boolean = {
    roles.exists(authorities.contains(_))
  }
}
