/*
 * OpenURP, Agile University Resource Planning Solution.
 *
 * Copyright © 2005, The OpenURP Software.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.openurp.app

import org.beangle.commons.lang.SystemInfo
import org.beangle.commons.logging.Logging
import org.beangle.commons.lang.Strings
import org.beangle.commons.io.IOs
import java.io.File

object UrpEnv extends Logging {

  def findHome(): String = {
    SystemInfo.properties.get("openurp.home").getOrElse(SystemInfo.user.home + "/.openurp")
  }

  def readConfig(location: String): Map[String, String] = {
    try {
      val configFile = new File(location)
      if (!configFile.exists) {
        Map.empty
      } else {
        IOs.readJavaProperties(configFile.toURI().toURL())
      }
    } catch {
      case e: Throwable => logger.error("Read config error", e); Map.empty
    }
  }
}

final class UrpEnv(val home: String, val properties: Map[String, String]) extends Logging {

  val base = readBase("openurp.base")

  val cas = readBase("openurp.cas")

  val portal = readBase("openurp.portal")

  val api = readBase("openurp.api")

  val webapp = readBase("openurp.webapp")

  val static = readBase("openurp.static")

  private def readBase(property: String): String = {
    var result =
      properties.get(property) match {
        case Some(v) => v
        case None =>
          if ("openurp.base" == property) {
            logger.warn("Cannot find openurp.base,using localhost/base")
            "localhost/base"
          } else {
            if (property == "openurp.webapp") this.base
            else this.base + "/" + Strings.replace(property, "openurp.", "")
          }
      }
    if (result.endsWith("/")) result = result.substring(0, result.length - 1)
    if (result.startsWith("http")) result else "http://" + result
  }

}
