/*
 * OpenURP, Agile University Resource Planning Solution.
 *
 * Copyright © 2014, The OpenURP Software.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful.
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.openurp.app.security

import org.beangle.cache.CacheManager
import org.beangle.commons.bean.Initializing
import org.beangle.commons.security.Request
import org.beangle.security.authc.DefaultAccount
import org.beangle.security.authz.Authorizer
import org.beangle.security.session.Session
import org.beangle.commons.lang.Strings
import org.beangle.commons.collection.Collections
import org.beangle.security.context.SecurityContext

/**
 * @author chaostone
 */
class RemoteAuthorizer(cacheManager: CacheManager) extends Authorizer {
  var unknownIsProtected = true
  val resources = cacheManager.getCache("security-resources", classOf[String], classOf[Resource])
  val authorities = cacheManager.getCache("security-authorities", classOf[String], classOf[collection.mutable.Set[Int]])
  val roots = cacheManager.getCache("security-roots", classOf[String], classOf[Set[String]])

  var publics = List.empty[String]

  override def isPermitted(context: SecurityContext): Boolean = {
    val resourceName = context.request.resource.toString
    val isPublic = publics.exists(x => resourceName.startsWith(x))
    if (isPublic) return true;

    if (resourceName.startsWith("/admin/")) {
      return context.root;
    }

    val rscOption = resources.get(resourceName)
    rscOption match {
      case None =>
        RemoteService.getResource(resourceName) match {
          case None => if (unknownIsProtected) context.isValid else false
          case Some(r) =>
            resources.put(resourceName, r)
            isAuthorized(context, r)
        }
      case Some(r) => isAuthorized(context, r)
    }
  }

  private def isAuthorized(context: SecurityContext, res: Resource): Boolean = {
    res.scope match {
      case 0 => true
      case 1 => context.isValid
      case _ =>
        context.session match {
          case Some(session) =>
            if (context.root) {
              true
            } else {
              val auths = authorities.get(session.id) match {
                case Some(aus) => aus
                case None =>
                  val newAuths = Collections.newSet[Int]
                  authorities.put(session.id, newAuths)
                  newAuths
              }
              if (auths.contains(res.id)) {
                true
              } else {
                val account = session.principal.asInstanceOf[DefaultAccount]
                var rs = res.matches(Strings.splitToInt(account.authorities).toSet)
                if (rs) auths.add(res.id)
                rs
              }
            }
          case None => false
        }
    }
  }

  def isRoot(user: String): Boolean = {
    roots.get("roots") match {
      case Some(rs) => rs.contains(user)
      case None => RemoteService.roots match {
        case Some(rs) =>
          roots.put("roots", rs); rs.contains(user)
        case None => false
      }
    }
  }
}
