/*
 * OpenURP, Agile University Resource Planning Solution.
 *
 * Copyright © 2014, The OpenURP Software.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful.
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.openurp.app.web.index

import java.time.LocalDate

import org.beangle.commons.lang.Numbers
import org.beangle.data.dao.{ EntityDao, OqlBuilder }
import org.beangle.security.realm.cas.{ Cas, CasConfig }
import org.beangle.webmvc.api.action.{ ActionSupport, ServletSupport }
import org.beangle.webmvc.api.annotation.{ action, mapping, param }
import org.beangle.webmvc.api.context.ActionContext
import org.beangle.webmvc.api.view.View
import org.openurp.app.web.NavContext
import org.openurp.edu.base.model.Project

/**
 * @author xinzhou
 */
@action("")
class ProjectAction extends ActionSupport with ServletSupport {
  var entityDao: EntityDao = _
  var casConfig: CasConfig = _

  @mapping("{project}")
  def project(@param("project") project: String): View = {
    val valid = get("project") match {
      case Some(i) => Numbers.isDigits(i)
      case None    => false
    }

    if (valid) {
      val p = entityDao.get(classOf[Project], project.toInt)
      var nav = NavContext.get(request)
      nav.params.put("school", p.school.id.toString)
      put("nav", nav)
      put("currentProject", p)
      put("projects", entityDao.getAll(classOf[Project]))
      forward()
    } else {
      val now = LocalDate.now
      val builder = OqlBuilder.from(classOf[Project], "p").where("p.beginOn <= :now and( p.endOn is null or p.endOn >= :now)", now).orderBy("p.code").cacheable()
      val projects = entityDao.search(builder)
      if (projects.isEmpty) throw new RuntimeException("Cannot find any valid projects")
      redirect("project", "&project=" + projects.head.id, null)
    }
  }

  def index(): View = {
    val now = LocalDate.now
    val builder = OqlBuilder.from(classOf[Project], "p").where("p.beginOn <= :now and( p.endOn is null or p.endOn >= :now)", now).orderBy("p.code").cacheable()
    val projects = entityDao.search(builder)
    if (projects.isEmpty) throw new RuntimeException("Cannot find any valid projects")
    redirect("project", "&project=" + projects.head.id, null)
  }

  def logout(): View = {
    redirect(to(Cas.cleanup(casConfig, ActionContext.current.request, ActionContext.current.response)), null)
  }
}
