/*
 * OpenURP, Agile University Resource Planning Solution.
 *
 * Copyright © 2014, The OpenURP Software.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful.
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.openurp.app.web

import java.util

import com.google.gson.Gson
import javax.servlet.http.{HttpServletRequest, HttpServletResponse}
import org.beangle.commons.web.util.CookieUtils
import org.beangle.webmvc.api.context.Params

object URPProfile {

  def check(request: HttpServletRequest, response: HttpServletResponse,
            profileJson: String, profile: URPProfile): URPProfile = {
    val profileIter = new Gson().fromJson(profileJson, classOf[util.List[util.Map[String, Any]]]).iterator()
    var finded = false
    var projectId: Int = 0
    var schoolId: Int = 0
    var currentProjectId = 0
    if (null != profile.edu) {
      currentProjectId = profile.edu.projectId
    }
    //[{properties:[{dimension:{},value:{}},{dimension:{},value:{}}]} ]
    //iterator each profile
    while (profileIter.hasNext && !finded) {
      val propertyIter = profileIter.next().get("properties").asInstanceOf[util.List[util.Map[String, Any]]].iterator()
      //iterator each property
      while (propertyIter.hasNext && !finded) {
        val propertyMap = propertyIter.next();
        val dimension = propertyMap.get("dimension").asInstanceOf[util.Map[String, Any]]
        if (dimension.get("name") == "projects") {
          val projectIter = propertyMap.get("value").asInstanceOf[util.List[util.Map[String, Any]]].iterator()
          //iterator each project
          while (projectIter.hasNext && !finded) {
            val project = projectIter.next()
            projectId = project.get("id").toString.toInt
            schoolId = project.get("schoolId").toString.toInt
            finded = currentProjectId == projectId || currentProjectId == 0
          }
        }
      }
    }
    if (profile.updateProject(projectId)) {
      profile.schoolId = schoolId
    }
    profile
  }

  def add(request: HttpServletRequest, response: HttpServletResponse, profile: URPProfile): Unit = {
    CookieUtils.addCookie(request, response, CookieName, profile.to, "/", COOKIE_AGE)
  }


  def get(request: HttpServletRequest, response: HttpServletResponse): URPProfile = {
    val p = Option(CookieUtils.getCookieValue(request, CookieName)).map(parse).getOrElse(new URPProfile)
    var newProjectId = 0
    //尝试从uri中获取project
    Params.getInt("project") foreach { pid =>
      newProjectId = pid
    }
    if (0 == newProjectId) {
      //在尝试从参数中获取?contextProjectId=id
      Params.getInt("contextProjectId") foreach { pid =>
        newProjectId = pid
      }
    }
    if (0 != newProjectId) {
      p.updateProject(newProjectId)
    }
    p
  }

  def update(request: HttpServletRequest, response: HttpServletResponse, profile: URPProfile, create: Boolean): Unit = {
    Option(CookieUtils.getCookieValue(request, CookieName)).map(parse) match {
      case Some(exists) =>
        if (exists != profile) {
          add(request, response, profile)
        }
      case None =>
        if (create) {
          add(request, response, profile)
        }
    }
  }

  private val CookieName = "URP_PROFILE"
  private val COOKIE_AGE = 60 * 60 * 24 * 7 // 7 days

  private def parse(cookieValue: String): URPProfile = {
    val gson = new Gson
    val v = gson.fromJson(cookieValue, classOf[URPProfile])
    v
  }
}

class URPProfile {
  var schoolId: Int = _
  var edu: EduProfile = _

  def hasValidProject: Boolean = {
    null != edu && edu.projectId > 0
  }

  def updateProject(newId: Int): Boolean = {
    if (null == edu) {
      edu = new EduProfile
      edu.projectId = newId
      true
    } else {
      if (edu.projectId != newId) {
        edu.projectId = newId
        true
      } else {
        false
      }
    }
  }

  override def equals(obj: Any): Boolean = {
    obj match {
      case o: URPProfile =>
        o.schoolId == this.schoolId && o.edu == this.edu
      case _ => false
    }
  }

  def to: String = {
    val gson = new Gson
    gson.toJson(this)
  }
}

class EduProfile {
  var projectId: Int = _
  var semesterId: Int = _

  override def equals(obj: Any): Boolean = {
    obj match {
      case o: EduProfile =>
        o.projectId == this.projectId && o.semesterId == this.semesterId
      case _ => false
    }
  }

}
