/*
 * OpenURP, Agile University Resource Planning Solution.
 *
 * Copyright © 2014, The OpenURP Software.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful.
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.openurp.edu.base.web.action

import java.time.LocalDate

import org.beangle.data.dao.OqlBuilder
import org.beangle.webmvc.api.annotation.action
import org.beangle.webmvc.api.view.View
import org.openurp.base.model.Department
import org.openurp.code.edu.model.AcademicLevel
import org.openurp.code.edu.model.{ ExamMode, GradingMode }
import org.openurp.edu.base.model.Course
import org.openurp.edu.base.web.action.helper.QueryHelper
import org.openurp.edu.base.code.model.CourseHourType
import org.openurp.edu.base.model.CourseHour
import org.openurp.edu.base.code.model.CourseType

@action("{project}/course")
class CourseAction extends ProjectRestfulAction[Course] {

  protected override def indexSetting(): Unit = {
    val courseTypes = findItems(classOf[CourseType])
    put("courseTypes", courseTypes)

    val departments = findItemsBySchool(classOf[Department])
    put("departments", departments)
  }

  override def getQueryBuilder(): OqlBuilder[Course] = {
    QueryHelper.addTemporalOn(super.getQueryBuilder(), getBoolean("active"))
  }

  override def editSetting(entity: Course) = {
    val courseTypes = findItems(classOf[CourseType])
    put("courseTypes", courseTypes)

    val examModes = findItems(classOf[ExamMode])
    put("examModes", examModes)

    val gradingModes = findItems(classOf[GradingMode])
    put("gradingModes", gradingModes)

    put("departments", findItemsBySchool(classOf[Department]))

    var levels = currentProject.levels.map(_.toLevel).toSet.toBuffer
    levels --= entity.levels
    put("levels", levels)

    val hourTypes = findItems(classOf[CourseHourType])
    put("hourTypes", hourTypes)

    //    var majors = findItemsByProject(classOf[Major]).toBuffer
    //    majors --= entity.majors
    //    put("majors", majors)
    //
    //    var xmajors = findItemsByProject(classOf[Major]).toBuffer
    //    xmajors --= entity.xmajors
    //    put("xmajors", xmajors)

    if (!entity.persisted) {
      entity.project = currentProject
      entity.beginOn = LocalDate.now
      entity.levels ++= levels
      levels.clear()
    }

    super.editSetting(entity)
  }

  protected override def saveAndRedirect(entity: Course): View = {
    val course = entity.asInstanceOf[Course]

    val hourTypes = findItems(classOf[CourseHourType])
    hourTypes foreach { ht =>
      val creditHour = getInt("creditHour" + ht.id)
      val week = getInt("week" + ht.id)
      course.hours find (h => h.hourType == ht) match {
        case Some(hour) =>
          if (week.isEmpty && creditHour.isEmpty) {
            course.hours -= hour
          } else {
            hour.weeks = week.getOrElse(0)
            hour.creditHours = creditHour.getOrElse(0)
          }
        case None =>
          if (!(week.isEmpty && creditHour.isEmpty)) {
            val newHour = new CourseHour()
            newHour.course = course
            newHour.hourType = ht
            newHour.weeks = week.getOrElse(0)
            newHour.creditHours = creditHour.getOrElse(0)
            course.hours += newHour
          }
      }
    }
    //    course.majors.clear()
    //    val majorIds = getAll("majorsId2nd", classOf[Long])
    //    course.majors ++= entityDao.find(classOf[Major], majorIds)
    //
    //    course.xmajors.clear()
    //    val xmajorIds = getAll("xmajorsId2nd", classOf[Long])
    //    course.xmajors ++= entityDao.find(classOf[Major], xmajorIds)

    course.levels.clear()
    val levelIds = getAll("levelId2nd", classOf[Int])
    course.levels ++= entityDao.find(classOf[AcademicLevel], levelIds)

    course.gradingModes.clear()
    val gradingModeIds = getAll("gradingModeId2nd", classOf[Int])
    course.gradingModes ++= entityDao.find(classOf[GradingMode], levelIds)

    super.saveAndRedirect(entity)
  }

}
