/*
 * OpenURP, Agile University Resource Planning Solution.
 *
 * Copyright © 2014, The OpenURP Software.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful.
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.openurp.edu.base.web.action

import org.beangle.commons.collection.Collections
import org.beangle.commons.lang.ClassLoaders
import org.beangle.data.dao.OqlBuilder
import org.beangle.data.transfer.excel.ExcelTemplateWriter
import org.beangle.data.transfer.exporter.Context
import org.beangle.data.transfer.importer.ImportListener
import org.beangle.data.transfer.importer.listener.ForeignerListener
import org.beangle.webmvc.api.annotation.{ action, mapping, param }
import org.beangle.webmvc.api.context.ActionContext
import org.beangle.webmvc.api.view.{ Status, Stream, View }
import org.openurp.base.model.{ Campus, Department }
import org.openurp.code.edu.model.EducationLevel
import org.openurp.edu.base.code.model.StdType
import org.openurp.edu.base.model.{ Direction, Instructor, Major, Squad, Student, StudentState, Teacher }
import org.openurp.edu.base.web.action.helper.QueryHelper

@action("{project}/squad")
class SquadAction extends ProjectRestfulAction[Squad] with ImportDataSupport[Squad] {

  protected override def indexSetting(): Unit = {
    put("levels", findItems(classOf[EducationLevel]))
    put("departments", findItemsBySchool(classOf[Department]))
    put("campuses", findItemsBySchool(classOf[Campus]))
  }

  override def getQueryBuilder(): OqlBuilder[Squad] = {
    QueryHelper.addTemporalOn(super.getQueryBuilder(), getBoolean("active"))
  }

  override def editSetting(entity: Squad) = {
    put("levels", findItems(classOf[EducationLevel]))
    put("departments", findItemsBySchool(classOf[Department]))
    put("campuses", findItemsBySchool(classOf[Campus]))

    val majors = findItemsByProject(classOf[Major])
    put("majors", majors)

    val directions = findItemsByProject(classOf[Direction])
    put("directions", directions)

    val stdTypes = findItems(classOf[StdType])
    put("stdTypes", stdTypes)

    val instructors = findItemsByProject(classOf[Instructor], "user.name")
    put("instructors", instructors)

    val tutors = findItemsByProject(classOf[Teacher], "user.name")
    put("tutors", tutors)

    super.editSetting(entity)
  }

  /**
   * 查看班级信息
   * @return @
   */
  @mapping(value = "{id}")
  override def info(@param("id") id: String): View = {
    val builder = OqlBuilder.from(classOf[StudentState], "studentState")
    builder.where("studentState.squad.id=:id", id.toLong)
    val studentStates = entityDao.search(builder)
    val students = Collections.newBuffer[Student]
    studentStates.foreach { studentState => students += studentState.std }
    val status = Collections.newMap[String, StudentState]
    studentStates.foreach { studentState => status.put(studentState.std.code, studentState) }
    put("students", students)
    put("status", status)
    super.info(id)
  }
  /**
   * 导出
   */
  def export(): View = {
    val list = collection.JavaConverters.asJavaCollection(entityDao.search(getQueryBuilder().limit(null)))
    val ctx = new Context
    ctx.datas.put("list", list)
    val path = ClassLoaders.getResource("template/squad.xls").get
    val response = ActionContext.current.response
    response.setContentType("application/x-excel")
    response.setHeader("Content-Disposition", "attachmentfilename=squad.xls")
    val os = response.getOutputStream()
    new ExcelTemplateWriter(path, os).write()
    os.close()
    Status.Ok
  }
  /**
   * 下载模板
   */
  def downloadSquadStdTemp: View = {
    Stream(ClassLoaders.getResourceAsStream("template/squad.xls").get, "application/vnd.ms-excel", "班级信息.xls")
  }

  /**
   * 维护班级学生
   */

  def setClassStudentForm(): View = {
    val squadId = longId("squad")
    val squad = entityDao.get(classOf[Squad], squadId)
    put("squad", squad)
    val builder = OqlBuilder.from[Student](classOf[StudentState].getName, "studentState")
    builder.select("studentState.std")
    builder.where("studentState.squad=:squad", squad)
    put("students", entityDao.search(builder))
    forward()
  }

  /**
   * 保存设置的班级学生
   *
   */
  def saveClassStudent(): View = {
    val squadId = longId("squad")
    val squades = entityDao.find(classOf[Squad], squadId)
    val squadCur = squades.head
    // 移除学生
    val removeIds = longIds("studentRemoveIds")
    val students = entityDao.find(classOf[Student], removeIds)
    val studentStates = Collections.newBuffer[StudentState]
    val ssBuilder = OqlBuilder.from(classOf[StudentState], "studentState")
    students.foreach { student =>
      ssBuilder.where("studentState.std=student", student)
      studentStates ++= entityDao.search(ssBuilder)
      studentStates.foreach { studentState =>
        val squad = studentState.squad
        studentState.squad = None
        //        entityDao.saveOrUpdate(studentState)
        entityDao.refresh(squad)
        //        entityDao.saveOrUpdate(squad)
      }
    }
    // 保存现在班级学生列表中可以看的到的所有学生
    val stdIds = longIds("student")
    if (!stdIds.isEmpty) {
      val students = entityDao.find(classOf[Student], stdIds)
      val msg = new StringBuffer()
      var errorNum = 0
      val studentStates = Collections.newBuffer[StudentState]
      val ssBuilder = OqlBuilder.from(classOf[StudentState], "studentState")
      students.foreach { student =>
        ssBuilder.where("studentState.std=student", student)
        studentStates ++= entityDao.search(ssBuilder)
        studentStates.foreach { studentState =>
          // 比较所关联学生与所关联 到的班级，若发现有一名学生的专业或者方向与班级中的专业或者方向不匹配，则不允许关联
          if (!studentState.major.equals(squadCur.major) || studentState.direction != squadCur.direction) {
            msg.append("\n失败：").append(studentState.std.code).append(" ").append(studentState.std.person.name)
            errorNum = errorNum + 1
          } else {
            val oriSquad = studentState.squad
            studentState.squad = Some(squadCur)
            entityDao.saveOrUpdate(studentState)
            if (oriSquad != null) {
              entityDao.refresh(oriSquad)
              entityDao.saveOrUpdate(oriSquad)
            }
          }
        }
        if (errorNum > 0) {
          addFlashMessage("\n不符合要求的学生 " + errorNum + "；分别是：" + msg)
        }
      }
      entityDao.refresh(squadCur)
      entityDao.saveOrUpdate(squadCur)
    }
    redirect("setClassStudentForm", "&squadId=" + squadId, "info.save.success")
  }

  protected override def importerListeners: List[_ <: ImportListener] = {
    List(new ForeignerListener(entityDao), new SquadImportListener(entityDao))
  }

  def getDeparts(): Seq[Department] = {
    val builder = OqlBuilder.from(classOf[Department])
    builder.orderBy("code")
    entityDao.search(builder)
  }

}
