/*
 * OpenURP, Agile University Resource Planning Solution.
 *
 * Copyright © 2014, The OpenURP Software.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful.
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.openurp.edu.base.web.action

import org.beangle.commons.collection.Collections
import org.beangle.commons.lang.ClassLoaders
import org.beangle.data.dao.OqlBuilder
import org.beangle.data.transfer.excel.ExcelTemplateWriter
import org.beangle.data.transfer.exporter.Context
import org.beangle.data.transfer.importer.ImportListener
import org.beangle.data.transfer.importer.listener.ForeignerListener
import org.beangle.webmvc.api.annotation.{ action, mapping, param }
import org.beangle.webmvc.api.context.ActionContext
import org.beangle.webmvc.api.view.{ Status, Stream, View }
import org.openurp.base.model.{ Campus, Department }
import org.openurp.code.edu.model.EducationLevel
import org.openurp.edu.base.code.model.StdType
import org.openurp.edu.base.model.{ Direction, Instructor, Major, Squad, Student, StudentState, Teacher }
import org.openurp.edu.base.web.action.helper.QueryHelper
import org.openurp.app.Urp
import org.beangle.commons.codec.digest.Digests

@action("{project}/squad")
class SquadAction extends ProjectRestfulAction[Squad] with ImportDataSupport[Squad] {

  protected override def indexSetting(): Unit = {
    put("levels", findItems(classOf[EducationLevel]))
    put("departments", findItemsBySchool(classOf[Department]))
    put("campuses", findItemsBySchool(classOf[Campus]))
  }

  override def getQueryBuilder(): OqlBuilder[Squad] = {
    QueryHelper.addTemporalOn(super.getQueryBuilder(), getBoolean("active"))
  }

  override def editSetting(entity: Squad) = {
    put("levels", findItems(classOf[EducationLevel]))
    put("departments", findItemsBySchool(classOf[Department]))
    put("campuses", findItemsBySchool(classOf[Campus]))

    val majors = findItemsByProject(classOf[Major])
    put("majors", majors)

    val directions = findItemsByProject(classOf[Direction])
    put("directions", directions)

    val stdTypes = findItems(classOf[StdType])
    put("stdTypes", stdTypes)

    val instructors = findItemsByProject(classOf[Instructor], "user.name")
    put("instructors", instructors)

    val tutors = findItemsByProject(classOf[Teacher], "user.name")
    put("tutors", tutors)

    super.editSetting(entity)
  }

  /**
   * 查看班级信息
   * @return @
   */
  @mapping(value = "{id}")
  override def info(@param("id") id: String): View = {
    val builder = OqlBuilder.from(classOf[StudentState], "studentState")
    builder.where("studentState.squad.id=:id", id.toLong)
    val studentStates = entityDao.search(builder)
    val students = Collections.newBuffer[Student]
    studentStates.foreach { studentState => students += studentState.std }
    val status = Collections.newMap[String, StudentState]
    studentStates.foreach { studentState => status.put(studentState.std.code, studentState) }
    put("students", students)
    put("status", status)
    put("urp", Urp)
    put("md5", Md5)
    super.info(id)
  }

  object Md5 {
    def digest(a: String): String = {
      Digests.md5Hex(a)
    }
  }
  /**
   * 导出
   */
  override def export(): View = {
    val list = collection.JavaConverters.asJavaCollection(entityDao.search(getQueryBuilder().limit(null)))
    val ctx = new Context
    ctx.datas.put("list", list)
    val path = ClassLoaders.getResource("template/squad.xls").get
    val response = ActionContext.current.response
    response.setContentType("application/x-excel")
    response.setHeader("Content-Disposition", "attachmentfilename=squad.xls")
    val os = response.getOutputStream()
    new ExcelTemplateWriter(path, os).write()
    os.close()
    Status.Ok
  }
  /**
   * 下载模板
   */
  def downloadSquadStdTemp: View = {
    Stream(ClassLoaders.getResourceAsStream("template/squad.xls").get, "application/vnd.ms-excel", "班级信息.xls")
  }

  protected override def importerListeners: List[_ <: ImportListener] = {
    List(new ForeignerListener(entityDao), new SquadImportListener(entityDao))
  }

  def getDeparts(): Seq[Department] = {
    val builder = OqlBuilder.from(classOf[Department])
    builder.orderBy("code")
    entityDao.search(builder)
  }

}
