/*
 * OpenURP, Agile University Resource Planning Solution.
 *
 * Copyright © 2014, The OpenURP Software.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful.
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.openurp.edu.base.web.action.project

import org.beangle.commons.codec.digest.Digests
import org.beangle.commons.collection.Collections
import org.beangle.commons.lang.ClassLoaders
import org.beangle.data.dao.OqlBuilder
import org.beangle.data.transfer.importer.ImportSetting
import org.beangle.data.transfer.importer.listener.ForeignerListener
import org.beangle.ems.app.Ems
import org.beangle.webmvc.api.annotation.{mapping, param}
import org.beangle.webmvc.api.view.{Stream, View}
import org.openurp.base.model.{Campus, Department}
import org.openurp.code.edu.model.EducationLevel
import org.openurp.edu.base.code.model.StdType
import org.openurp.edu.base.model._
import org.openurp.edu.base.web.action.ProjectRestfulAction
import org.openurp.edu.base.web.helper.{QueryHelper, SquadImportListener}

class SquadAction extends ProjectRestfulAction[Squad] {

  protected override def indexSetting(): Unit = {
    put("levels", getCodes(classOf[EducationLevel]))
    put("departments", findInSchool(classOf[Department]))
    put("campuses", findInSchool(classOf[Campus]))
  }

  override def getQueryBuilder: OqlBuilder[Squad] = {
    QueryHelper.addTemporalOn(super.getQueryBuilder, getBoolean("active"))
  }

  override def editSetting(entity: Squad) = {
    put("levels", getCodes(classOf[EducationLevel]))
    put("departments", findInSchool(classOf[Department]))
    put("campuses", findInSchool(classOf[Campus]))

    val majors = findInProject(classOf[Major])
    put("majors", majors)
    val directions = findInProject(classOf[Direction])
    put("directions", directions)
    val stdTypes = getCodes(classOf[StdType])
    put("stdTypes", stdTypes)

    super.editSetting(entity)
    put("project",getProject)
    put("urp",Ems)
  }

  /**
   * 查看班级信息
   *
   * @return @
   */
  @mapping(value = "{id}")
  override def info(@param("id") id: String): View = {
    val builder = OqlBuilder.from(classOf[StudentState], "studentState")
    builder.where("studentState.squad.id=:id", id.toLong)
    val studentStates = entityDao.search(builder)
    val students = Collections.newBuffer[Student]
    studentStates.foreach { studentState => students += studentState.std }
    val status = Collections.newMap[String, StudentState]
    studentStates.foreach { studentState => status.put(studentState.std.code, studentState) }
    put("students", students)
    put("status", status)
    put("urp", Ems)
    put("md5", Md5)
    super.info(id)
  }

  object Md5 {
    def digest(a: String): String = {
      Digests.md5Hex(a)
    }
  }

  /**
   * 下载模板
   */
  def downloadSquadStdTemp: View = {
    Stream(ClassLoaders.getResourceAsStream("template/squad.xls").get, "application/vnd.ms-excel", "班级信息.xls")
  }

  protected override def configImport(setting: ImportSetting): Unit = {
    val fk=new ForeignerListener(entityDao)
    fk.addForeigerKey("name")
    setting.listeners = List(fk, new SquadImportListener(getProject,entityDao))
  }

}
