/*
 * OpenURP, Agile University Resource Planning Solution.
 *
 * Copyright © 2014, The OpenURP Software.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful.
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.openurp.edu.base.web.action.admin

import java.time.LocalDate

import org.beangle.commons.collection.Order
import org.beangle.data.dao.OqlBuilder
import org.beangle.data.transfer.importer.ImportSetting
import org.beangle.data.transfer.importer.listener.ForeignerListener
import org.beangle.webmvc.api.view.View
import org.openurp.base.model.Department
import org.openurp.code.edu.model.{AcademicLevel, ExamMode, GradingMode}
import org.openurp.edu.base.code.model.{CourseCategory, CourseHourType, CourseType}
import org.openurp.edu.base.model.{Course, CourseHour}
import org.openurp.edu.base.web.action.ProjectRestfulAction
import org.openurp.edu.base.web.helper.{CourseImportListener, QueryHelper}

class CourseAction extends ProjectRestfulAction[Course] {

  protected override def indexSetting(): Unit = {
    put("courseTypes", getCodes(classOf[CourseType]))
    put("courseCategories", getCodes(classOf[CourseCategory]))
    val departments = findInSchool(classOf[Department])
    put("departments", departments)
  }

  override def getQueryBuilder: OqlBuilder[Course] = {
    val builder: OqlBuilder[Course] = OqlBuilder.from(entityName, simpleEntityName)
    getDate("beginOn").foreach(beginOn => {
      getDate("endOn").foreach(endOn => {
        builder.where("course.beginOn between :beginOn and :endOn", beginOn, endOn)
      })
    })
    populateConditions(builder)
    builder.where(simpleEntityName + ".project = :project", getProject)
    builder.orderBy(get(Order.OrderStr).orNull).limit(getPageLimit)
    QueryHelper.addTemporalOn(builder, getBoolean("active"))
  }

  override def editSetting(entity: Course) = {
    put("courseTypes", getCodes(classOf[CourseType]))
    put("examModes", getCodes(classOf[ExamMode]))
    put("gradingModes", getCodes(classOf[GradingMode]))
    put("courseCategories", getCodes(classOf[CourseCategory]))
    put("departments", findInSchool(classOf[Department]))

    var levels = getProject.levels.map(_.toLevel).toSet.toBuffer
    levels --= entity.levels
    put("levels", levels)
    put("hourTypes", getCodes(classOf[CourseHourType]))
    if (!entity.persisted) {
      entity.project = getProject
      entity.beginOn = LocalDate.now
      entity.levels ++= levels
      levels.clear()
    }
    super.editSetting(entity)
  }

  protected override def saveAndRedirect(entity: Course): View = {
    val course = entity.asInstanceOf[Course]

    val hourTypes = getCodes(classOf[CourseHourType])
    hourTypes foreach { ht =>
      val creditHour = getInt("creditHour" + ht.id)
      val week = getInt("week" + ht.id)
      course.hours find (h => h.hourType == ht) match {
        case Some(hour) =>
          if (week.isEmpty && creditHour.isEmpty) {
            course.hours -= hour
          } else {
            hour.weeks = week.getOrElse(0)
            hour.creditHours = creditHour.getOrElse(0)
          }
        case None =>
          if (!(week.isEmpty && creditHour.isEmpty)) {
            val newHour = new CourseHour()
            newHour.course = course
            newHour.hourType = ht
            newHour.weeks = week.getOrElse(0)
            newHour.creditHours = creditHour.getOrElse(0)
            course.hours += newHour
          }
      }
    }
    val orphan=course.hours.filter(x=> !hourTypes.contains(x.hourType))
    course.hours --= orphan
    val levelIds = getAll("levelId2nd", classOf[Int])
    val newLevels = entityDao.find(classOf[AcademicLevel], levelIds)
    val removed = course.levels filter { x => !newLevels.contains(x) }
    course.levels.subtractAll(removed)
    newLevels foreach { l =>
      if (!course.levels.contains(l))
        course.levels += l
    }

    course.gradingModes.clear()
    val gradingModeIds = getAll("gradingModeId2nd", classOf[Int])
    course.gradingModes ++= entityDao.find(classOf[GradingMode], gradingModeIds)

    super.saveAndRedirect(entity)
  }

  protected override def configImport(setting: ImportSetting): Unit = {
    val fl = new ForeignerListener(entityDao)
    fl.addForeigerKey("name")
    setting.listeners = List(fl, new CourseImportListener(entityDao))
  }

  def newCourses(): View = {
    getDate("beginOn").foreach(beginOn => {
      getDate("endOn").foreach(endOn => {
        put("courses", entityDao.search(getQueryBuilder))
      })
    })
    forward()
  }
}
