/*
 * OpenURP, Agile Development Scaffold and Toolkit
 *
 * Copyright (c) 2014-2015, OpenURP Software.
 *
 * OpenURP is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * OpenURP is distributed in the hope that it will be useful.
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OpenURP.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.openurp.edu.base

import java.sql.Date

import scala.collection.mutable.Buffer

import org.beangle.data.model.{ Coded, Entity, LongIdEntity, Named, TemporalOn, Updated }
import org.openurp.base.Department
import org.openurp.code.edu.{ DisciplineCategory, Education }

/**
 * 专业(校内专业）信息
 * </p>
 * 每个专业设立多个<code>Direction</code>方向。为了便于管理，专业应体现在具体的教学项目<code>Project</code>上
 * </p>
 * 因为专业在业务上是体现学校专业建设的过程实体。负责体现专业带头人、设立年月、招生年月、核心课程、教研室等，这些都和层次、部门有着密切联系。
 * 因此相同二级学科跨部门的情况（设立两个不同的方向），建议设立两个校内专业，并设置相同的二级学科。
 * </p>
 * 同学科对应多学历层次情况，采用多个校内专业。例如研究生学科目录中一个专业既培养硕士研究生又培养博士研究生，系统中仍然建立两个专业。
 * </p>
 *
 * @composed 1 - n Direction
 * @depend - - - Project
 * @depend - - - Department
 * @depend - - - Education
 * @depend - - - Degree
 * @author chaostone
 * @since 2005
 */
trait Major extends LongIdEntity with ProjectBased with TemporalOn with Updated with Coded with Named {

  /** 专业英文名 */
  var enName: String = _
  /** 备注 */
  var remark: String = _
  /** 获得方向 */
  var directions: collection.mutable.Set[Direction] = new collection.mutable.HashSet[Direction]
  /** 学科门类 */
  var category: DisciplineCategory = _
  /** 培养层次 */
  var educations: collection.mutable.Set[Education] = new collection.mutable.HashSet[Education]
  /** 修读年限 */
  var duration: Float = _
  /** 部门 */
  var journals: Buffer[MajorJournal] = new collection.mutable.ListBuffer[MajorJournal]

  /** 简称 */
  var shortName: String = _

  def allDepartments: Set[Department] = {
    journals.map(_.depart).toSet
  }

  def departments: Set[Department] = {
    departments(new Date(System.currentTimeMillis))
  }

  def departments(date: Date): Set[Department] = {
    journals.filter(j => date.after(j.beginOn) && (null == j.endOn || date.before(j.endOn)))
      .map(_.depart).toSet
  }
}

trait MajorJournal extends LongIdEntity with TemporalOn {
  /**专业*/
  var major: Major = _
  /** 教育部代码 */
  var disciplineCode: String = _
  /** 学科门类 */
  var category: DisciplineCategory = _
  /** 修读年限 */
  var duration: Float = _
  /**培养层次*/
  var education: Education = _
  /**部门*/
  var depart: Department = _
  /**备注*/
  var remark: String = _

}