/*
 * OpenURP, Agile Development Scaffold and Toolkit
 *
 * Copyright (c) 2014-2015, OpenURP Software.
 *
 * OpenURP is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * OpenURP is distributed in the hope that it will be useful.
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OpenURP.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.openurp.edu.base

import java.sql.Date

import org.beangle.data.model.{Coded, LongIdEntity, Named, YearId}
import org.openurp.base.{Campus, Department}
import org.openurp.code.edu.{Education, StudyType}
import org.openurp.edu.base.code.{StdLabel, StdLabelType, StdStatus, StdType}
import org.openurp.people.base.Person

/**
 * 学籍信息
 * </p>
 * 学籍信息记录了四部分内容： <li>基本内容 学号、姓名、英文名(拼音)、性别</li> <li>培养内容 项目、年级、院系、专业、方向、班级、培养层次、学习形式、学生分类标签</li> <li>
 * 培养时间 录取时间、入学时间、预计毕业时间、学制</li> <li>学籍状态日志 各时段的是否在校、专业、方向以及学籍状态</li>
 *
 * @depend - - - Gender
 * @depend - - - Department
 * @depend - - - Major
 * @depend - - - Direction
 * @depend - - - StdType
 * @depend - - - StudyType
 * @depend - - - EducationType
 * @has 1..* has 1..* StdLabel
 * @composed 1 has * StudentJournal
 * @author chaostone
 * @since 2005
 */
trait Student extends LongIdEntity with EducationBased with YearId with Coded with Named {

  /** 年级 表示现在年级，不同于入学时间 */
  var grade: String = _
  /** 管理院系 行政管理院系 */
  var department: Department = _
  /** 专业 当前修读专业 */
  var major: Major = _
  /** 方向 当前修读方向 */
  var direction: Direction = _
  /** 专业所在院系 */
  var majorDepart: Department = _
  /** 学生类别 所在项目内的学生类别 */
  var stdType: StdType = _
  /** 学生分类标签 */
  var labels: collection.mutable.Map[StdLabelType, StdLabel] = new collection.mutable.HashMap[StdLabelType, StdLabel]
  /** 校区 */
  var campus: Campus = _
  /** 学制 学习年限（允许0.5年出现） */
  var duration: Float = _
  /** 是否有学籍 */
  var registed: Boolean = _
  /** 入学报到日期 */
  var enrollOn: Date = _
  /** 学籍生效日期 */
  var registOn: Date = _
  /** 应毕业时间 预计毕业日期 */
  var graduateOn: Date = _
  /** 行政班级 */
  var adminclass: Adminclass = _
  /** 备注 */
  var remark: String = _
  /** 学习形式 全日制/业余/函授 */
  var studyType: StudyType = _
  /** 状态变化日志 */
  var journals: collection.mutable.Seq[StudentJournal] = _
  /**导师*/
  var tutor: Teacher = _
  /**基本信息*/
  var person: Person = _

  /**培养方案*/
  var program: Program = _

  def year = Integer.valueOf(grade.substring(0, 4))
}

/**
 * 学籍状态日志
 * </p>
 * 学籍状态日志记录从起始时间到结束时间之间的学籍状态。主要记录学生的 <li>年级</li> <li>管理院系</li> <li>专业</li> <li>方向</li> <li>行政班级</li>
 * <li>是否在校</li> <li>学籍状态</li> [beginOn,endOn)
 *
 * @author chaostone
 */
trait StudentJournal extends LongIdEntity {

  /** 学生 */
  var std: Student = _
  /** 年级 */
  var grade: String = _
  /** 管理院系 */
  var department: Department = _
  /** 专业 */
  var major: Major = _
  /**专业方向*/
  var direction: Direction = _
  /** 行政班级 */
  var adminclass: Adminclass = _
  /** 是否在校 */
  var inschool: Boolean = _
  /** 学籍状态 */
  var status: StdStatus = _
  /** 起始日期 */
  var beginOn: Date = _
  /** 结束日期 */
  var endOn: Date = _
  /** 备注 */
  var remark: String = _
}

/**
 * 基于学生信息的实体
 *
 * @author Administrator
 */
trait StudentBased {

  var std: Student = _
}

/**
 * 基于培养层次的实体接口
 * </p>
 * 基于项目和培养层次的实体接口，标准化了培养层次的属性名称。
 *
 * @see Adminclass
 * @see Student
 * @author chaostone
 */

trait EducationBased extends ProjectBased {
  var education: Education = _

}
