/*
 * OpenURP, Agile University Resource Planning Solution.
 *
 * Copyright © 2014, The OpenURP Software.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful.
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.openurp.edu.base.web

import java.util

import com.google.gson.Gson
import javax.servlet.http.{HttpServletRequest, HttpServletResponse}
import org.beangle.commons.lang.{Numbers, Strings}
import org.beangle.data.dao.{EntityDao, OqlBuilder}
import org.beangle.security.Securities
import org.beangle.security.authc.DefaultAccount
import org.openurp.app.web.URPProfile
import org.openurp.base.model.School
import org.openurp.edu.base.model.Project

class URPProfileHelper(entityDao: EntityDao) {

  def getSchool(req: HttpServletRequest, res: HttpServletResponse): School = {
    findSchoolByServerName(req) match {
      case Some(s) => s
      case None =>
        var school: School = null
        val profile = URPProfile.get(req, res)
        if (profile.hasValidProject) {
          val project = entityDao.get(classOf[Project], profile.edu.projectId)
          school = project.school
          profile.schoolId = school.id
          URPProfile.update(req, res, profile, true)
        }
        //尝试从用户的权限中
        if (null == school) {
          val account = Securities.session.get.principal.asInstanceOf[DefaultAccount]
          val profileStr = account.details.get("profiles_edu").asInstanceOf[String]
          if (Strings.isNotBlank(profileStr)) {
            val profileIter = new Gson().fromJson(profileStr, classOf[util.List[util.Map[String, String]]]).iterator()
            var finded = false
            var project: Project = null
            while (profileIter.hasNext && !finded) {
              val pstr = profileIter.next().get("projects")
              if (null != pstr) {
                if (pstr == "*") {
                  project = getFirstProject()
                } else if (Numbers.isDigits(pstr)) {
                  project = entityDao.get(classOf[Project], pstr.toInt)
                }

                if (null != project) {
                  finded = true
                  profile.updateProject(project.id)
                  profile.schoolId = project.school.id
                  school = project.school
                  URPProfile.update(req, res, profile, true)
                }
              }
            }
          }
        }
        if (null == school) {
          school = entityDao.getAll(classOf[School]).head
          profile.schoolId = school.id
          school = school
          URPProfile.update(req, res, profile, true)
        }
        school
    }
  }

  def getProject(req: HttpServletRequest, res: HttpServletResponse): Project = {
    findProjectByServerName(req) match {
      case Some(p) => p
      case None =>
        var project: Project = null
        val profile = URPProfile.get(req, res)

        if (profile.hasValidProject) {
          project = entityDao.get(classOf[Project], profile.edu.projectId)
          URPProfile.update(req, res, profile, true)
        }
        //尝试从用户的权限中
        if (null == project) {
          val account = Securities.session.get.principal.asInstanceOf[DefaultAccount]
          account.details.get("profiles_edu") foreach { profileStr =>
            val profileIter = new Gson().fromJson(profileStr, classOf[util.List[util.Map[String, String]]]).iterator()
            var finded = false
            while (profileIter.hasNext && !finded) {
              val pstr = profileIter.next().get("projects")
              if (null != pstr) {
                if (pstr == "*") {
                  project = getFirstProject()
                } else if (Numbers.isDigits(pstr)) {
                  project = entityDao.get(classOf[Project], pstr.toInt)
                }

                if (null != project) {
                  finded = true
                  profile.updateProject(project.id)
                  URPProfile.update(req, res, profile, true)
                }
              }
            }
          }
        }

        if (null == project) {
          project = getFirstProject()
        }
        project
    }
  }

  private def findSchoolByServerName(request: HttpServletRequest): Option[School] = {
    val builder = OqlBuilder.from(classOf[School], "p").where("p.code =:pcode", request.getServerName)
      .cacheable()
    entityDao.search(builder).headOption
  }

  private def findProjectByServerName(request: HttpServletRequest): Option[Project] = {
    val builder = OqlBuilder.from(classOf[Project], "p").where("p.code =:pcode", request.getServerName)
      .cacheable()
    entityDao.search(builder).headOption
  }

  private def getFirstProject(): Project = {
    entityDao.getAll(classOf[Project]).head
  }

}
