/*
 * Copyright (C) 2014, The OpenURP Software.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package org.openurp.edu.grade.web.action.course

import org.beangle.commons.bean.orderings.PropertyOrdering
import org.beangle.commons.collection.Collections
import org.beangle.data.dao.OqlBuilder
import org.beangle.webmvc.annotation.{mapping, param}
import org.beangle.webmvc.support.action.RestfulAction
import org.beangle.webmvc.view.View
import org.openurp.base.model.{Project, Semester}
import org.openurp.base.std.model.Student
import org.openurp.base.util.FloatSegment
import org.openurp.code.edu.model.{CourseTakeType, ExamStatus, GradeType}
import org.openurp.edu.clazz.model.Clazz
import org.openurp.edu.grade.domain.{CourseGradeProvider, DefaultGpaPolicy}
import org.openurp.edu.grade.service.stat.GradeSegStat
import org.openurp.edu.his.model.HisCourseGrade
import org.openurp.starter.web.support.ProjectSupport

/** 学生历史成绩查询
 */
class HisAction extends RestfulAction[HisCourseGrade], ProjectSupport {

  var courseGradeProvider: CourseGradeProvider = _

  override protected def indexSetting(): Unit = {
    given project: Project = getProject

    put("project", project)
    put("courseTakeTypes", getCodes(classOf[CourseTakeType]))
    put("examStatuses", getCodes(classOf[ExamStatus]))
  }

  override def getQueryBuilder: OqlBuilder[HisCourseGrade] = {
    val q = super.getQueryBuilder
    val p = getProject
    q.where("courseGrade.project=:project", p)
    q
  }

  /** 查看单个学生单个学期的所有成绩
   *
   * @param stdId
   * @param semesterId
   * @return
   */
  @mapping("semester/{std.id}/{semester.id}")
  def semester(@param("std.id") stdId: Long, @param("semester.id") semesterId: Int): View = {
    val std = entityDao.get(classOf[Student], stdId)
    val semester = entityDao.get(classOf[Semester], semesterId)

    val q = OqlBuilder.from(classOf[HisCourseGrade], "g")
    q.where("g.std=:std and g.semester=:semester", std, semester)

    val grades = entityDao.search(q).sorted(PropertyOrdering.by("crn,course.code"))
    put("std", std)
    put("semester", semester)
    put("grades", grades)
    put("stdGpa", new DefaultGpaPolicy().calc(std, grades.map(_.convert()), false))
    forward()
  }

  /** 查看单个学生所有的成绩
   *
   * @param stdId
   * @param semesterId
   * @return
   */
  @mapping("all/{std.id}")
  def all(@param("std.id") stdId: Long): View = {
    val std = entityDao.get(classOf[Student], stdId)

    val grades = courseGradeProvider.getAll(std)
    put("std", std)
    put("semesterGrades", grades.groupBy(_.semester))
    put("stdGpa", new DefaultGpaPolicy().calc(std, grades, true))
    val segments = Collections.newBuffer[FloatSegment]
    segments += new FloatSegment(50, 59)
    segments += new FloatSegment(0, 49)
    Range(50, 100, 5) foreach { i =>
      segments += new FloatSegment(i, i + 5)
    }
    val segs = GradeSegStat(grades, new GradeType(GradeType.Final), segments)
    put("segs", segs)
    forward("../std/all")
  }

  /** 查看单个教学任务的成绩
   *
   * @param clazzId
   * @return
   */
  @mapping("clazz/{clazz.id}")
  def clazz(@param("clazz.id") clazzId: Long): View = {
    val grades = entityDao.findBy(classOf[HisCourseGrade], "clazz.id", clazzId)
    put("grades", grades)
    put("clazz", entityDao.get(classOf[Clazz], clazzId))
    forward("../std/clazz")
  }

  override protected def simpleEntityName: String = "courseGrade"
}
