/*
 * Copyright (C) 2014, The OpenURP Software.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package org.openurp.edu.grade.web.action.regular

import org.beangle.commons.collection.Collections
import org.beangle.commons.lang.Strings
import org.beangle.data.dao.OqlBuilder
import org.beangle.webmvc.annotation.{mapping, param}
import org.beangle.webmvc.support.action.RestfulAction
import org.beangle.webmvc.view.View
import org.openurp.base.model.Project
import org.openurp.base.util.FloatSegment
import org.openurp.edu.grade.model.{RegularGrade, RegularGradeState}
import org.openurp.edu.his.model.HisRegularGrade
import org.openurp.starter.web.support.ProjectSupport

import scala.collection.mutable

/** 教学任务对应的平时成绩管理
 */
class ClazzAction extends RestfulAction[RegularGradeState], ProjectSupport {

  override def indexSetting(): Unit = {
    given project: Project = getProject

    val semester = getSemester

    val q = OqlBuilder.from(classOf[RegularGradeState], "state")
    q.where("state.clazz.project=:project and state.clazz.semester=:semester", project, semester)
    q.select("distinct state.clazz.teachDepart")
    put("departs", entityDao.search(q))

    put("semester", semester)
    put("project", project)
  }

  @mapping(value = "{id}")
  override def info(@param("id") id: String): View = {
    val state = entityDao.get(classOf[RegularGradeState], id.toLong)
    val components = state.components
    val scores = Collections.newMap[String, mutable.ArrayBuffer[Float]]
    put("clazz", state.clazz)

    if (state.clazz.semester.year.archived) {
      val grades = entityDao.findBy(classOf[HisRegularGrade], "clazz", state.clazz).sortBy(_.std.code)
      grades foreach { grade =>
        val tests = grade.tests
        components foreach { c =>
          scores.getOrElseUpdate(c._1, new mutable.ArrayBuffer[Float]).addOne(tests.get(c._1).map(_.score).getOrElse(0f))
        }
      }
      put("grades", grades)
    } else {
      val grades = entityDao.findBy(classOf[RegularGrade], "clazz", state.clazz).sortBy(_.std.code)
      grades foreach { grade =>
        val tests = grade.tests
        components foreach { c =>
          scores.getOrElseUpdate(c._1, new mutable.ArrayBuffer[Float]).addOne(tests.get(c._1).map(_.score).getOrElse(0f))
        }
      }
      put("grades", grades)
    }

    val segments = Collections.newBuffer[FloatSegment]
    segments += new FloatSegment(95, 100)
    segments += new FloatSegment(90, 94)
    segments += new FloatSegment(80, 89)
    segments += new FloatSegment(70, 79)
    segments += new FloatSegment(60, 69)
    segments += new FloatSegment(50, 59)
    segments += new FloatSegment(0, 49)
    put("segments", segments.toSeq)
    val stats = scores.map { s => (s._1, SegmentStat.stat(s._2, segments)) }
    put("stats", stats)
    put("state", state)

    forward()
  }

  override def getQueryBuilder: OqlBuilder[RegularGradeState] = {
    val query = super.getQueryBuilder
    query.where("state.clazz.project=:project", getProject)
    get("teacher.name") foreach { t =>
      if (Strings.isNotBlank(t)) {
        query.where("exists(from state.clazz.teachers t where t.name like :teacherName)", "%" + t + "%")
      }
    }
    query
  }

  override protected def simpleEntityName: String = {
    "state"
  }
}
