/*
 * OpenURP, Agile University Resource Planning Solution.
 *
 * Copyright © 2014, The OpenURP Software.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful.
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.openurp.edu.spa.web.action.admin

import java.time.LocalDate

import org.beangle.data.dao.OqlBuilder
import org.beangle.webmvc.api.annotation.{mapping, param}
import org.beangle.webmvc.api.view.View
import org.beangle.webmvc.entity.action.RestfulAction
import org.openurp.edu.spa.model.{DocType, DownloadLog}

class DownloadStatAction extends RestfulAction[DownloadLog] {

  override def index(): View = {
    val year= get("year") match {
      case Some(y)=> y.toInt
      case None => LocalDate.now.getYear
    }
    redirect("year","&year="+year,"")
  }

  @mapping("year/{year}")
  def year(@param("year") year: Int): View ={
    val builder = OqlBuilder.from(classOf[DownloadLog], "pr")
    builder.select("extract(year from pr.updatedAt) ,count(*) as num")
    builder.groupBy("extract(year from pr.updatedAt)")
    builder.orderBy("extract(year from pr.updatedAt) desc")
    val datas = entityDao.search(builder).asInstanceOf[Seq[Array[Any]]]
    put("docTypes", entityDao.search(OqlBuilder.from(classOf[DocType], "dt").where("dt.enabled = true")))
    put("datas", datas)

    if (datas.nonEmpty) {
      put("year", year)
      docTypeStat(year.toInt)
      forward()
    } else {
      forward("nodata")
    }
  }

  private def docTypeStat(year: Int): Unit = {
    val builder = OqlBuilder.from(classOf[DownloadLog], "pr")
    builder.select("pr.docType.id ,count(*) as num")
    builder.where("extract(year from pr.updatedAt)=:year", year)
    builder.groupBy("pr.docType.id")
    builder.orderBy("pr.docType.id")
    val datas = entityDao.search(builder).asInstanceOf[Seq[Any]]
    val docTypes = entityDao.search(OqlBuilder.from(classOf[DocType], "dt").where("dt.enabled = true"))
    val docTypeMap = docTypes.map { dc => (dc.id.toString, dc.name) }.toMap
    val ids = for (data <- datas) yield data.asInstanceOf[Array[Any]](0).toString
    val names = for (data <- datas) yield docTypeMap.get(data.asInstanceOf[Array[Any]](0).toString).getOrElse("")
    val values = for (data <- datas) yield data.asInstanceOf[Array[Any]](1)
    put("docTypeIds", ids)
    put("docTypeNames", names)
    put("docTypeCounts", values)
  }

  def monthChart(@param("year") year: Int): View = {
    val builder = OqlBuilder.from(classOf[DownloadLog], "pr")
    builder.select("extract(month from pr.updatedAt) ,count(*) as num")
    builder.where("extract(year from pr.updatedAt)=:year", year)
    getInt("docType.id").foreach(docTypeId => {
      builder.where("pr.docType.id=:id", docTypeId)
      put("docType", entityDao.get(classOf[DocType], docTypeId))
    })
    builder.groupBy("extract(month from pr.updatedAt)")
    builder.orderBy("extract(month from pr.updatedAt)")
    putNamesAndValues(entityDao.search(builder))
    put("year", year)
    forward()
  }

  def dayChart(@param("year") year: Int, @param("month") month: Int): View = {
    val builder = OqlBuilder.from(classOf[DownloadLog], "pr")
    builder.select("extract(day from pr.updatedAt) ,count(*) as num")
    builder.where("extract(year from pr.updatedAt)=:year", year)
    builder.where("extract(month from pr.updatedAt)=:month", month)
    getInt("docType.id").foreach(docTypeId => {
      builder.where("pr.docType.id=:id", docTypeId)
      put("docType", entityDao.get(classOf[DocType], docTypeId))
    })
    builder.groupBy("extract(day from pr.updatedAt)")
    builder.orderBy("extract(day from pr.updatedAt)")
    putNamesAndValues(entityDao.search(builder))
    put("year", year)
    put("month", month)
    forward()
  }

  protected def putNamesAndValues(datas: Seq[Any]): Unit = {
    putNamesAndValues(datas, data => data(0))
  }

  protected def putNamesAndValues(datas: Seq[Any], nf: Array[Any] => Any): Unit = {
    val names = for (data <- datas) yield nf(data.asInstanceOf[Array[Any]])
    val values = for (data <- datas) yield data.asInstanceOf[Array[Any]](1)
    put("names", names)
    put("values", values)
    put("datas", datas)
  }

}
