/*
 * OpenURP, Agile University Resource Planning Solution.
 *
 * Copyright © 2014, The OpenURP Software.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful.
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.openurp.edu.spa.web.action.student

import java.time.{Instant, LocalDate}

import org.beangle.commons.collection.Collections
import org.beangle.commons.web.util.RequestUtils
import org.beangle.data.dao.{EntityDao, OqlBuilder}
import org.beangle.security.Securities
import org.beangle.webmvc.api.action.{ActionSupport, ServletSupport}
import org.beangle.webmvc.api.view.{Status, View}
import org.openurp.base.model.User
import org.openurp.edu.spa.model._

class DocAction extends ActionSupport with ServletSupport {

  var entityDao: EntityDao = _

  def index(): View = {
    val builder = OqlBuilder.from(classOf[DocType], "dt")
    builder.where("dt.enabled=true")
    builder.orderBy("dt.code")
    val docTypes = entityDao.search(builder)

    put("docTypes", docTypes)
    val users = entityDao.findBy(classOf[User], "code", List(Securities.user))
    val user = users.head
    put("user", user)

    val prQuery = OqlBuilder.from(classOf[PrintLog], "pr")
    prQuery.where("pr.user=:user", user)
    prQuery.orderBy("pr.updatedAt desc")
    val prs = entityDao.search(prQuery)
    put("printLogs", prs)

    Securities.session foreach { s =>
      put("URP_SID", s.id)
    }

    //加载打印配置
    val printConfigQuery = OqlBuilder.from(classOf[PrintConfig], "pc")
    printConfigQuery.where(":now >= pc.beginOn and (pc.endOn is null or :now <= pc.endOn)", LocalDate.now)
    printConfigQuery.cacheable()
    val configs = entityDao.search(printConfigQuery)
    val configMap = configs.map { pc => (pc.docType, pc) }.toMap
    put("configs", configMap)

    //加载优惠券
    val couponQuery = OqlBuilder.from(classOf[Coupon], "coupon")
    couponQuery.where(":now between coupon.beginOn and coupon.endOn", LocalDate.now)
    couponQuery.cacheable()
    val coupons = entityDao.search(couponQuery).map { c => (c.docType, c) }.toMap

    //统计各类打印次数
    val psQuery = OqlBuilder.from(classOf[PrintQuota], "ps")
    psQuery.where("ps.user=:user", user)
    val stats = Collections.newMap[DocType, PrintQuota]
    entityDao.search(psQuery) foreach (ps => stats.put(ps.docType, ps))

    //使用优惠券打印的次数
    val freePrintedCoupon = Collections.newMap[DocType, Int]
    val freePrinted = Collections.newMap[DocType, Int]
    val printed = Collections.newMap[DocType, Int]
    prs foreach { pr =>
      val docType = pr.docType
      val stat = stats.getOrElseUpdate(docType, new PrintQuota(user, docType))
      // 统计最后打印时间
      if (stat.lastPrintAt == null || pr.updatedAt.isAfter(stat.lastPrintAt)) {
        stat.lastPrintAt = pr.updatedAt
      }

      if (pr.payed == 0) {
        freePrinted.put(docType, freePrinted.getOrElseUpdate(docType, 0) + 1)
        coupons.get(docType) foreach { coupon =>
          if (coupon.validAt(pr.updatedAt)) {
            val p = freePrintedCoupon.getOrElseUpdate(docType, 0)
            freePrintedCoupon.put(docType, p + 1)
          }
        }
      } else {
        printed.put(docType, printed.getOrElseUpdate(docType, 0) + 1)
      }
    }
    //计算每种文档能够打印的免费次数
    val frees = Collections.newMap[DocType, Int]
    coupons foreach { case (docType, coupon) =>
      val free = coupon.count - freePrintedCoupon.getOrElseUpdate(docType, 0)
      frees.put(docType, free)
    }
    put("frees", frees)

    //超出最大配合的文档类型
    val prohibits = Collections.newBuffer[DocType]
    //更新打印统计
    stats foreach { case (docType, stat) =>
      stat.printCnt = printed.getOrElse(docType, 0)
      configMap.get(docType) foreach { config =>
        if (stat.printCnt >= config.maxLimit) {
          prohibits += docType
        }
      }
      stat.freeCnt = freePrinted.getOrElse(docType, 0)
      stat.frees = frees.getOrElse(docType, 0)
    }

    entityDao.saveOrUpdate(stats.values)
    put("prohibits", prohibits)
    put("stats", stats)
    forward()
  }

  def print(): View = {
    val users = entityDao.findBy(classOf[User], "code", List(Securities.user))
    val user = users.head

    val docs = getDocType
    val payed = getInt("payed").getOrElse(0)
    if (docs.size == 1) {
      val pr = new PrintLog()
      pr.user = user
      pr.docType = docs.head
      pr.payed = payed
      pr.updatedAt = Instant.now
      pr.remark=get("remark")
      pr.ip = RequestUtils.getIpAddr(request)
      entityDao.saveOrUpdate(pr)
      Status.Ok
    } else {
      Status.NotFound
    }
  }

  private def getDocType: Option[DocType] = {
    val query = OqlBuilder.from(classOf[DocType], "doc")
    query.where("doc.code =:code", get("docType", ""))
    query.cacheable()
    val docs = entityDao.search(query)
    docs.headOption
  }

  def view(): View = {
    val docs = getDocType
    docs foreach { doc =>
      put("docType", doc)
      var sep = "?"
      var doc_url = doc.url
      if (doc_url.indexOf("?") > 0) {
        sep = "&"
      }
      doc_url = doc_url + sep + "URP_SID=" + Securities.session.map(_.id).getOrElse("")
      put("doc_url", doc_url)
    }
    forward()
  }

}
