/*
 * OpenURP, Agile University Resource Planning Solution.
 *
 * Copyright © 2014, The OpenURP Software.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful.
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.openurp.edu.spa.web.action.student

import java.io.File
import java.net.URL
import java.time.Instant

import com.itextpdf.text.pdf.PdfWriter
import org.beangle.commons.codec.digest.Digests
import org.beangle.commons.lang.Strings
import org.beangle.commons.web.url.UrlBuilder
import org.beangle.commons.web.util.RequestUtils
import org.beangle.data.dao.{EntityDao, OqlBuilder}
import org.beangle.doc.pdf.{Encryptor, SPD}
import org.beangle.security.Securities
import org.beangle.webmvc.api.action.{ActionSupport, ServletSupport}
import org.beangle.webmvc.api.view.{Stream, View}
import org.openurp.base.model.User
import org.openurp.edu.spa.model._

class PdfAction extends ActionSupport with ServletSupport {

  var entityDao: EntityDao = _

  def index(): View = {
    val builder = OqlBuilder.from(classOf[DocType], "dt")
    builder.where("dt.enabled=true")
    builder.where("dt.downloadable=true")
    builder.orderBy("dt.code")
    val docTypes = entityDao.search(builder)

    put("docTypes", docTypes)
    val users = entityDao.findBy(classOf[User], "code", List(Securities.user))
    val user = users.head
    put("user", user)

    Securities.session foreach { s =>
      put("URP_SID", s.id)
    }
    forward()
  }


  private def getDocType: Option[DocType] = {
    val query = OqlBuilder.from(classOf[DocType], "doc")
    query.where("doc.code =:code", get("docType", ""))
    query.cacheable()
    val docs = entityDao.search(query)
    docs.headOption
  }

  def view(): View = {
    val docs = getDocType
    docs foreach { doc =>
      put("docType", doc)
      var sep = "?"
      var doc_url = doc.url
      if (doc_url.indexOf("?") > 0) {
        sep = "&"
      }
      doc_url = doc_url + sep + "URP_SID=" + Securities.session.map(_.id).getOrElse("")
      put("doc_url", doc_url)
    }
    forward()
  }

  def download: View = {
    val doc = getDocType.get
    var doc_url = doc.url
    if (doc_url.startsWith("{origin}")) {
      val urlBuilder = UrlBuilder(request)
      doc_url = Strings.replace(doc_url, "{origin}", urlBuilder.buildOrigin())
    }
    val sep = if (doc_url.indexOf("?") > 0) "&" else "?"
    val user = entityDao.findBy(classOf[User], "code", List(Securities.user)).head
    val url = doc_url + sep + "URP_SID=" + Securities.session.map(_.id).getOrElse("")
    val pdf = File.createTempFile("doc", ".pdf")
    SPD.convertURL(new URL(url), pdf, Map("orientation" -> doc.orientation.toString))
    Encryptor.encrypt(pdf, Some(Securities.user), Digests.md5Hex(user.code + "_" + user.name), PdfWriter.ALLOW_PRINTING)
    val pr = new DownloadLog()
    pr.user = user
    pr.docType = doc
    pr.updatedAt = Instant.now
    pr.ip = RequestUtils.getIpAddr(request)
    entityDao.saveOrUpdate(pr)

    Stream(pdf, user.name + "_" + doc.name)
  }

}
