/*
 * OpenURP, Agile Development Scaffold and Toolkit
 *
 * Copyright (c) 2014-2015, OpenURP Software.
 *
 * OpenURP is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * OpenURP is distributed in the hope that it will be useful.
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OpenURP.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.openurp.edu.teach.grade

import scala.collection.mutable.{ Buffer, ListBuffer }
import org.beangle.data.model.{ Entity, IntIdEntity, LongIdEntity, Updated }
import org.openurp.base.{ Department, Semester }
import org.openurp.edu.base.ProjectBased
import org.openurp.edu.base.code._
import org.openurp.edu.teach.code.{ GradeType, _ }
import org.openurp.edu.teach.code.model.GradeTypeBean
import org.openurp.edu.teach.lesson.Lesson
import org.openurp.edu.base.Student
import org.openurp.edu.base.Course

object Grade {
  object Status {
    val New = 0
    val Confirmed = 1
    val Published = 2
  }
}

/**
 * 成绩接口
 * </p>
 * 系统中所有成绩的抽象接口，记录成绩的原始分数、呈现方式、状态和是否通过。
 * 成绩中设立原始分数和分数字面值，分别表示数字型分数和文字型评价。分数用于和其他记录方式进行转换，
 * 分数字面值则为最后的转换结果。
 *
 * @depend - - - ScoreMarkStyle
 * @author chaostone
 * @since 2006
 */
trait Grade extends Ordered[Grade] with LongIdEntity with Updated {
  def score: java.lang.Float
  def std: Student
  def scoreText: String
  def passed: Boolean
  def published: Boolean = {
    status == Grade.Status.Published
  }
  def confirmed: Boolean = {
    status == Grade.Status.Published || status == Grade.Status.Confirmed
  }
  def status: Int
  def status_=(newStatus: Int)
  def markStyle: ScoreMarkStyle
  def markStyle_=(style: ScoreMarkStyle)
  def gradeType: GradeType
  def operator: String
}

/**
 * 课程成绩
 * </p>
 * 学生因上课取得的成绩，业务唯一主键为[学生、项目、培养类型、学期、课程]。
 * </p>
 * 课程成绩由多个考试成绩组成，一般为平时、期末、补考、缓考、总评等成绩成分。
 *
 * @depend - - - Lesson
 * @depend - - - Course
 * @depend - - - CoursEntity[java.lang.Long] eType
 * @depend - - - CourseTakeType
 * @composed 1 has * ExamGrade
 * @depend - - - Project
 * @depend - - - Education
 * @author chaostone
 * @since 2006
 */

trait CourseGrade extends LongIdEntity with ProjectBased with Grade {
  /**
   * 设置学生
   */
  var std: Student = _
  /**
   * 设置课程
   */
  var course: Course = _
  /**
   * 获得修读类别
   */
  var courseTakeType: CourseTakeType = _
  /**
   * 学期
   */
  var semester: Semester = _
  /**
   * 任务序号
   */
  var lessonNo: String = _
  /**
   * 课程类别
   */
  var courseType: CourseType = _
  /**
   * 设置绩点
   */
  var gp: java.lang.Float = _

  var bonus: java.lang.Float = _

  /**
   * 总评成绩
   */
  var gaGrades: Buffer[GaGrade] = new ListBuffer[GaGrade]
  /**
   * 考核成绩
   */
  var examGrades: Buffer[ExamGrade] = new ListBuffer[ExamGrade]
  /**
   * 得到指定的考试成绩
   */
  def getGrade(gradeType: GradeType): Grade = {
    if (gradeType.isGa) gaGrades.find(eg => eg.gradeType == gradeType).orNull
    else examGrades.find(eg => eg.gradeType == gradeType).orNull
  }

  def getGrade(gradeTypeId: Integer): Grade = {
    getGrade(new GradeTypeBean(gradeTypeId))
  }
  /**
   * 考核方式
   */
  var examMode: ExamMode = _
  /**
   * 备注
   */
  var remark: String = _
  var score: java.lang.Float = _
  var scoreText: String = _
  var passed: Boolean = _
  var status: Int = _
  var markStyle: ScoreMarkStyle = _
  var operator: String = _
  var lesson: Lesson = _

  def gradeType: GradeType = {
    new GradeTypeBean(GradeType.Final)
  }
  // 大的成绩放前面
  override def compare(grade: Grade): Int = {
    if (null == score) return 1
    else if (null == grade.score) return -1
    return grade.score.compareTo(score)
  }
}