/*
 * Copyright (c) 2005, John Mettraux, OpenWFE.org
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * . Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.  
 * 
 * . Redistributions in binary form must reproduce the above copyright notice, 
 *   this list of conditions and the following disclaimer in the documentation 
 *   and/or other materials provided with the distribution.
 * 
 * . Neither the name of the "OpenWFE" nor the names of its contributors may be
 *   used to endorse or promote products derived from this software without
 *   specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: RsaKey.java 1882 2005-05-17 16:41:07Z jmettraux $
 */

//
// RsaKey.java
//
// jmettraux@openwfe.org
//
// generated with 
// jtmpl 1.0.04 31.10.2002 John Mettraux (jmettraux@openwfe.org)
//

package openwfe.org.crypto;

import java.math.BigInteger;
import java.security.KeyPair;
import java.security.KeyFactory;
import java.security.KeyPairGenerator;
import java.security.spec.RSAPublicKeySpec;
import java.security.spec.RSAPrivateKeySpec;
import java.security.interfaces.RSAKey;
import java.security.interfaces.RSAPublicKey;
import java.security.interfaces.RSAPrivateKey;


/**
 * Something that is easily XML encodable...
 *
 * <p><font size=2>CVS Info :
 * <br>$Author: jmettraux $
 * <br>$Date: 2005-05-17 18:41:07 +0200 (Tue, 17 May 2005) $
 * <br>$Id: RsaKey.java 1882 2005-05-17 16:41:07Z jmettraux $ </font>
 *
 * @author jmettraux@openwfe.org
 */
public class RsaKey

    implements java.io.Serializable

{

    static final long serialVersionUID = -2578029002953409978L;

    //
    // CONSTANTS (definitions)

    public final static String PUBLIC = "public";
    public final static String PRIVATE = "private";

    public final static String PUBLIC_KEY = "public-key";
    public final static String PRIVATE_KEY = "private-key";
    public final static String EXP = "exp";
    public final static String MOD = "mod";

    //
    // FIELDS

    protected String keyType = null;
    protected String exp = null;
    protected String mod = null;

    //
    // CONSTRUCTORS

    public RsaKey ()
    {
        super();
    }

    public RsaKey (final String keyType, final String exp, final String mod)
    {
        super();
        this.keyType = keyType;
        this.exp = exp;
        this.mod = mod;
    }

    public RsaKey (RSAKey key)
        throws CryptoException
    {
        this.mod = key.getModulus().toString();

        if (key instanceof RSAPublicKey)
        {
            this.keyType = PUBLIC;
            this.exp = ((RSAPublicKey)key).getPublicExponent().toString();
        }
        else if (key instanceof RSAPrivateKey)
        {
            this.keyType = PRIVATE;
            this.exp = ((RSAPrivateKey)key).getPrivateExponent().toString();
        }
        else
        {
            throw new CryptoException
                ("Cannot create RsaKey from key of class '"+
                 key.getClass().getName()+"'");
        }
    }

    //
    // BEAN METHODS

    public String getKeyType () { return this.keyType; }
    public String getMod () { return this.mod; }
    public String getExp () { return this.exp; }

    public void setKeyType (String keyType) { this.keyType = keyType; }
    public void setMod (String s) { this.mod = s; }
    public void setExp (String s) { this.exp = s; }

    //
    // METHODS

    public RSAKey assembleRealKey ()
        throws CryptoException
    {
        try
        {
            KeyFactory factory = KeyFactory.getInstance("RSA");

            if (this.keyType.equals(PUBLIC))
            {
                RSAPublicKeySpec spec = new RSAPublicKeySpec
                    (new BigInteger(this.mod), new BigInteger(this.exp));

                return (RSAKey)factory.generatePublic(spec);
            }

            RSAPrivateKeySpec spec = new RSAPrivateKeySpec
                (new BigInteger(this.mod), new BigInteger(this.exp));

            return (RSAKey)factory.generatePrivate(spec);
        }
        catch (Exception e)
        {
            throw new CryptoException
                ("Failed to assemble real RSAKey", e);
        }
    }

    /**
     * lightweight xml output
     */
    public String toString ()
    {
        StringBuffer sb = new StringBuffer();

        if (this.keyType.equals(PUBLIC))
            sb.append("<"+PUBLIC_KEY+">\n");
        else
            sb.append("<"+PRIVATE_KEY+">\n");

        sb.append("    <"+EXP+">");
        sb.append(this.exp);
        sb.append("</"+EXP+">\n");

        sb.append("    <"+MOD+">");
        sb.append(this.mod);
        sb.append("</"+MOD+">\n");

        if (this.keyType.equals(PUBLIC))
            sb.append("</"+PUBLIC_KEY+">\n");
        else
            sb.append("</"+PRIVATE_KEY+">\n");

        return sb.toString();
    }

    public org.jdom.Element toXml ()
    {
        org.jdom.Element eKey = null;

        if (this.keyType.equals(PUBLIC))
            eKey = new org.jdom.Element(PUBLIC_KEY);
        else
            eKey = new org.jdom.Element(PRIVATE_KEY);

        org.jdom.Element eExp = new org.jdom.Element(EXP);
        eExp.addContent(this.exp);
        eKey.addContent(eExp);

        org.jdom.Element eMod = new org.jdom.Element(MOD);
        eMod.addContent(this.mod);
        eKey.addContent(eMod);

        return eKey;
    }

    //
    // STATIC METHODS

    public static RsaKey buildKey (org.jdom.Element elt)
    {
        if (elt == null) return null;

        String eName = elt.getName();

        String keyType = PRIVATE;
        if (eName.equals(PUBLIC_KEY)) keyType = PUBLIC;

        String exp = elt.getChildTextTrim(EXP);
        String mod = elt.getChildTextTrim(MOD);

        return new RsaKey(keyType, exp, mod);
    }

    public static RsaKey[] generateKeyPair (int strength)
        throws CryptoException
    {
        try
        {
            KeyPairGenerator generator = KeyPairGenerator.getInstance("RSA");
            generator.initialize(strength);

            KeyPair pair = generator.generateKeyPair();

            RsaKey[] result = new RsaKey[] 
            { 
                new RsaKey((RSAKey)pair.getPublic()), 
                new RsaKey((RSAKey)pair.getPrivate())
            };

            return result;
        }
        catch (Exception e)
        {
            throw new CryptoException
                ("Failed to generate key pair", e);
        }
    }

    public static void main (String[] args)
    {
        try
        {
            System.out.println("Generating key pair...");
            
            RsaKey[] keyPair = generateKeyPair(2048);

            /*
            java.beans.XMLEncoder encoder = 
                new java.beans.XMLEncoder(System.out);

            encoder.writeObject(keyPair[0]);
            encoder.writeObject(keyPair[1]);

            encoder.flush();
            */

            System.out.println(keyPair[0].toString());
            System.out.println(keyPair[1].toString());
        }
        catch (Exception e)
        {
            e.printStackTrace();
        }
    }

}
