/*
 * Copyright (c) 2005, John Mettraux, OpenWFE.org
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * . Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.  
 * 
 * . Redistributions in binary form must reproduce the above copyright notice, 
 *   this list of conditions and the following disclaimer in the documentation 
 *   and/or other materials provided with the distribution.
 * 
 * . Neither the name of the "OpenWFE" nor the names of its contributors may be
 *   used to endorse or promote products derived from this software without
 *   specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: Scheduler.java 2505 2006-04-17 15:11:13Z jmettraux $
 */

//
// Scheduler.java
//
// john.mettraux@openwfe.org
//
// generated with 
// jtmpl 1.1.01 2004/05/19 (john.mettraux@openwfe.org)
//

package openwfe.org.time;


/**
 * A scheduler.
 *
 * <p><font size=2>CVS Info :
 * <br>$Author: jmettraux $
 * <br>$Id: Scheduler.java 2505 2006-04-17 15:11:13Z jmettraux $ </font>
 *
 * @author john.mettraux@openwfe.org
 */
public class Scheduler
{

    private final static org.apache.log4j.Logger log = org.apache.log4j.Logger
        .getLogger(Scheduler.class.getName());

    //
    // CONSTANTS & co

    //
    // FIELDS

    private String name = null;

    private long sleepTime = 250L;

    private SchedulerThread thread = null;

    private java.util.List pendingJobs = new java.util.LinkedList();

    private boolean daemonIfIdle = false;

    private java.util.HashMap cronEntries = null;

    private long lastGivenCronId = -1L;

    private long lastCronMinute = -1L;

    //
    // CONSTRUCTORS

    public Scheduler ()
    {
        this("scheduler");
    }

    public Scheduler (final String name)
    {
        super();

        this.name = name;

        this.thread = new SchedulerThread(this.name);

        this.cronEntries = new java.util.LinkedHashMap();
    }

    //
    // public METHODS

    /**
     * Returns this scheduler name.
     */
    public String getName ()
    {
        return this.name;
    }

    /**
     * Is equivalent to Thread.setDaemon(b).
     */
    public void setDaemon (final boolean b)
    {
        //log.debug("setDaemon() to "+b);

        this.thread.setDaemon(b);
    }

    /**
     * Is equivalent to Thread.isDaemon().
     */
    public boolean isDaemon ()
    {
        return this.thread.isDaemon();
    }

    /**
     * When this value is set to true and that there are no pending jobs,
     * the scheduling tread will flag itself as a daemon thread allowing
     * the VM to exit unblocked.
     * By default this value is set to 'false'.<br>
     * Cron jobs do not count : if there are no pending jobs but cron entries,
     * the scheduler is considered idle anyway.
     */
    public void setDaemonIfIdle (final boolean b)
    {
        this.daemonIfIdle = b;
    }

    /**
     * @see #setDaemonIfIdle
     */
    public boolean isDaemonIfIdle ()
    {
        return this.daemonIfIdle;
    }

    /**
     * The default precision is 0.25, implying that the scheduler will wake
     * up every 250ms (4 times per sec).
     */
    public void setPrecision (final float f)
    {
        this.sleepTime = (int)(f * 1000);
    }

    /**
     * Stops this scheduler
     */
    public void stop ()
    {
        this.thread.setRunning(false);

        this.thread = new SchedulerThread(this.name);
    }

    /**
     * Starts this scheduler
     */
    public void start ()
    {
        this.thread.start();
    }

    /**
     * Registers a Schedule for given execution at a given time (at).
     *
     * @returns true if the schedule was accepted.
     */
    public synchronized boolean scheduleAt 
        (final long at, final Schedulable s, final Object[] params)
    {
        //log.debug
        //    ("scheduleAt() at "+new java.util.Date(at)+
        //     " for "+s.getClass().getName());

        if (at <= System.currentTimeMillis() + this.sleepTime) 
            return false;

        //
        // the job will be inserted : 
        // sets the daemon as a non-daemon thread immediately

        restartThread(false);

        //
        // continue job scheduling

        final JobEntry newjob = new JobEntry(at, s, params);

        if (this.pendingJobs.size() < 1)
        {
            this.pendingJobs.add(newjob);
            return true;
        }

        int insertionPoint = 0;

        for (int i=0; i<=this.pendingJobs.size(); i++)
        {
            if (i == this.pendingJobs.size())
            {
                this.pendingJobs.add(newjob);
                break; // or continue...
            }

            final JobEntry je = (JobEntry)this.pendingJobs.get(i);

            if (at == je.getAt())
            {
                this.pendingJobs.add(i, newjob);
                break;
            }

            if (at > je.getAt())
            {
                insertionPoint = i;
                continue;
            }

            // if (at < je.getAt())

            this.pendingJobs.add(insertionPoint, newjob);
            break;
        }

        return true;
    }

    /**
     * Schedules a job for a given iso date
     *
     * @see openwfe.org.time.Time
     */
    public boolean scheduleAt 
        (final String isoDate, final Schedulable s, final Object[] params)
    throws 
        java.text.ParseException
    {
        return scheduleAt(Time.fromIsoDate(isoDate), s, params);
    }

    /**
     * Schedules a job to occur in a given 'in' time.
     */
    public boolean scheduleIn 
        (final long in, final Schedulable s, final Object[] params)
    {
        return scheduleAt(System.currentTimeMillis() + in, s, params);
    }

    /**
     * Schedules a job to occur in a given 'in' time; this 'in' String
     * is expressed as a time string. 
     *
     * @see openwfe.org.time.Time
     */
    public boolean scheduleIn 
        (final String in, final Schedulable s, final Object[] params)
    {
        return scheduleIn(Time.parseTimeString(in), s, params);
    }

    /**
     * Schedules a 'cron job'.
     *
     * @see CronLine
     * @return a long id 
     */
    public synchronized Long schedule
        (final CronLine cl, final Schedulable s, final Object[] params)
    {
        this.lastGivenCronId++;

        final Long cronId = new Long(this.lastGivenCronId);

        this.cronEntries.put(cronId, new CronEntry(cl, s, params));

        return cronId;
    }

    /**
     * Schedules a 'cron job'.
     *
     * @see CronLine
     * @return a long id 
     */
    public Long schedule
        (final String sCron, final Schedulable s, final Object[] params)
    {
        return schedule(CronLine.parse(sCron), s, params);
    }

    /**
     * Unschedules a given cron job
     */
    public synchronized boolean unschedule (final Long cronId)
    {
        final Object o = this.cronEntries.remove(cronId);

        return (o != null);
    }

    //
    // METHODS

    private synchronized void restartThread (final boolean daemon)
    {
        if (daemon == this.thread.isDaemon()) return;
            //
            // if the thread is already in the requested state,
            // then leave it like that

        this.thread.setRunning(false);

        this.thread = new SchedulerThread(this.name);
        this.thread.setDaemon(daemon);
        this.thread.start();
    }

    /**
     * This is the method called by the scheduling thread and actually 
     * triggering the scheduled jobs.
     */
    protected synchronized void wakeup ()
    {
        while (true)
        {
            final long now = System.currentTimeMillis();

            //
            // cron entries

            final long minute = now / 60000;

            //log.debug("wakeup() minute : "+minute);
            //log.debug("wakeup() lastCronMinute : "+this.lastCronMinute);

            if (minute > this.lastCronMinute)
            {
                //log.debug("wakeup() cron!");

                this.lastCronMinute = minute;

                final java.util.Iterator it = 
                    this.cronEntries.keySet().iterator();
                while (it.hasNext())
                {
                    final CronEntry ce = 
                        (CronEntry)this.cronEntries.get(it.next());

                    if (ce.matches(now)) ce.trigger();
                }
            }

            //
            // pending jobs

            //log.debug
            //    ("wakeup() pending jobs :      "+this.pendingJobs.size());
            //log.debug
            //    ("wakeup() thread.isDaemon() : "+this.thread.isDaemon());

            if (this.pendingJobs.size() < 1)
            {
                //log.debug("wakeup() no more pending jobs...");

                if (this.daemonIfIdle)
                {
                    //log.debug
                    //    ("wakeup() "+
                    //     "daemonIfIdle:true restartThread(daemon:true)");

                    restartThread(true);
                }

                break;
            }

            final JobEntry job = (JobEntry)this.pendingJobs.get(0);

            if (job.getAt() > now) break;

            if (job.getAt() <= now) job.trigger();

            this.pendingJobs.remove(0);
        }
    }

    //
    // STATIC METHODS

    //
    // INNER CLASSES

    private static class Entry
    {
        protected Schedulable target = null;
        protected Object[] params = null;

        public Entry (final Schedulable s, final Object[] params)
        {
            super();

            this.target = s;
            this.params = params;
        }

        public void trigger ()
        {
            this.target.trigger(this.params);
        }
    }

    private static class CronEntry extends Entry
    {
        protected CronLine cronLine = null;

        public CronEntry
            (final CronLine cl,
             final Schedulable s,
             final Object[] params)
        {
            super(s, params);

            this.cronLine = cl;
        }

        public boolean matches (final long t)
        {
            return this.cronLine.matches(t);
        }
    }

    private static class JobEntry extends Entry
    {
        protected long at = -1;

        public JobEntry 
            (final long at, 
             final Schedulable target, 
             final Object[] params)
        {
            super(target, params);

            this.at = at;
        }

        public long getAt ()
        {
            return this.at;
        }
    }

    private class SchedulerThread 
        
        extends Thread

    {
        private boolean running = false;

        public SchedulerThread (final String name)
        {
            super();

            this.setName(name);
        }

        public void run ()
        {
            while (true)
            {
                try
                {

                    //
                    // wakeup

                    if ( ! this.running) break;

                    Scheduler.this.wakeup();

                    //
                    // sleep

                    this.sleep(Scheduler.this.sleepTime);

                }
                catch (final InterruptedException ie)
                {
                    // ignore
                }
                catch (final Throwable t)
                {
                    log.debug("run() schedule problem", t);
                }
            }
        }

        public void start ()
        {
            if (this.running) return;

            this.running = true;

            super.start();
        }

        public void setRunning (final boolean b)
        {
            this.running = b;
        }
    }

    //
    // MAIN METHOD

    public static void main (String[] args)
        throws Exception
    {
        final Scheduler s = new Scheduler();
        s.setDaemon(false);
        s.start();

        System.out.println("* Scheduler started at "+(new java.util.Date()));

        //
        // test one

        long at = System.currentTimeMillis() + 5 * 1000;

        System.out.println("    + scheduling job at "+(new java.util.Date(at)));

        s.scheduleAt
            (at,
             new Schedulable ()
             {
                 public void trigger (Object[] params)
                 {
                     System.out.println
                        (". (1) triggered at "+(new java.util.Date()));
                 }

                 public Long reschedule (Scheduler s)
                 {
                     // nada
                     return new Long(-1);
                 }
             },
             null);

        s.schedule
            (CronLine.parse("4-6/2 * * * *"),
             new Schedulable ()
             {
                 public void trigger (Object[] params)
                 {
                     System.out.println
                        (". (c) triggered at "+(new java.util.Date()));
                 }

                 public Long reschedule (Scheduler s)
                 {
                     // nada
                     return new Long(-1);
                 }
             },
             null);

        //
        // test two

        /*
        final long every = 10 * 1000;
        at = System.currentTimeMillis() + 7 * 1000;

        System.out.println
            ("    + scheduling job at "+(new java.util.Date(at))+
             " and then every 10s");

        s.scheduleAt
            (at,
             new Schedulable ()
             {
                 public void trigger (Object[] params)
                 {
                     System.out.println
                        (". (2) triggered at "+(new java.util.Date()));
                     
                     s.scheduleIn(every, this, null);
                 }
             },
             null);
         */

        //s.setDaemonIfIdle(true);
    }

}
