/*
 * Copyright (c) 2005, John Mettraux, OpenWFE.org
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * . Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.  
 * 
 * . Redistributions in binary form must reproduce the above copyright notice, 
 *   this list of conditions and the following disclaimer in the documentation 
 *   and/or other materials provided with the distribution.
 * 
 * . Neither the name of the "OpenWFE" nor the names of its contributors may be
 *   used to endorse or promote products derived from this software without
 *   specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: XconfBuilder.java 2495 2006-04-16 13:44:21Z jmettraux $
 */

//
// XconfBuilder.java
//
// jmettraux@openwfe.org
//
// generated with 
// jtmpl 1.0.04 20.11.2001 John Mettraux (jmettraux@openwfe.org)
//

package openwfe.org.xconf;

import openwfe.org.Utils;
import openwfe.org.ServiceException;
import openwfe.org.xml.XmlUtils;


/**
 * An XconfBuilder is dedicated to extract configuration out of an XML file.
 * The bulk of the extract work is done by an XconfElementBuilder, but the
 * builder takes care of checking if the XML file changed, potentially
 * triggering a config refreshment.
 *
 * <p><font size=2>CVS Info :
 * <br>$Author: jmettraux $
 * <br>$Date: 2006-04-16 15:44:21 +0200 (Sun, 16 Apr 2006) $
 * <br>$Id: XconfBuilder.java 2495 2006-04-16 13:44:21Z jmettraux $ </font>
 *
 * @author jmettraux@openwfe.org
 */
public class XconfBuilder
{

    private final static org.apache.log4j.Logger log = org.apache.log4j.Logger
        .getLogger(XconfBuilder.class.getName());

    //
    // CONSTANTS (definitions)

    //
    // FIELDS

    private XconfElementBuilder eltBuilder = null;

    private java.net.URL sourceUrl = null;
    private long sourceLastModified = 0;

    //
    // CONSTRUCTORS

    /**
     * Instantiates an XconfBuilder for a given config url and a associated
     * with a given implementation of XconfElementBuilder. The detail
     * work is done by the XconfElementBuilder, the bulk work (+ auto refresh)
     * is done by the XconfBuilder.
     */
    public XconfBuilder 
        (String configurationFileUrl, final XconfElementBuilder eltBuilder)
    throws
        ServiceException
    {
        this.eltBuilder = eltBuilder;

        //configurationFileUrl = Utils.ensureProtocol(configurationFileUrl);
            //
            // done in XmlUtils.extractXml(url)

        log.debug("XconfBuilder() considering url "+configurationFileUrl);

        if (configurationFileUrl.startsWith("resource:"))
        {
            this.sourceUrl = this.getClass()
                .getResource(configurationFileUrl.substring(9));
        }
        else
        {
            try
            {
                this.sourceUrl = new java.net.URL(configurationFileUrl);
            }
            catch (final java.net.MalformedURLException mue)
            {
                throw new ServiceException
                    ("Could not build URL out of '"+configurationFileUrl+"'", 
                     mue);
            }
        }

        //buildConfig(0);
    }

    //
    // METHODS

    /**
     * Returns the URL from which this builder gets config info.
     */
    public java.net.URL getSourceUrl () { return this.sourceUrl; }

    /**
     * Builds the configuration.
     */
    public void buildConfig ()
        throws ServiceException
    {
        this.eltBuilder.clearConfig();
            // clear configuration before reparsing and reloading...

        buildConfig(this.sourceUrl);

        this.sourceLastModified = sourceLastModified();
    }

    private void buildConfig (final java.net.URL mapUrl)
        throws ServiceException
    {
        log.debug("buildConfig(u) building "+mapUrl);

        this.eltBuilder.clearConfig();

        org.jdom.Element rootElement = null;
        try
        {
            rootElement = XmlUtils.extractXml(mapUrl, false);
        }
        catch (final Throwable t)
        {
            throw new ServiceException
                ("Failed to build configuration out of '"+mapUrl+"'", t);
        }

        //
        // build participants

        final java.util.Iterator it = rootElement.getChildren().iterator();
        while (it.hasNext())
        {
            final org.jdom.Element elt = (org.jdom.Element)it.next();

            if (elt.getName().equals("include"))
            {
                final String sIncludeUrl = elt.getAttributeValue("url");
                try
                {
                    buildConfig(new java.net.URL(sIncludeUrl));
                }
                catch (final Throwable t)
                {
                    log.warn
                        ("buildConfig(u) cannot build include "+
                         sIncludeUrl+". Skipped", t);
                }

                continue;
            }

            this.eltBuilder.parseElement(this, elt);
        }
    }

    /*
     * If this method returns a number inferior or than zero, it means the
     * source hasn't been modified; else it returns the current last 
     * modification time of the source.
     */
    private long sourceLastModified ()
    {
        if (this.sourceUrl == null) return -1;

        final long fileLastModified = Utils.getLastModified(this.sourceUrl);

        log.debug
            ("newLastModified() this.sourceLastModified = "+
             this.sourceLastModified);
        log.debug
            ("newLastModified() fileLastModified        = "+
             fileLastModified);

        if (this.sourceLastModified < fileLastModified) 
            return fileLastModified;

        return -1;
    }

    /**
     * Will return true if the source has changed since the last time
     * this builder parsed it.
     */
    public boolean hasSourceBeenModified ()
    {
        return (sourceLastModified() > -1);
    }

    /**
     * You can call this method to check wether this instance should
     * reset and reload its content, ie if its config file has changed.
     */
    public void refreshConfig ()
        throws ServiceException
    {
        if (hasSourceBeenModified())
        {
            log.info
                ("refreshConfig() config file "+this.sourceUrl+
                 " modified => reparsing");

            //this.buildConfig(newLastModified);
            this.buildConfig();
        }
    }

}
