/*
 * Copyright (c) 2005, John Mettraux, OpenWFE.org
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * . Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.  
 * 
 * . Redistributions in binary form must reproduce the above copyright notice, 
 *   this list of conditions and the following disclaimer in the documentation 
 *   and/or other materials provided with the distribution.
 * 
 * . Neither the name of the "OpenWFE" nor the names of its contributors may be
 *   used to endorse or promote products derived from this software without
 *   specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: MapUtils.java 2377 2006-03-02 13:06:49Z jmettraux $
 */

//
// MapUtils.java
//
// john.mettraux@openwfe.org
//
// generated with 
// jtmpl 1.1.00 16.08.2003 John Mettraux (jmettraux@openwfe.org)
//

package openwfe.org;

import openwfe.org.time.Time;


/**
 * Methods for extracting values out of maps
 *
 * <p><font size=2>CVS Info :
 * <br>$Author: jmettraux $
 * <br>$Id: MapUtils.java 2377 2006-03-02 13:06:49Z jmettraux $ </font>
 *
 * @author john.mettraux@openwfe.org
 */
public abstract class MapUtils
{

    private final static org.apache.log4j.Logger log = org.apache.log4j.Logger
        .getLogger(MapUtils.class.getName());

    //
    // METHODS

    public static String getAsString (java.util.Map m, String key)
    {
        Object value = m.get(key);
        if (value == null) return null;
        return value.toString();
    }

    public static String getAsString
        (java.util.Map m, String key, String defaultValue)
    {
        String value = getAsString(m, key);
        if (value == null) return defaultValue;
        return value;
    }

    public static String getMandatoryString 
        (final java.util.Map m, final String key)
    {
        final String result = getAsString(m, key);

        if (result == null)
        {
            throw new IllegalArgumentException
                ("mandatory parameter '"+key+"' is missing.");
        }

        return result;
    }

    /**
     * Tries to map to an integer the value of parameter given by its key.
     * Will throw an IllegalArgumentException if it can't.
     */
    public static int getMandatoryInt
        (final java.util.Map m, final String key)
    {
        final String sValue = getAsString(m, key);

        if (sValue == null)
        {
            throw new IllegalArgumentException
                ("mandatory int parameter '"+key+"' is missing.");
        }

        try
        {
            return Integer.parseInt(sValue);
        }
        catch (final NumberFormatException e)
        {
            throw new IllegalArgumentException
                ("mandatory int parameter '"+key+
                 "' has value '"+sValue+
                 "' which cannot be mapped to an integer");
        }
    }

    /**
     * Returns a Long and a String : the count in milliseconds described as a
     * 'time string' in the String, a 'time string' is something like
     * "3h4m30s".
     */
    public static Object[] getMandatoryTimeAndString
        (final java.util.Map m, final String key)
    {
        final String sValue = getMandatoryString(m, key);

        try
        {
            final long tValue = Time.parseTimeString(sValue);

            return new Object[] { new Long(tValue), sValue };
        }
        catch (final NumberFormatException e)
        {
            throw new IllegalArgumentException
                ("Failed to turn '"+sValue+"' into a count of milliseconds.");
        }
    }

    public static long getAsTime 
        (final java.util.Map m, final String key)
    throws 
        NumberFormatException
    {
        String sValue = getAsString(m, key);
        if (sValue == null)
        {
            throw new NumberFormatException
                ("No assigned value for '"+key+"'");
        }
        return Time.parseTimeString(sValue);
    }

    public static long getAsTime
        (final java.util.Map m, final String key, long defaultValue)
    throws 
        NumberFormatException
    {
        String sValue = getAsString(m, key);
        if (sValue == null) return defaultValue;
        return Time.parseTimeString(sValue);
    }

    public static long getAsTime
        (final java.util.Map m, final String key, final String defaultValue)
    throws 
        NumberFormatException
    {
        String sValue = getAsString(m, key);
        if (sValue == null) return Time.parseTimeString(defaultValue);
        return Time.parseTimeString(sValue);
    }

    public static long getAsLong 
        (final java.util.Map m, final String key)
    throws 
        NumberFormatException
    {
        String sValue = getAsString(m, key);
        if (sValue == null) 
        {
            throw new NumberFormatException
                ("No assigned value for '"+key+"'");
        }
        return Long.parseLong(sValue);
    }

    public static long getAsLong 
        (final java.util.Map m, final String key, final long defaultValue)
    {
        String sValue = getAsString(m, key);
        if (sValue == null) return defaultValue;
        try
        {
            return Long.parseLong(sValue);
        }
        catch (NumberFormatException nfe)
        {
            log.info
                ("'"+sValue+"' isn't a long value. Returning default "+
                 defaultValue);
            return defaultValue;
        }
    }

    public static int getAsInt 
        (final java.util.Map m, final String key, final int defaultValue)
    {
        String sValue = getAsString(m, key);
        if (sValue == null) return defaultValue;
        try
        {
            return Integer.parseInt(sValue);
        }
        catch (NumberFormatException nfe)
        {
            log.info
                ("'"+sValue+"' isn't an int value. Returning default "+
                 defaultValue);
            return defaultValue;
        }
    }

    public static boolean getAsBoolean 
        (final java.util.Map m, final String key, final boolean defaultValue)
    {
        final Object o = m.get(key);

        if (o == null) return defaultValue;

        if (o instanceof Boolean) return ((Boolean)o).booleanValue();

        String sValue = o.toString();

        return sValue.equalsIgnoreCase("true");
    }

    public static double getAsDouble 
        (final java.util.Map m, final String key)
    throws 
        NumberFormatException
    {
        String sValue = getAsString(m, key);
        if (sValue == null) 
        {
            throw new NumberFormatException
                ("No assigned value for '"+key+"'");
        }
        return Double.parseDouble(sValue);
    }

    /**
     * Returns a float value for the given parameter name.
     */
    public static float getAsFloat 
        (final java.util.Map m, 
         final String paramName, 
         final float defaultValue)
    {
        try
        {
            return Float.parseFloat(getAsString(m, paramName));
        }
        catch (final Throwable t)
        {
        }
        return defaultValue;
    }

}
