/*
 * Copyright (c) 2005, John Mettraux, OpenWFE.org
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * . Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.  
 * 
 * . Redistributions in binary form must reproduce the above copyright notice, 
 *   this list of conditions and the following disclaimer in the documentation 
 *   and/or other materials provided with the distribution.
 * 
 * . Neither the name of the "OpenWFE" nor the names of its contributors may be
 *   used to endorse or promote products derived from this software without
 *   specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: SqlPasswdCodec.java 2027 2005-08-12 11:00:54Z jmettraux $
 */

//
// SqlPasswdCodec.java
//
// john.mettraux@openwfe.org
//
// generated with 
// jtmpl 1.1.01 2004/05/19 (john.mettraux@openwfe.org)
//

package openwfe.org.auth.sql;

import openwfe.org.ReflectionUtils;
import openwfe.org.ServiceException;
import openwfe.org.ApplicationContext;
import openwfe.org.sql.SqlUtils;
import openwfe.org.sql.SqlParameters;
import openwfe.org.sql.ds.OwfeDataSource;
import openwfe.org.auth.Grant;
import openwfe.org.auth.Passwd;
import openwfe.org.auth.PasswdCodec;
import openwfe.org.auth.Permission;
import openwfe.org.auth.Principal;
import openwfe.org.auth.AuthException;


/**
 * Retrieving passwd info from a database (and storing it into a database)
 *
 * <p><font size=2>CVS Info :
 * <br>$Author: jmettraux $
 * <br>$Id: SqlPasswdCodec.java 2027 2005-08-12 11:00:54Z jmettraux $ </font>
 *
 * @author john.mettraux@openwfe.org
 */
public class SqlPasswdCodec

    implements PasswdCodec

{

    private final static org.apache.log4j.Logger log = org.apache.log4j.Logger
        .getLogger(SqlPasswdCodec.class.getName());

    //
    // CONSTANTS & co

    //protected final static String USER_TABLE = "puser";
    protected final static String T_GRANTED = "pgranted";
    protected final static String T_GRANT = "pgrant";
    protected final static String T_PARAMETER = "pparameter";

    protected final static String C_NAME = "name";
    protected final static String C_CODEBASE = "codebase";
    protected final static String C_USERNAME = "username";
    protected final static String C_GRANTNAME = "grantname";

    protected final static String G_PRINCIPAL = "principal";
    protected final static String G_PERMISSION = "permission";

    protected final SqlParameters sqlParams = new SqlParameters
        (T_PARAMETER,
         "pid",
         "pgroup",
         "pkey",
         "pval");

    //
    // FIELDS

    private ApplicationContext context = null;
    private java.util.Map serviceParams = null;

    //
    // CONSTRUCTORS

    public void init
        (final ApplicationContext context,
         final java.util.Map serviceParams)
    {
        this.context = context;
        this.serviceParams = serviceParams;
    }

    //
    // METHODS from PasswdCodec

    public void encodePasswd (final Passwd p) 
        throws AuthException
    {

        OwfeDataSource ds = null;
        try
        {
            ds = SqlUtils.lookupDataSource(context, serviceParams);
        }
        catch (final ServiceException se)
        {
            throw new AuthException
                ("didn't find data source", se);
        }

        java.sql.Statement st = null;

        //
        // clean tables
        
        try
        {
            st = ds.getConnection().createStatement();
            
            //st.execute("DELETE FROM "+T_USER);
            st.execute("DELETE FROM "+T_GRANTED);
            st.execute("DELETE FROM "+T_GRANT);
            st.execute("DELETE FROM "+T_PARAMETER);
        }
        catch (java.sql.SQLException se)
        {
            SqlUtils.closeStatement(st);

            throw new AuthException
                ("Failed to clean tables", se);
        }

        //
        // re-insert passwd data

        /* no need to insert 'name' and 'principalCodecClass' */

        // insert users (principals)
        
        try
        {

            java.util.Iterator it = p.getPrincipalMap().values().iterator();
            while (it.hasNext())
            {
                Principal principal = (Principal)it.next();

                encodePrincipal(ds.getConnection(), principal);
            }
        }
        catch (final java.sql.SQLException se)
        {
            throw new AuthException
                ("Failed to insert principal parameters", se);
        }
        
        // insert grants
        
        try
        {
            java.util.Iterator it = p.getGrantMap().values().iterator();
            while (it.hasNext())
            {
                Grant g = (Grant)it.next();
                encodeGrant(ds.getConnection(), g);
            }
        }
        catch (java.sql.SQLException se)
        {
            throw new AuthException
                ("Failed to insert grants", se);
        }

        // that's all folks !
        
        ds.releaseConnection();
    }

    public Passwd decodePasswd (final Object o)
        throws AuthException
    {
        return this.decodePasswd(null);
    }

    public Passwd decodePasswd ()
        throws AuthException
    {
        OwfeDataSource ds = null;
        try
        {
            ds = SqlUtils.lookupDataSource(context, serviceParams);
        }
        catch (final ServiceException se)
        {
            throw new AuthException
                ("didn't find data source", se);
        }

        try
        {
            return new Passwd
                ("passwd",  // ???????????TODO??????????????????????????????
                 decodePrincipals(ds.getConnection()), 
                 decodeGrants(ds.getConnection()));
        }
        catch (Exception e)
        {
            throw new AuthException
                ("failed to load passwd info", e);
        }
        finally
        {
            ds.releaseConnection();
        }
    }

    //
    // METHODS

    //
    // decode

    protected java.util.Map decodePrincipals (final java.sql.Connection con)
        throws Exception
    {
        final java.util.List rawPrincipals = 
            this.sqlParams.extractParameters
                (con,
                 G_PRINCIPAL,
                 C_NAME);

        final java.util.Map principals = 
            new java.util.HashMap(rawPrincipals.size());

        final java.util.Iterator it = rawPrincipals.iterator();
        while (it.hasNext())
        {
            final java.util.Map params = (java.util.Map)it.next();

            final Principal principal = (Principal)ReflectionUtils
                .initObject(params);

            /*
            log.debug
                ("decodePrincipals() principal named '"+
                 principal.getName()+"'");
            */

            principals.put(principal.getName(), principal);
        }

        decodeGranted(con, principals);

        return principals;
    }

    protected void decodeGranted 
        (final java.sql.Connection con, final java.util.Map principals)
    throws
        java.sql.SQLException
    {
        final StringBuffer sb = new StringBuffer();
        sb.append("SELECT ");
        sb.append(C_USERNAME); sb.append(", ");
        sb.append(C_GRANTNAME);
        sb.append(" FROM "); sb.append(T_GRANTED);
        //sb.append(";");
        final String query = sb.toString();

        java.sql.Statement st = null;
        java.sql.ResultSet rs = null;
        try
        {
            st = con.createStatement();
            rs = st.executeQuery(query);

            while (rs.next())
            {
                final String username = rs.getString(1);
                final String grantname = rs.getString(2);

                log.debug("decodeGranted() looking for >"+username+"<");

                final Principal principal = (Principal)principals.get(username);

                if (principal == null)
                {
                    log.warn
                        ("decodeGranted() username >"+username+
                         "< not found. Skipping grant.");
                    continue;
                }

                principal.addGrant(grantname);
            }
        }
        finally
        {
            SqlUtils.closeStatement(st, rs);
        }
    }

    protected java.util.Map decodeGrants (final java.sql.Connection con)
        throws Exception
    {
        final StringBuffer sb = new StringBuffer();
        sb.append("SELECT ");
        sb.append(C_NAME); sb.append(", ");
        sb.append(C_CODEBASE);
        sb.append(" FROM "); sb.append(T_GRANT);
        //sb.append(";");
        final String query = sb.toString();

        final java.util.Map grants = new java.util.HashMap();

        java.sql.Statement st = null;
        java.sql.ResultSet rs = null;
        try
        {
            st = con.createStatement();
            rs = st.executeQuery(query);

            while (rs.next())
            {
                final String grantName = rs.getString(1);
                final String codebase = rs.getString(2);

                final Grant g = new Grant(grantName, codebase);

                grants.put(grantName, g);
            }
        }
        finally
        {
            SqlUtils.closeStatement(st, rs);
        }

        decodePermissions(con, grants);

        return grants;
    }

    protected void decodePermissions 
        (final java.sql.Connection con, final java.util.Map grants)
    throws
        Exception
    {
        final java.util.List rawPermissions = 
            this.sqlParams.extractParameters
                (con,
                 G_PERMISSION,
                 C_GRANTNAME);
        
        final java.util.Iterator it = rawPermissions.iterator();
        while (it.hasNext())
        {
            final java.util.Map params = (java.util.Map)it.next();

            //log.debug("decodePermissions() class "+params.get("class"));
            //log.debug("decodePermissions() grant "+params.get("grantname"));

            /*
            final Permission permission = (Permission)ReflectionUtils
                .initObject(params);
            */

            final String className = (String)params
                .get(ReflectionUtils.P_CLASS);
            final Class permissionClass = 
                Class.forName(className);
            final Permission permission = (Permission)ReflectionUtils
                .buildInstance(permissionClass, params);

            final Grant grant = (Grant)grants
                .get((String)params.get(C_GRANTNAME));
            grant.addPermission(permission);
        }
    }

    //
    // SOME POETRY

    /*
     * Effeuille la rose des vents
     * Voici que bruissent les orages dchans
     * Les trains roulent en tourbillon sur les rseaux enchevtrs
     * Bilboquets diaboliques
     * Il y a des trains qui ne se rencontrent jamais
     * D'autres se perdent en route
     * Les chefs de gare jouent aux checs
     * Tric-trac
     * Billard
     * Caramboles
     * Paraboles
     * La voie ferre est une nouvelle gomtrie
     * Syracuse
     * Archimde
     * Et les soldats qui l'gorgrent
     * Et les galres
     * Et les vaisseaux
     * Et les engins prodigieux qu'il inventa
     * Et toutes les tueries
     * L'histoire antique
     * L'histoire moderne
     * Les tourbillons
     * Les naufrages
     * Mme celui du Titanic que j'ai lu dans le journal
     * Autant d'images-associations que je ne peux pas dvelopper dans mes vers
     * Car je suis encore fort mauvais pote
     * Car l'univers me dborde
     * Car j'ai nglig de m'assurer contre les accidents de chemin de fer
     * Car je ne sais pas aller jusqu'au bout
     * Et j'ai peur.
     *
     *                       Un fragment de
     *                       
     *                       'La prose du Transsibrien 
     *                        et de la petite Jeanne de France'
     *
     *                           de Blaise Cendrars
     */

    // 
    // encode

    protected void encodePrincipal
        (final java.sql.Connection con, final Principal principal)
    throws 
        java.sql.SQLException
    {
        //
        // insert principals

        final long id = this.sqlParams.insertParams
            (con, 
             G_PRINCIPAL, 
             principal.getInitParameters());

        //
        // link grant to users
        
        java.sql.Statement st = null;
        try
        {
            java.util.Iterator it = principal.getGrants().iterator();
            while (it.hasNext())
            {
                final String grantName = (String)it.next();

                final StringBuffer sb = new StringBuffer();

                sb.append("INSERT INTO ");
                sb.append(T_GRANTED);
                sb.append(" (");
                sb.append(C_USERNAME);
                sb.append(", ");
                sb.append(C_GRANTNAME);
                sb.append(") VALUES (");
                sb.append(SqlUtils.prepareString(principal.getName()));
                sb.append(", ");
                sb.append(SqlUtils.prepareString(grantName));
                //sb.append(");");
                sb.append(")");

                final String sInsert = sb.toString();

                st = con.createStatement();

                st.execute(sInsert);
            }
        }
        finally
        {
            SqlUtils.closeStatement(st);
        }
    }

    protected void encodeGrant 
        (final java.sql.Connection con, final Grant grant)
    throws 
        java.sql.SQLException
    {
        java.sql.Statement st = null;
        try
        {
            //
            // prepare codebase

            final String codebase = grant.getUrl();

            log.debug("encodeGrant() codebase is >"+codebase+"<");

            //
            // insert grants

            final StringBuffer sb = new StringBuffer();

            sb.append("INSERT INTO ");
            sb.append(T_GRANT);
            sb.append(" (");
            sb.append(C_NAME); sb.append(", ");
            sb.append(C_CODEBASE);
            sb.append(") VALUES (");
            sb.append(SqlUtils.prepareString(grant.getName()));
            sb.append(", ");
            sb.append(SqlUtils.prepareString(codebase));
            //sb.append(");");
            sb.append(")");

            final String sInsert = sb.toString();

            st = con.createStatement();
            st.execute(sInsert);

            //
            // insert permissions

            java.util.Enumeration en = grant.getPermissions().elements();
            while (en.hasMoreElements())
            {
                final Permission permission = (Permission)en.nextElement();

                final java.util.Map pParams = 
                    new java.util.HashMap(permission.getParameters().size());

                pParams.put
                    (C_GRANTNAME, grant.getName());
                pParams.put
                    (ReflectionUtils.P_CLASS, permission.getClass().getName());

                pParams.putAll(permission.getParameters());

                this.sqlParams.insertParams(con, G_PERMISSION, pParams);
            }
        }
        finally
        {
            SqlUtils.closeStatement(st);
        }
    }

    //
    // STATIC METHODS

    /* *
     * Fetch data source from context
     * /
    protected static OwfeDataSource lookupDataSource 
        (final ApplicationContext context,
         final java.util.Map serviceParams)
    {
        final String dataSourceName = 
            (String)serviceParams.get(OwfeDataSource.DATA_SOURCE);

        return (OwfeDataSource)context.lookup(dataSourceName);
    }
     */

    //
    // INNER CLASSES

}
