/*
 * Copyright (c) 2005, John Mettraux, OpenWFE.org
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * . Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.  
 * 
 * . Redistributions in binary form must reproduce the above copyright notice, 
 *   this list of conditions and the following disclaimer in the documentation 
 *   and/or other materials provided with the distribution.
 * 
 * . Neither the name of the "OpenWFE" nor the names of its contributors may be
 *   used to endorse or promote products derived from this software without
 *   specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: Cache.java 2351 2006-02-25 10:55:57Z jmettraux $
 */

//
// Cache.java
//
// jmettraux@openwfe.org
//
// generated with 
// jtmpl 1.0.04 20.11.2001 John Mettraux (jmettraux@openwfe.org)
//

package openwfe.org.misc;

import openwfe.org.xml.XmlUtils;


/**
 * A cache implementation : it's in fact a map with a max capacity.
 * When you put an item in a cache that is full, the least recently used item
 * will be thrown out to give its place to the incoming item.
 *
 * <p><font size=2>CVS Info :
 * <br>$Author: jmettraux $
 * <br>$Date: 2006-02-25 11:55:57 +0100 (Sat, 25 Feb 2006) $
 * <br>$Id: Cache.java 2351 2006-02-25 10:55:57Z jmettraux $ </font>
 *
 * @author jmettraux@openwfe.org
 */
public class Cache
{

    private final static org.apache.log4j.Logger log = org.apache.log4j.Logger
        .getLogger(Cache.class.getName());

    //
    // FIELDS

    private int maxSize = -1;
    private LruMap cache = null;

    //
    // CONSTRUCTORS

    /**
     * Builds a cache with a custom maxSize.
     */
    public Cache (final int maxSize)
    {
        super();

        this.maxSize = maxSize;

        if (maxSize < 1) 
        {
            throw new IllegalArgumentException
                ("Cannot build a cache with a maxSize < 1 : "+maxSize);
        }

        this.cache = new LruMap(maxSize);
    }

    /**
     * Builds a cache with the default capacity (maxSize set to 100).
     */
    public Cache ()
    {
        this(100);
    }

    //
    // METHODS

    /**
     * Like the get() method of a Map.
     */
    public synchronized Object get (final Object key)
    {
        return this.cache.get(key);
    }

    /**
     * Like the remove() method of a Map.
     */
    public synchronized Object remove (final Object key)
    {
        return this.cache.remove(key);
    }

    /**
     * Like the put() method of a Map.
     */
    public synchronized void put (final Object key, final Object value)
    {
        this.cache.put(key, value);
    }

    /**
     * Returns the count of items currently stored in this cache.
     */
    public int size ()
    {
        return this.cache.size();
    }

    /**
     * Returns the capacity attributed to this cache.
     */
    public int maxSize ()
    {
        return this.maxSize;
    }

    /**
     * Beware : you have to handle sync by yourself
     */
    public java.util.Iterator iterator ()
    {
        return this.cache.values().iterator();
    }

    /**
     * Like the keySet() method of a Map.
     */
    public java.util.Set keySet ()
    {
        return this.cache.keySet();
    }

    /**
     * This method is used by the getStatus() method of services 
     * using a cache
     */
    public org.jdom.Element getStatus ()
    {
        org.jdom.Element cacheElt = new org.jdom.Element("cache");

        cacheElt.addContent(XmlUtils.getRevisionElt("$Id: Cache.java 2351 2006-02-25 10:55:57Z jmettraux $"));

        java.util.Iterator it = this.cache.keySet().iterator();
        while (it.hasNext())
        {
            org.jdom.Element keyElt = new org.jdom.Element("key");
            keyElt.addContent(it.next().toString());
            cacheElt.addContent(keyElt);
        }

        return cacheElt;
    }

    //
    // INNER CLASSES

    private static class LruMap 
        extends java.util.LinkedHashMap
    {
        private int capacity = -1;

        public LruMap (final int capacity)
        {
            super(capacity, 0.75f, true);
                //
                // setting accessOrder to true (LRU)

            this.capacity = capacity;
        }

        protected boolean removeEldestEntry (final java.util.Map.Entry eldest)
        {
            return (this.size() > this.capacity);
        }
    }

}
