/*
 * Copyright (c) 2005, John Mettraux, OpenWFE.org
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * . Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.  
 * 
 * . Redistributions in binary form must reproduce the above copyright notice, 
 *   this list of conditions and the following disclaimer in the documentation 
 *   and/or other materials provided with the distribution.
 * 
 * . Neither the name of the "OpenWFE" nor the names of its contributors may be
 *   used to endorse or promote products derived from this software without
 *   specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: CronLine.java 2126 2005-09-21 15:11:51Z jmettraux $
 */

//
// CronLine.java
//
// john.mettraux@openwfe.org
//
// generated with 
// jtmpl 1.1.01 2004/05/19 (john.mettraux@openwfe.org)
//

package openwfe.org.time;

import java.util.Calendar;


/**
 * A 'cron line' is a line like you find in a crontab file (man 5 crontab).
 *
 * <p><font size=2>CVS Info :
 * <br>$Author: jmettraux $
 * <br>$Id: CronLine.java 2126 2005-09-21 15:11:51Z jmettraux $ </font>
 *
 * @author john.mettraux@openwfe.org
 */
public class CronLine
{

    private final static org.apache.log4j.Logger log = org.apache.log4j.Logger
        .getLogger(CronLine.class.getName());

    //
    // CONSTANTS & co

    //
    // FIELDS

    private int[] minutes = null;
    private int[] hours = null;
    private int[] days = null;
    private int[] months = null;
    private int[] daysOfWeek = null;

    //
    // CONSTRUCTORS

    protected CronLine ()
    {
        super();
    }

    //
    // METHODS

    public boolean matches (final long t)
    {
        final Calendar cal = Calendar.getInstance();
        cal.setTimeInMillis(t);

        //
        // minutes

        if (noMatch(cal.get(Calendar.MINUTE), minutes))
            return false;

        //
        // hours

        if (noMatch(cal.get(Calendar.HOUR), hours))
            return false;

        //
        // days

        if (noMatch(cal.get(Calendar.DAY_OF_MONTH), days))
            return false;

        //
        // months

        if (noMatch(cal.get(Calendar.MONTH), months))
            return false;

        //
        // days of week

        if (noMatch(cal.get(Calendar.DAY_OF_WEEK), daysOfWeek))
            return false;

        return true;
    }

    private static boolean noMatch (final int value, final int[] cronValues)
    {
        if (cronValues == null) return false;

        //log.debug("noMatch() "+value+" in "+dumpIntArray(cronValues)+" ?");

        for (int i=0; i<cronValues.length; i++)
            if (value == cronValues[i]) return false;

        return true;
    }

    /*
    private static String dumpIntArray (final int[] is)
    {
        final StringBuffer sb = new StringBuffer();

        sb.append("[ ");

        for (int i=0; i<is.length; i++)
        {
            sb.append(is[i]);
            sb.append(", ");
        }
        
        sb.append("]");

        return sb.toString();
    }
    */

    //
    // STATIC METHODS

    /**
     * Parses a cron[tab] string.
     */
    public static CronLine parse (final String cronString)
    {
        final String[] ss = cronString.split(" ");

        final CronLine result = new CronLine();

        //
        // minutes

        result.minutes = parseItem(ss[0], 0, 59);

        //
        // hours

        result.hours = parseItem(ss[1], 0, 24);

        //
        // days

        result.days = parseItem(ss[2], 1, 31);

        //
        // months

        result.months = parseItem(ss[3], 1, 12);

        //
        // days of week

        result.daysOfWeek = parseItem(ss[4], 1, 7);

        // done

        return result;
    }

    private static int[] parseItem 
        (final String s, final int min, final int max)
    {
        //log.debug("parseItem() s >"+s+"<");

        if (s.trim().equals("*")) return null;

        if (s.indexOf(",") > -1) 
        {
            return parseList(s, min, max);
        }

        if (s.indexOf("-") > -1 ||
            s.indexOf("*") > -1)
        {
            return parseRange(s, min, max);
        }

        int i = Integer.parseInt(s);

        if (i < min) i = min;
        else if (i > max) i = max;

        return new int[] { i };
    }

    private static int[] parseList 
        (final String s, final int min, final int max)
    {
        final String[] ss = s.split(",");

        final int[] result = new int[ss.length];

        for (int i=0; i<ss.length; i++)
        {
            result[i] = Integer.parseInt(ss[i]);
            if (result[i] < min) result[i] = min;
            if (result[i] > max) result[i] = max;
        }

        return result;
    }

    private static int[] parseRange
        (final String s, final int min, final int max)
    {
        final int i = s.indexOf("-");
        final int j = s.indexOf("/");

        int inc = 1;
        if (j > -1) inc = Integer.parseInt(s.substring(j+1));


        int start = -1;
        int end = -1;

        if (i > -1)
        {
            start = Integer.parseInt(s.substring(0, i));

            if (j > -1)
                end = Integer.parseInt(s.substring(i+1, j));
            else
                end = Integer.parseInt(s.substring(i+1));
        }
        else // case : */x
        {
            start = min;
            end = max;
        }

        if (start < min) start = min;
        if (end > max) end = max;

        final java.util.List list = new java.util.LinkedList();

        //log.debug("parseRange() start : "+start);
        //log.debug("parseRange() end :   "+end);
        //log.debug("parseRange() inc :   "+inc);

        for (int value = start; value <= end; value = value + inc)
        {
            //log.debug("parseRange() v : "+value);
            list.add(new Integer(value));
        }

        //
        // return result as an int array

        int index = 0;
        final int[] result = new int[list.size()];
        final java.util.Iterator it = list.iterator();
        while (it.hasNext())
            result[index++] = ((Integer)it.next()).intValue();

        return result;
    }

}
