/*
 * Copyright (c) 2001-2006, John Mettraux, OpenWFE.org
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * . Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.  
 * 
 * . Redistributions in binary form must reproduce the above copyright notice, 
 *   this list of conditions and the following disclaimer in the documentation 
 *   and/or other materials provided with the distribution.
 * 
 * . Neither the name of the "OpenWFE" nor the names of its contributors may be
 *   used to endorse or promote products derived from this software without
 *   specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: SqlUtils.java 2713 2006-06-01 14:38:45Z jmettraux $
 */

//
// SqlUtils.java
//
// john.mettraux@openwfe.org
//
// generated with 
// jtmpl 1.1.01 2004/05/19 (john.mettraux@openwfe.org)
//

package openwfe.org.sql;

import openwfe.org.ServiceException;
import openwfe.org.ApplicationContext;
import openwfe.org.sql.ds.OwfeDataSource;


/**
 * SQL related help methods
 *
 * <p><font size=2>CVS Info :
 * <br>$Author: jmettraux $
 * <br>$Id: SqlUtils.java 2713 2006-06-01 14:38:45Z jmettraux $ </font>
 *
 * @author john.mettraux@openwfe.org
 */
public abstract class SqlUtils
{

    private final static org.apache.log4j.Logger log = org.apache.log4j.Logger
        .getLogger(SqlUtils.class.getName());

    //
    // CONSTANTS

    //
    // METHODS

    /**
     * This method is useful in finally clauses.
     *
     * <pre>
     *    java.sql.Statement st = null;
     *    java.sql.ResultSet rs = null;
     *    try
     *    {
     *        st = con.createStatement();
     *        rs = st.executeQuery("SELECT...");
     *        (do something)
     *    }
     *    finally
     *    {
     *        SqlUtils.closeStatement(st, rs);
     *    }
     * </pre>
     *
     * No need to watch for NPE and stuff like this.
     */
    public static void closeStatement 
        (final java.sql.Statement st, final java.sql.ResultSet rs)
    {
        try
        {
            if (rs != null) rs.close();
            if (st != null) st.close();
        }
        catch (final Throwable t)
        {
            log.debug
                ("closeStatement() problem when disposing of sql resources", t);
        }
    }

    /**
     * This method is useful in finally clauses.
     *
     * <pre>
     *    java.sql.Statement st = null;
     *    try
     *    {
     *        st = con.createStatement();
     *        rs = st.executeUpdate("...");
     *    }
     *    finally
     *    {
     *        SqlUtils.closeStatement(st);
     *    }
     * </pre>
     *
     * No need to watch for NPE and stuff like this.
     */
    public static void closeStatement 
        (final java.sql.Statement st)
    {
        closeStatement(st, null);
    }

    /**
     * This method will close the given connection passed to it; it will
     * do it silently, not emitting the slightest Throwable, making it
     * elegant to call from a finally clause, after a SqlUtils.closeStatement()
     * call.
     */
    public static void closeConnection
        (final java.sql.Connection con)
    {
        try
        {
            con.close();
        }
        catch (final Throwable t)
        {
            // ignore
        }
    }

    /**
     * Rolls back a connection (as it's usually done in a finally clause,
     * this method doesn't throw any exception at all for convenience).
     */
    public static void rollback
        (final java.sql.Connection con)
    {
        try
        {
            con.rollback();
        }
        catch (final Throwable t)
        {
            // ignore
        }
    }

    /**
     * Does a SELECT MAX(idColName) FROM tableName.
     */
    public static long getNewInsertId
        (final java.sql.Connection con,
         final String tableName,
         final String idColName)
    throws
        java.sql.SQLException
    {
        java.sql.Statement st = null;
        java.sql.ResultSet rs = null;
        try
        {
            StringBuffer sb = new StringBuffer();

            sb.append("SELECT MAX(");
            sb.append(idColName);
            sb.append(") FROM ");
            sb.append(tableName);

            String query = sb.toString();

            st = con.createStatement();

            rs = st.executeQuery(query);

            if (rs.next())
                return rs.getLong(1)+1;

            return 0;
        }
        finally
        {
            rs.close();
            st.close();
        }
    }

    /**
     * Returns an insert string for a prepared statement.
     */
    public static String buildPreparedInsertString
        (final String tableName,
         final String[] colNames)
    {
        final java.util.List values = new java.util.ArrayList(colNames.length);

        for (int i=0; i<colNames.length; i++) values.add(new Character('?'));

        return buildInsertString(tableName, colNames, values);
    }

    /**
     * Takes care of producing an INSERT string, also takes care of wrapping
     * string values inside quotes and to protect special chars.
     */
    public static String buildInsertString
        (final String tableName,
         final String[] colNames,
         final java.util.List values)
    {
        if (colNames.length != values.size())
        {
            throw new IllegalArgumentException
                ("number of columns doesn't match number of values for table '"+
                 tableName+"'");
        }

        final StringBuffer sb = new StringBuffer();

        sb.append("INSERT INTO ");
        sb.append(tableName);
        sb.append(" (");
        for (int i=0; i<colNames.length; i++)
        {
            sb.append(colNames[i]);
            if (i < colNames.length-1) sb.append(", ");
        }
        sb.append(") VALUES (");
        java.util.Iterator it = values.iterator();
        while (it.hasNext())
        {
            Object oVal = it.next();

            if (oVal == null)
            {
                sb.append("NULL");
            }
            else if (oVal instanceof String)
            {
                sb.append(prepareString(oVal.toString()));
            }
            else if (oVal instanceof java.sql.Date)
            {
                sb
                    .append("'")
                    .append(oVal.toString())
                    .append("'");
            }
            else
            {
                sb.append(oVal.toString());
            }

            if (it.hasNext()) sb.append(", ");
        }
        sb.append(")");

        final String result =  sb.toString();

        //log.debug("buildInsertString() :\n"+result);

        return result;
    }

    /**
     * Builds an 'update' SQL string.
     */
    public static String buildUpdateString
        (final String tableName,
         final String[] colNames,
         final java.util.List values,
         final String whereClause)
    {
        if (colNames.length != values.size())
        {
            throw new IllegalArgumentException
                ("number of columns doesn't match number of values for table '"+
                 tableName+"'");
        }

        final StringBuffer sb = new StringBuffer();

        sb.append("UPDATE ");
        sb.append(tableName);
        sb.append(" SET ");

        final java.util.Iterator it = values.iterator();

        for (int i=0; i<colNames.length; i++)
        {
            sb.append(colNames[i]);

            sb.append(" = ");

            Object oVal = it.next();

            if (oVal == null)
                sb.append("NULL");
            else if (oVal instanceof String)
                sb.append(prepareString(oVal.toString()));
            else if (oVal instanceof java.util.Date)
                sb.append(prepareString(oVal.toString()));
            else
                sb.append(oVal.toString());

            if (i < colNames.length-1) sb.append(", ");
        }

        if (whereClause != null)
        {
            sb.append(" WHERE ");
            sb.append(whereClause);
        }
        else
        {
            log.debug("buildUpdateString() Warning : null 'whereClause' !");
        }

        final String result =  sb.toString();

        if (log.isDebugEnabled())
            log.debug("buildUpdateString() :\n>"+result+"<");

        return result;
    }

    public static String buildDeleteString
        (final String tableName,
         final String whereClause)
    {
        final StringBuffer sb = new StringBuffer();

        sb.append("DELETE FROM "); sb.append(tableName);

        sb.append(" WHERE "); sb.append(whereClause);

        return sb.toString();
    }

    public static String buildQueryString
        (final String tableName,
         final String[] colNames,
         final String whereClause)
    {
        return buildQueryString
            (tableName, colNames, whereClause, null);
    }

    public static String buildQueryString
        (final String tableName,
         final String[] colNames,
         final String whereClause,
         final String orderClause)
    {
        final StringBuffer sb = new StringBuffer();

        sb.append("SELECT ");

        for (int i=0; i<colNames.length; i++)
        {
            sb.append(colNames[i]);
            if (i < colNames.length-1) sb.append(", ");
        }

        sb.append(" FROM ").append(tableName);

        if (whereClause != null)
            sb.append(" WHERE ").append(whereClause);

        if (orderClause != null)
            sb.append(" ORDER BY ").append(orderClause);

        return sb.toString();
    }

    /**
     * Looks in the serviceParams for the value of the parameter named
     * 'dataSource' (openwfe.org.sql.ds.OwfeDataSource.P_DATA_SOURCE), and
     * then given this value (a service name) returns the DataSource 
     * stored in the application context under this name.
     */
    public static OwfeDataSource lookupDataSource
        (final ApplicationContext context,
         final java.util.Map serviceParams)
    throws
        ServiceException
    {
        return OwfeDataSource.lookupDataSource(context, serviceParams);
    }

    /**
     * Replaces occurences of "'" (single quote) in the input String
     * by "''" in the output String, and takes care of surrounding the
     * output String with single quotes (official SQL).
     * If the input is null, "NULL" will be returned.
     *
     * <pre>
     * prepareString("toto d'avry") --&gt; "'toto d''avry'"
     * prepareString(null) --&gt; "NULL"
     * </pre>
     */
    public static String prepareString (final String in)
    {
        if (in == null) return "NULL";

        String result;

        //result = in.replaceAll("'", "\\\\'");
        result = in.replaceAll("'", "''");

        result = "'" + result + "'";

        return result;
    }

    /**
     * Trying to prevent sql injection...
     */
    public static String norm (String in)
    {
        if (in == null) return null;

        in = in.trim();

        if (in.length() < 1) return null;

        int i = in.indexOf("\"");
        int j = in.indexOf(";");

        if (j > -1 && j < i) i = j;

        if (i > -1) in = in.substring(0, i);

        in = in.replaceAll("'", "''");

        return in;

        //return "'" + in + "'";
    }

}
