/*
 * Copyright (c) 2001-2006, John Mettraux, OpenWFE.org
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * . Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.  
 * 
 * . Redistributions in binary form must reproduce the above copyright notice, 
 *   this list of conditions and the following disclaimer in the documentation 
 *   and/or other materials provided with the distribution.
 * 
 * . Neither the name of the "OpenWFE" nor the names of its contributors may be
 *   used to endorse or promote products derived from this software without
 *   specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: Utils.java 2902 2006-07-01 13:31:15Z jmettraux $
 */

//
// Utils.java
//
// jmettraux@openwfe.org
//
// generated with 
// jtmpl 1.0.04 20.11.2001 John Mettraux (jmettraux@openwfe.org)
//

package openwfe.org;

import openwfe.org.misc.ByteUtils;


/**
 * Static methods for misc usages
 *
 * <p><font size=2>CVS Info :
 * <br>$Author: jmettraux $
 * <br>$Date: 2006-07-01 15:31:15 +0200 (Sat, 01 Jul 2006) $
 * <br>$Id: Utils.java 2902 2006-07-01 13:31:15Z jmettraux $ </font>
 *
 * @author john.mettraux@openwfe.org
 */
public abstract class Utils
{

    private final static org.apache.log4j.Logger log = org.apache.log4j.Logger
        .getLogger(Utils.class.getName());


    /**
     * Turns a serializable object into a byte array
     */
    public static byte[] serialize (java.io.Serializable object)
        throws java.io.IOException
    {
        java.io.ByteArrayOutputStream baos =
            new java.io.ByteArrayOutputStream();
        java.io.ObjectOutputStream oos =
            new java.io.ObjectOutputStream(baos);
        oos.writeObject(object);
        oos.flush();

        return baos.toByteArray();
    }

    /**
     * A method for copying files
     */
    public static void copy 
        (final java.io.File file, final String destinationPath)
    {
        java.io.FileInputStream in = null;
        java.io.FileOutputStream out = null;
        try
        {
            in = new java.io.FileInputStream(file);
            out = new java.io.FileOutputStream(destinationPath);

            byte[] buffer = new byte[1024];
            while (true)
            {
                int count = in.read(buffer);

                if (count < 1) break;

                out.write(buffer, 0, count);

                if (count < 1024) break;
            }

            //file.delete();
        }
        catch (final java.io.IOException ie)
        {
            try
            {
                in.close();
                out.close();
            }
            catch (final Throwable t)
            {
                // ignore
            }

            log.warn
                ("copy() "+
                 "Failed to copy "+file.getPath()+" to "+destinationPath, 
                 ie);
        }
    }

    /**
     * simply calls the other copy method
     */
    public static void copy 
        (final String fileName, final String destinationFileName)
    {
        copy(new java.io.File(fileName), destinationFileName);
    }

    /**
     * A method for moving files.
     */
    public static void move 
        (final java.io.File file, final String destinationPath)
    {
        copy(file, destinationPath);

        final boolean b = file.delete();

        if ( ! b) 
        {
            log.warn
                ("move() "+
                 "failed to delete file '"+file.getPath()+"' after move.");
        }
    }

    /**
     * A method for moving files.
     */
    public static void move
        (final String fileName, final String destinationFileName)
    {
        move(new java.io.File(fileName), destinationFileName);
    }

    public static String getMd5Digest (byte[] data)
    {
        java.security.MessageDigest digest = null;
        try
        {
            digest = java.security.MessageDigest.getInstance("MD5");
        }
        catch (java.security.NoSuchAlgorithmException nsae)
        {
            log.warn("cannot compute MD5 digest", nsae);
            return null;
        }

        digest.update(data);

        return ByteUtils.toString(digest.digest());
    }

    /**
     * On a unix-like platform, will return "file:",
     * on a windows one, will return "file:/".
     */
    public static String getFileProtocolPrefix ()
    {
        if (java.io.File.separatorChar == '\\') return "file:/";
        return "file:";
    }

    /**
     * Makes sure the file protocol prefix is removed.
     */
    public static String removeFileProtocolPrefix (final String url)
    {
        String result = url.substring(5);

        if (getFileProtocolPrefix().length() == 6)
            //
            // win
        {
            if (url.startsWith(getFileProtocolPrefix())) 
                result = url.substring(6);
        }

        if (log.isDebugEnabled())
            log.debug("removeFileProtocolPrefix() returning '"+result+"'");

        return result;
    }

    /**
     * Concats the app directory and the path and
     * return the resulting canonical (unique) path.
     */
    public static String getCanonicalPath
        (final String applicationDirectory, final String path)
    {
        java.io.File f = new java.io.File(path);

        if ( ! f.isAbsolute())
            f = new java.io.File(applicationDirectory + path);

        try
        {
            final String result = f.getCanonicalPath();

            if (log.isDebugEnabled())
                log.debug("getCanonicalPath() returning "+result);

            return result;
        }
        catch (final java.io.IOException ie)
        {
            log.warn("getCanonicalPath() exception", ie);
        }
        return f.getAbsolutePath();
    }

    /**
     * Returns true if the given string is an URL.
     */
    public static boolean isUrl (final String s)
    {
        try
        {
            new java.net.URL(s);
        }
        catch (final Throwable t)
        {
            return false;
        }
        return true;
    }

    /**
     * Turns a relative file url into an absolute one (and makes sure that
     * the "file:" prefix is set).
     * If the url is for another protocol, the output will be the same as the
     * input.
     */
    public static String expandUrl (final String url)
    {
        return expandUrl(null, url);
    }

    /**
     * Makes sure the result is an URL, and if its a path URL, will make
     * sure that it's an absolute and canonical path.
     */
    public static String expandUrl 
        (final String applicationDirectory, final String url)
    {
        if (log.isDebugEnabled())
            log.debug("expandUrl() considering >"+url+"<");

        // openwfe exceptions...
        if (url.startsWith("field:")) return url;
        if (url.startsWith("resource:")) return url;

        if (url.startsWith("file:"))
        {
            final String fn = expandFileName
                (applicationDirectory, removeFileProtocolPrefix(url));

            final String result =
                getFileProtocolPrefix() + fn;

            if (log.isDebugEnabled())
                log.debug("expandUrl() 0 returning >"+result+"<");

            return result;
        }

        if ( ! isUrl(url))
            return expandUrl(applicationDirectory, getFileProtocolPrefix()+url);

        if (log.isDebugEnabled())
            log.debug("expandUrl() 1 returning >"+url+"<");

        return url;

        //
        // java.io.File.toUrl() ??
        //
    }

    /**
     * Canonicalizes a filename.
     */
    public static String expandFileName
        (final String applicationDirectory, final String filename)
    {
        //log.debug("expandFileName() considering >"+filename+"<");
        //log.debug("expandFileName() adir is '"+applicationDirectory+"'");

        String result = null;
        String adir = "";

        if (applicationDirectory != null) 
            adir = applicationDirectory.trim() + java.io.File.separator;

        //log.debug("expandFileName() adir is '"+adir+"'");

        if (java.io.File.separatorChar == '\\')
            //
            // windows
        {
            //if (adir.equals(".\\") || adir.equals(".\\\\")) adir = "";

            if (filename.charAt(1) == ':')
                //
                // absolute
            {
                result = filename;
            }
            else
                //
                // relative
            {
                result = adir + filename;
            }
        }
        else
            //
            // *nix
        {
            //if (adir.equals("./") || adir.equals(".//")) adir = "";

            if (filename.charAt(0) == java.io.File.separatorChar)
                //
                // absolute
            {
                result = filename;
            }
            else
                //
                // relative
            {
                result = adir + filename;
            }
        }

        try
        {
            if (result.endsWith("*"))
            {
                result = 
                    (new java.io.File
                        (result.substring(0, result.length()-1)))
                            .getCanonicalPath() + 
                         java.io.File.separatorChar + 
                         "*";
            }
            else
            {
                result = 
                    (new java.io.File(result)).getCanonicalPath();
            }
        }
        catch (final java.io.IOException ie)
        {
            log.warn("expandFileName() failed to get canonical path", ie);
        }

        if (log.isDebugEnabled())
            log.debug("expandFileName() returning >"+result+"<");

        return result;
    }

    /**
     * Takes as input a String containing an url and makes sure
     * to output the same url with a protocol if missing, the protocol
     * added is "file:".
     */
    public static String ensureProtocol (String url)
    {
        url = url.trim();

        final int colon = url.indexOf(":");
        if (colon < 0 ||
            colon == 1) // probably something like "D:\nada\"
        {
            return getFileProtocolPrefix() + url;
        }

        return url;
    }

    /**
     * builds a list of all the files (their path) that are under
     * the given rootPath
     */
    public static java.util.Set buildFileSet (String rootPath)
    {
        java.util.Set result = new java.util.HashSet();

        buildFileSet(result, new java.io.File(rootPath));

        return result;
    }

    private static void buildFileSet
        (java.util.Set result, 
         java.io.File rootPath)
    {
        java.io.File[] files = rootPath.listFiles();

        for (int i=0; i<files.length; i++)
        {
            java.io.File f = files[i];

            if (f.isDirectory())
            {
                buildFileSet(result, f);
                continue;
            }

            result.add(f.getPath());
        }
    }

    /**
     * Given a List l, locates the elt thisObject in it and replaces it with
     * thatObject.
     */
    public static boolean replace 
        (final java.util.List l, 
         final Object thisObject, 
         final Object thatObject)
    {
        synchronized (l)
        {
            int index = -1;

            for (int i=0; i<l.size(); i++)
            {
                final Object o = l.get(i);

                if (o.equals(thisObject))
                {
                    index = i;
                    break;
                }
            }

            if (index > -1)
            {
                l.remove(index);
                l.add(index, thatObject);
                return true;
            }

            return false;
        }
    }

    /**
     * Turns a real path into a URL, given the realRootPath.<br>
     * <br>
     * changeToWebPath("/home/jmettraux/", "http://localhost", "/home/jmettraux/nada/file.htm")
     * <br>
     * will return "http://localhost/nada/file.htm".<br>
     * <br>
     * Should also work on windows.
     */
    public static String changeToWebPath
        (String realRootPath, String webRootPath, String pathToChange)
    {
        pathToChange = pathToChange.substring(realRootPath.length());
        pathToChange = pathToChange.replace('\\', '/');

        if ( ! webRootPath.endsWith("/") &&
             ! pathToChange.startsWith("/"))
        {
            webRootPath += "/";
        }

        return 
            webRootPath + pathToChange;
    }

    /**
     * given a map of string &lt;-&gt; integers, adds one to the integer
     * corresponding to the given key
     */
    public static void inc (final java.util.Map map, final String key)
    {
        inc(map, key, 1);
    }

    /**
     * given a map of string &lt;-&gt; integers, adds one to the integer
     * corresponding to the given key
     */
    public static void inc 
        (final java.util.Map map, final String key, final int count)
    {
        Integer ii = (Integer)map.get(key);
        if (ii == null) ii = new Integer(0);
        ii = new Integer(ii.intValue() + count);
        map.put(key, ii);
    }

    /**
     * precondition : the 2 values are not null
     */
    public static int compareValues 
        (final Object value, final Object otherValue)
    {
        //
        // try numeric comparison first
        
        try
        {
            double dVal = Double.parseDouble(value.toString());
            double dOtherVal = Double.parseDouble(otherValue.toString());

            return (int)(dVal - dOtherVal);
        }
        catch (NumberFormatException nfe)
        {
            // forget it
            //log.debug("non numeric values, proceeding with string comparison");
        }

        //
        // string comparison else
        
        return (value.toString().compareTo(otherValue.toString()));
    }

    /**
     * Returns true if the o.toString() equals "true" or "false"
     * (case is not relevant).
     * If the object is null, will return false.
     */
    public static boolean toBoolean (Object o)
    {
        if (o == null) return false;

        String s = o.toString().trim();

        return 
            ("yes".equalsIgnoreCase(s) || 
             "ok".equalsIgnoreCase(s) ||
             "true".equalsIgnoreCase(s));
    }

    /**
     * Returns the encoding for the system
     * by default, it's ISO-8859-1.
     * Can be set by changing the ENCODING system property.
     */
    public static String getEncoding ()
    {
        return System.getProperty("openwfe.xml.encoding", "ISO-8859-1");
    }

    /**
     * As getEncoding() returns the String value of the encoding to user,
     * this method returns the corresponding charset.
     */
    public static java.nio.charset.Charset getCharset ()
    {
        return java.nio.charset.Charset.forName(getEncoding());
    }

    /**
     * Returns a copy of a String (or null if the in string is null)
     */
    public static String copyString (String in)
    {
        if (in == null) return null;
        return new String(in);
    }

    /**
     * returns true if the two strings are equal,
     * returns false as soon as one of them is null, or
     * if they are not equals
     */
    public static boolean stringEquals (String s1, String s2)
    {
        if (s1 == null && s2 == null) return true;

        if (s1 == null || s2 == null) return false;

        return s1.equals(s2);
    }

    /**
     * This method is useful for printing stack traces from a JSP : it accepts 
     * the 'out' writer has parameter.
     */
    public static void printStackTrace 
        (final Throwable t, final java.io.Writer w)
    {
        java.io.PrintWriter pw = new java.io.PrintWriter(w);
        t.printStackTrace(pw);
        pw.flush();
    }

    /**
     * Dumps a stack trace in a String.
     */
    public static String stackTraceToString (final Throwable t)
    {
        final java.io.StringWriter sw = new java.io.StringWriter();
        final java.io.PrintWriter pw = new java.io.PrintWriter(sw);

        t.printStackTrace(pw);

        pw.flush();

        return sw.toString();
    }

    /**
     * Outputs at DEBUG level the content of a map
     */
    public static void debugMap 
        (final org.apache.log4j.Logger log,
         final java.util.Map map)
    {
        if ( ! log.isDebugEnabled()) return;

        final StringBuffer sb = new StringBuffer();

        sb.append("\n*** debugMap ***\n");

        final java.util.Iterator it = map.keySet().iterator();
        while (it.hasNext())
        {
            final Object key = it.next();
            final Object value = map.get(key);

            sb.append("   * ");
            sb.append(key);
            sb.append("   -->  ");
            sb.append(value);
            sb.append("\n");
        }

        log.debug(sb.toString());
    }

    /**
     * Simply returning the platform dependent temp directory.
     */
    public static String getTempDir ()
    {
        return System.getProperty("java.io.tmpdir");
    }

    /**
     * A method for dumping data in a temp file. Returns the filename.
     */
    public static String dump (final String prefix, final byte[] data)
    {
        try
        {
            final java.io.File tempFile = 
                java.io.File.createTempFile(prefix, ".dmp");

            if (log.isDebugEnabled())
                log.debug("dump() to "+tempFile.getPath());

            final java.io.FileOutputStream fos = 
                new java.io.FileOutputStream(tempFile);

            fos.write(data);
            fos.flush();
            fos.close();

            return tempFile.getPath();
        }
        catch (final Throwable t)
        {
            if (log.isDebugEnabled())
                log.debug("dump() failure "+t);
        }

        return null;
    }

    /**
     * Ensures (return a sure copy) a string for use in a fileName.
     * I.e.  'toto nada' --&gt; 'toto_nada'
     */
    public static String ensureForFileName (final String in)
    {
        return in.replaceAll(" ", "_");
    }

    /**
     * Logs the current stack trace.
     */
    public static void logStackTrace 
        (final org.apache.log4j.Logger log,
         final String message)
    {
        try
        {
            throw new Exception("openwfe.org.Utils.logStackTrace()");
        }
        catch (final Exception e)
        {
            log.info(message, e);
        }
    }

    /**
     * Logs the current stack trace.
     */
    public static void logStackTrace 
        (final org.apache.log4j.Logger log)
    {
        logStackTrace(log, "logStackTrace()");
    }

    /**
     * Copies a map (returns a brand new HashMap).
     */
    public static java.util.Map copyHashMap (final java.util.Map m)
    {
        if (m == null) return null;

        final java.util.Map copy = new java.util.HashMap(m.size());

        /*
        final java.util.Iterator it = m.entrySet().iterator();
        while (it.hasNext())
        {
            final java.util.Map.Entry e = (java.util.Map.Entry)it.next();

            copy.put(e.getKey(), e.getValue());
        }
        */

        copy.putAll(m);

        return copy;
    }

    /**
     * Returns the first element in the list that is of the given class.
     */
    public static Object getFirstElementOfClass 
        (final java.util.List l, final Class clazz)
    {
        final java.util.Iterator it = l.iterator();
        while (it.hasNext())
        {
            final Object o = it.next();

            if (clazz.isAssignableFrom(o.getClass())) return o;
        }

        return null;
    }

    /**
     * Returns the last element in the list that is of the given class.
     */
    public static Object getLastElementOfClass
        (final java.util.List l, final Class clazz)
    {
        for (int i = l.size(); i > 0; i--)
        {
            final Object o = l.get(i-1);

            if (clazz.isAssignableFrom(o.getClass())) return o;
        }

        return null;
    }

    /**
     * Returns a long indicating when the resource given by its url 
     * was last modified.
     */
    public static long getLastModified (final String url)
    {
        try
        {
            return getLastModified(new java.net.URL(url));
        }
        catch (final Throwable t)
        {
            log.info("getLastModified() failure", t);
        }

        return -1;
    }

    /**
     * Returns a long indicating when the resource given by its url 
     * was last modified.
     */
    public static long getLastModified (final java.net.URL u)
    {
        try
        {
            return u.openConnection().getLastModified();
        }
        catch (final Throwable t)
        {
            log.info("getLastModified() failure", t);
        }

        return -1;
    }

    /**
     * Turns a list to an array of String.
     */
    public static String[] toStringArray (final java.util.List l)
    {
        if (l == null) return null;

        final String[] ss = new String[l.size()];

        for (int i=0; i<l.size(); i++)
            ss[i] = l.get(i).toString();

        return ss;
    }

}
