/*
 * Copyright (c) 2001-2006, John Mettraux, OpenWFE.org
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * . Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.  
 * 
 * . Redistributions in binary form must reproduce the above copyright notice, 
 *   this list of conditions and the following disclaimer in the documentation 
 *   and/or other materials provided with the distribution.
 * 
 * . Neither the name of the "OpenWFE" nor the names of its contributors may be
 *   used to endorse or promote products derived from this software without
 *   specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: ReflectionUtils.java 2673 2006-05-26 21:08:46Z jmettraux $
 */

//
// ReflectionUtils.java
//
// john.mettraux@openwfe.org
//
// generated with 
// jtmpl 1.1.01 2004/05/19 (john.mettraux@openwfe.org)
//

package openwfe.org;

import java.lang.reflect.Array;
import java.lang.reflect.Field;
import java.lang.reflect.Method;


/**
 * Methods for instatiating classes on the fly...
 *
 * <p><font size=2>CVS Info :
 * <br>$Author: jmettraux $
 * <br>$Id: ReflectionUtils.java 2673 2006-05-26 21:08:46Z jmettraux $ </font>
 *
 * @author john.mettraux@openwfe.org
 */
public abstract class ReflectionUtils
{

    private final static org.apache.log4j.Logger log = org.apache.log4j.Logger
        .getLogger(ReflectionUtils.class.getName());

    /**
     * contains simply 'class', this is supposed to be the name of the parameter
     * holding the classname for the initObject(java.util.Map params) method.
     */
    public final static String P_CLASS
        = "class";

    private final static String IS = "is";
    private final static String GET = "get";
    private final static String SET = "set";

    //
    // caching the pair of get/set methods for each bean class

    private static java.util.Map fieldCache = new java.util.HashMap();

    //
    // METHODS

    /**
     * finds the init(java.util.Map initParameters) method in an object
     * and invokes it
     */
    public static Object initObject 
        (Object o, java.util.Map initParameters)
    throws 
        NoSuchMethodException, 
        IllegalAccessException, 
        java.lang.reflect.InvocationTargetException
    {
        //
        // not a permission

        Class clazz = o.getClass();

        java.lang.reflect.Method initMethod = 
            clazz.getMethod("init", new Class[] { java.util.Map.class });

        //log.debug("------ found the init() method");

        initMethod.invoke(o, new Object[] { initParameters });

        //log.debug("------ init() method invocation seems successful");

        return o;
    }

    /**
     * This method is intended for classes with constructor that takes no
     * parameter and a init(java.util.Map initParameters) method.
     * By using this method, you can build an instance of such a class very
     * easily.
     */
    public static Object initObjectOfClass
        (final String className, final java.util.Map initParameters)
    throws 
        Exception
    {
        Class clazz = Class.forName(className);
        Object instance = clazz.newInstance();

        return initObject(instance, initParameters);
    }

    /**
     * init an object whose class is contained in a P_CLASS named parameter 
     * (usually 'class')
     */
    public static Object initObject 
        (final java.util.Map initParameters)
    throws 
        Exception 
    {
        final String className = (String)initParameters.get(P_CLASS);

        if (className == null)
        {
            throw new IllegalArgumentException
                ("There is no parameter '"+P_CLASS+
                 "' ("+ReflectionUtils.class.getName()+
                 ".P_CLASS). Cannot init instance.");
        }

        return initObjectOfClass(className, initParameters);
    }


    /**
     * A reflective method.
     * 
     * Customer c = new Customer("john");
     * Object o = Utils.getAttribute(c, "name");
     * ...
     * o will yield the value "john" (because "john".equals(c.getName()))
     * ...
     */
    public static Object getAttribute 
        (Object o, String attributeName)
    throws 
        NoSuchMethodException, 
        IllegalAccessException, 
        java.lang.reflect.InvocationTargetException
    {
        Class clazz = o.getClass();

        String methodName = 
            "get" +
            attributeName.substring(0, 1).toUpperCase() +
            attributeName.substring(1);

        java.lang.reflect.Method getMethod = 
            clazz.getMethod(methodName, new Class[] {});

        return getMethod.invoke(o, new Object[] {});
    }

    /**
     * Given an instance, the name of one of its attribute (property) sets a 
     * new value for it.
     */
    public static Object setAttribute 
        (Object o, String attributeName, Object value)
    throws 
        Exception
    {
        Class clazz = o.getClass();

        String methodName = 
            "set" +
            attributeName.substring(0, 1).toUpperCase() +
            attributeName.substring(1);

        java.lang.reflect.Method method = null;
        Class parameterClass = null;

        java.lang.reflect.Method[] methods = clazz.getMethods();
        for (int i=0; i<methods.length; i++)
        {
            java.lang.reflect.Method m = methods[i];

            if (m.getName().equals(methodName) &&
                m.getParameterTypes().length == 1)
            {
                method = m;
                parameterClass = m.getParameterTypes()[0];
                break;
            }
        }

        if (method == null)
        {
            throw new NoSuchMethodException
                ("No method named '"+methodName+
                 "' with 1! parameter in class '"+clazz.getName()+"'");
        }

        // narrow value type
        value = buildInstance(parameterClass, value);

        return method.invoke(o, new Object[] { value });
    }

    /**
     * For a given bean attribute name, returns its class.
     */
    public static Class getAttributeClass 
        (final Class beanClass, final String attributeName)
    throws
        Exception
    {
        final String methodName = 
            "get"+
            attributeName.substring(0, 1).toUpperCase()+
            attributeName.substring(1);

        final Method m = beanClass.getMethod(methodName, new Class[] { });

        //log.debug
        //    ("getAttributeClass() "+
        //     "return type is "+m.getReturnType().getName());

        return m.getReturnType();
    }

    /**
     * Given an array class, returns its atomic class
     */
    public static Class getItemClass (final Class arrayClass)
        throws ClassNotFoundException
    {
        final String arrayClassName = arrayClass.getName();

        if (arrayClassName.length() > 2)
        {
            final String itemClassName = 
                arrayClassName.substring(2, arrayClassName.length()-1);

            return Class.forName(itemClassName);
        }

        final char c = arrayClassName.charAt(1);

        if (c == 'I') return Integer.class;
        if (c == 'J') return Long.class;
        if (c == 'F') return Float.class;
        if (c == 'D') return Double.class;
        if (c == 'B') return Byte.class;
        if (c == 'C') return Character.class;
        if (c == 'S') return Short.class;
        if (c == 'Z') return Boolean.class;

        return null;
    }

    /**
     * Returns true if the given class represents an array of primitives
     * like int[] or boolean[].
     */
    public static boolean isPrimitiveArray (final Class clazz)
    {
        if ( ! clazz.isArray()) return false;

        final String s = clazz.getName();

        if (s.length() != 2) return false;

        if (s.charAt(0) != '[') return false;

        final char c = s.charAt(1);

        return 
            (c == 'I' ||
             c == 'J' ||
             c == 'F' ||
             c == 'D' ||
             c == 'B' ||
             c == 'C' ||
             c == 'S' ||
             c == 'Z');
    }

    /**
     * Returns a list of attribute (field) names for a given object.
     */
    public static java.util.List listAttributes (final Object o)
    {
        final Class clazz = o.getClass();

        final java.lang.reflect.Method[] methods = clazz.getMethods();

        final java.util.List attributeNames = 
            new java.util.ArrayList(methods.length);

        for (int i=0; i<methods.length; i++)
        {
            final java.lang.reflect.Method m = methods[i];

            if (m.getName().startsWith("get") &&
                m.getParameterTypes().length == 0)
            {
                final String attributeName = 
                    m.getName().substring(3, 4).toLowerCase() + 
                    m.getName().substring(4);

                attributeNames.add(attributeName);
            }
        }

        return attributeNames;
    }


    /* *
     * Returns true if the given class is in fact a type (like 'int' or
     * 'double').
     * /
    public static boolean isPrimitiveType (final Class c)
    {
        final String s = c.getName();

        return 
            s.equals("int") ||
            s.equals("char") ||
            s.equals("byte") ||
            s.equals("long") ||
            s.equals("float") ||
            s.equals("double") ||
            s.equals("boolean");
    }
     */

    /**
     * Builds a primitive type (instance).
     */
    public static Object buildPrimitive 
        (final Class primitiveType, final Object value)
    {
        final String s = primitiveType.getName();

        //log.debug("buildPrimitive() type name is "+s);

        final String v = value.toString().trim();

        if (s.equals("char"))
        {
            if (v.length() < 1) return null;
            return new Character(v.charAt(0));
        }

        if (s.equals("int"))
        {
            return new Integer(v);
        }
        if (s.equals("short"))
        {
            return new Short(v);
        }
        if (s.equals("byte"))
        {
            return new Byte(v);
        }
        if (s.equals("long"))
        {
            return new Long(v);
        }
        if (s.equals("float"))
        {
            return new Float(v);
        }
        if (s.equals("double"))
        {
            return new Double(v);
        }

        if (s.equals("boolean"))
        {
            return new Boolean(v.trim().toLowerCase().equals("true"));
        }

        return null;
    }

    /**
     * Narrows an instance array into a primitive array, for example
     * an Integer[] into an int[].
     */
    public static Object buildPrimitiveArray
        (final Class primitiveArrayType, final Object value)
    throws
        ClassNotFoundException
    {
        final char c = primitiveArrayType.getName().charAt(1);

        Object result = null;

        if (c =='I') 
            result = new int[Array.getLength(value)];
        else if (c =='J') 
            result = new long[Array.getLength(value)];
        else if (c =='F') 
            result = new float[Array.getLength(value)];
        else if (c =='D') 
            result = new double[Array.getLength(value)];
        else if (c =='B') 
            result = new byte[Array.getLength(value)];
        else if (c =='C') 
            result = new char[Array.getLength(value)];
        else if (c =='S') 
            result = new short[Array.getLength(value)];
        else if (c =='Z') 
            result = new boolean[Array.getLength(value)];

        if (result == null)
        {
            throw new ClassNotFoundException
                ("Cannot build array of class "+primitiveArrayType.getName());
        }

        for (int i=0; i<Array.getLength(value); i++)
            Array.set(result, i, Array.get(value, i));

        return result;
    }

    /**
     * tries to locate a constructor with a single parameter
     * in the targetClass and to build a target object then
     */
    public static Object buildInstance
        (Class targetClass, Object value)
    throws 
        Exception
    {
        //log.debug
        //    ("buildInstance() - targetClass is '"+targetClass.getName()+"'");
        //log.debug
        //    ("buildInstance() - value.class is '"+value.getClass().getName()+"'");
        //log.debug
        //    ("buildInstance() - value.toString() is '"+value+"'");

        if (value == null) 
        {
            log.debug("buildInstance() - returning null");

            return null;
        }

        //if (isPrimitiveType(targetClass))
        if (targetClass.isPrimitive())
            return buildPrimitive(targetClass, value);

        if (isPrimitiveArray(targetClass))
            return buildPrimitiveArray(targetClass, value);

        if (targetClass.isAssignableFrom(value.getClass()))
            return value; // fast lane

        java.lang.reflect.Constructor[] constructors = 
            targetClass.getConstructors();

        java.lang.reflect.Constructor constructor = null;
        java.lang.reflect.Constructor secondMatchConstructor = null;

        Class valueClass = value.getClass();
        Long longValue = null;

        if (targetClass.equals(java.sql.Date.class) ||
            targetClass.equals(java.util.Date.class))
        {
            valueClass = Long.class;
            longValue = new Long(((java.util.Date)value).getTime());

            /*
            log.debug
                ("buildInstance() - valueClass is set to 'Long'");
            log.debug
                ("buildInstance() - longValue is set to '"+
                 longValue.longValue()+"'");
            */
        }

        for (int i=0; i<constructors.length; i++)
        {
            java.lang.reflect.Constructor c = constructors[i];

            if (c.getParameterTypes().length != 1) continue;

            //if (valueClass.isAssignableFrom(c.getParameterTypes()[0]))
            if (c.getParameterTypes()[0].isAssignableFrom(valueClass))
            {
                constructor = c;
                break;
            }

            if (c.getParameterTypes()[0].equals(String.class))
            {
                secondMatchConstructor = c;
            }
        }

        if (constructor == null &&
            secondMatchConstructor == null)
        {
            throw new NoSuchMethodException
                ("Couldn't find any constructor in class '"+
                 targetClass.getName()+
                 "' that could accept 'java.lang.String' or '"+
                 valueClass.getName()+"' as parameter type");
        }

        if (constructor == null)
        {
            constructor = secondMatchConstructor;
            value = value.toString();
        }

        if (longValue != null)
        {
            value = longValue;
        }

        return 
            constructor.newInstance(new Object[] { value });
    }

    /** 
     * Returns a new instance of the given object.
     * The object must have a constructor with no parameters, else
     * this method will throw an IllegalArgumentException
     */
    public static Object newInstance (final Object o)
    {
        try
        {
            return o.getClass().newInstance();
        }
        catch (final Throwable t)
        {
            throw new IllegalArgumentException
                ("Failed to build new instance of class "+o.getClass()+
                 "because\n"+t);
        }
    }

    /**
     * Builds an instance for a given class name.
     */
    public static Object buildNewInstance (final String className)
    {
        try
        {
            final Class clazz = Class.forName(className);
            return clazz.newInstance();
        }
        catch (final Throwable t)
        {
            throw new IllegalArgumentException
                ("Failed to build new instance of class '"+className+
                 "' because of "+t);
        }
    }

    /**
     * Invokes the given method on the target object and return the
     * result if any.
     */
    public static Object invokeStatic
        (final Class target, 
         final String methodName, 
         final Class[] argClasses,
         final Object[] args)
    throws 
        Exception
    {
        final java.lang.reflect.Method method = 
            target.getMethod(methodName, argClasses);

        return method.invoke(null, args);
    }

    /**
     * Invokes the given method on the target object and return the
     * result if any.
     */
    public static Object invoke 
        (final Object target, 
         final String methodName, 
         final Class[] argClasses,
         final Object[] args)
    throws 
        Exception
    {
        final java.lang.reflect.Method method = 
            target.getClass().getMethod(methodName, argClasses);

        return method.invoke(target, args);
    }

    /**
     * Invoke the given method in its own thread : don't wait on its output.
     * Problems with running the invocation only appear in the logs.
     * This method is especially used by the ExpressionPool implementations to
     * decrease reply time.
     */
    public static void invokeInOwnThread
        (final Object target, 
         final String methodName, 
         final Class[] argClasses,
         final Object[] args,
         final String potentialErrorMessage)
    {
        (new Thread ()
         {
             public void run ()
             {
                 try
                 {
                     invoke(target, methodName, argClasses, args);
                 }
                 catch (Throwable t)
                 {
                     log.warn(potentialErrorMessage, t);
                 }
             }
         }).start();
    }

    /**
     * Invoke the given method in its own thread : don't wait on its output.
     * This variant uses a default potential error message.
     * Problems with running the invocation only appear in the logs.
     * This method is especially used by the ExpressionPool implementations to
     * decrease reply time.
     */
    public static void invokeInOwnThread
        (final Object target, 
         final String methodName, 
         final Class[] argClasses,
         final Object[] args)
    {
        final String potentialErrorMessage = 
            "invoking the method '"+methodName+
            "' on object of class '"+target.getClass().getName()+
            "' failed";

        invokeInOwnThread
            (target, methodName, argClasses, args, potentialErrorMessage);
    }

    /**
     * Returns true if the given class has a constructor with no parameters.
     * (a potential bean class).
     */
    public static boolean hasNoParamConstructor (final Class c)
    {
        try
        {
            c.getConstructor(new Class[] {});
        }
        catch (final NoSuchMethodException e)
        {
            return false;
        }

        return true;
    }

    /**
     * Returns a list of Method[] arrays that contain the get/set pair for
     * each read/write field of the given instance.
     * This method is heavily used by openwfe.org.xml.XmlCoder.
     */
    public static java.util.List listReadWriteFields 
        (final Object instance)
    {
        Class clazz = instance.getClass();
        if (instance instanceof Class) clazz = (Class)instance;

        final java.util.List cachedResult = 
            (java.util.List)fieldCache.get(clazz);
        if (cachedResult != null) 
            return cachedResult;

        final java.util.List result = new java.util.ArrayList(10);

        final Method[] ms = clazz.getMethods();

        for (int i=0; i<ms.length; i++)
        {
            final Method getMethod = ms[i];

            final Class returnType = getMethod.getReturnType();
            final String getMethodName = getMethod.getName();

            //log.debug("listReadWriteFields() considering >"+getMethodName+"<");

            if (returnType == null) continue;
            if (getMethod.getParameterTypes().length > 1) continue;

            //log.debug("listReadWriteFields() return and params ok");

            if (( ! getMethodName.startsWith(GET)) &&
                ( ! getMethodName.startsWith(IS)))
            {
                continue;
            }

            //log.debug("listReadWriteFields() prefix ok");

            String rawFieldName = getMethodName.substring(3);
            if (getMethodName.startsWith(IS)) 
                rawFieldName = getMethodName.substring(2);

            Method setMethod = null;
            try
            {
                setMethod = clazz.getMethod
                    (SET + rawFieldName, new Class[] { returnType });

                final String sReturnType = ""+setMethod.getReturnType();

                if (( ! sReturnType.equals("void")) &&
                    ( ! sReturnType.equals("null")))
                {
                    continue;
                }
            }
            catch (final NoSuchMethodException nsme)
            {
                continue;
            }

            //log.debug("listReadWriteFields() adding >"+rawFieldName+"<");

            result.add(new Method[] { getMethod, setMethod });
        }

        //
        // cache result
        
        fieldCache.put(instance.getClass(), result);

        //
        // that's all folks

        return result;
    }

    /**
     * Given a path like "openwfe.org.engine.Definitions.OPENWFE_VERSION"
     * returns the value of the constant (as a String).
     */
    public static String lookupConstantValue (final String constantPath)
    {
        try
        {
            final int i = constantPath.lastIndexOf(".");
            final String className = constantPath.substring(0, i);
            final String constantName = constantPath.substring(i+1);

            //log.debug("extractConstant() className     >"+className+"<");
            //log.debug("extractConstant() constantName  >"+constantName+"<");

            final Class clazz = Class.forName(className);
            final Field fConstant = clazz.getField(constantName);

            return ""+fConstant.get(null);
        }
        catch (final Throwable t)
        {
            log.warn("extractConstant() failed to determine value", t);
            return null;
        }
    }

}
