/*
 * Copyright (c) 2001-2006, John Mettraux, OpenWFE.org
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * . Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.  
 * 
 * . Redistributions in binary form must reproduce the above copyright notice, 
 *   this list of conditions and the following disclaimer in the documentation 
 *   and/or other materials provided with the distribution.
 * 
 * . Neither the name of the "OpenWFE" nor the names of its contributors may be
 *   used to endorse or promote products derived from this software without
 *   specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: FileUtils.java 2977 2006-07-31 16:00:45Z jmettraux $
 */

//
// FileUtils.java
//
// jmettraux@openwfe.org
//
// generated with 
// jtmpl 1.0.04 20.11.2001 John Mettraux (jmettraux@openwfe.org)
//

package openwfe.org;

import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;

import openwfe.org.misc.ByteUtils;


/**
 * Static methods for files, paths and URLs manipulation.
 *
 * <p><font size=2>CVS Info :
 * <br>$Author: jmettraux $
 * <br>$Date: 2006-07-31 18:00:45 +0200 (Mon, 31 Jul 2006) $
 * <br>$Id: FileUtils.java 2977 2006-07-31 16:00:45Z jmettraux $ </font>
 *
 * @author john.mettraux@openwfe.org
 */
public abstract class FileUtils
{

    private final static org.apache.log4j.Logger log = org.apache.log4j.Logger
        .getLogger(FileUtils.class.getName());


    /**
     * A method for copying files
     */
    public static void copy 
        (final java.io.File file, final String destinationPath)
    {
        java.io.FileInputStream in = null;
        java.io.FileOutputStream out = null;
        try
        {
            in = new java.io.FileInputStream(file);
            out = new java.io.FileOutputStream(destinationPath);

            ByteUtils.copy(in, out, 1024);

            //file.delete();
        }
        catch (final java.io.IOException ie)
        {
            try
            {
                in.close();
                out.close();
            }
            catch (final Throwable t)
            {
                // ignore
            }

            log.warn
                ("copy() "+
                 "Failed to copy "+file.getPath()+" to "+destinationPath, 
                 ie);
        }
    }

    /**
     * simply calls the other copy method
     */
    public static void copy 
        (final String fileName, final String destinationFileName)
    {
        copy(new java.io.File(fileName), destinationFileName);
    }

    /**
     * A method for moving files.
     */
    public static void move 
        (final java.io.File file, final String destinationPath)
    {
        copy(file, destinationPath);

        final boolean b = file.delete();

        if ( ! b) 
        {
            log.warn
                ("move() "+
                 "failed to delete file '"+file.getPath()+"' after move.");
        }
    }

    /**
     * A method for moving files.
     */
    public static void move
        (final String fileName, final String destinationFileName)
    {
        move(new java.io.File(fileName), destinationFileName);
    }

    /**
     * On a unix-like platform, will return "file:",
     * on a windows one, will return "file:/".
     */
    public static String getFileProtocolPrefix ()
    {
        if (java.io.File.separatorChar == '\\') return "file:/";
        return "file:";
    }

    /**
     * Makes sure the file protocol prefix is removed.
     */
    public static String removeFileProtocolPrefix (final String url)
    {
        String result = url.substring(5);

        if (getFileProtocolPrefix().length() == 6)
            //
            // win
        {
            if (url.startsWith(getFileProtocolPrefix())) 
                result = url.substring(6);
        }

        if (log.isDebugEnabled())
            log.debug("removeFileProtocolPrefix() returning '"+result+"'");

        return result;
    }

    /**
     * Concats the app directory and the path and
     * return the resulting canonical (unique) path.
     */
    public static String getCanonicalPath
        (final String applicationDirectory, final String path)
    {
        java.io.File f = new java.io.File(path);

        if ( ! f.isAbsolute())
            f = new java.io.File(applicationDirectory + path);

        try
        {
            final String result = f.getCanonicalPath();

            if (log.isDebugEnabled())
                log.debug("getCanonicalPath() returning "+result);

            return result;
        }
        catch (final java.io.IOException ie)
        {
            log.warn("getCanonicalPath() exception", ie);
        }
        return f.getAbsolutePath();
    }

    /**
     * Canonicalizes a filename.
     */
    public static String expandFileName
        (final String applicationDirectory, final String filename)
    {
        //log.debug("expandFileName() considering >"+filename+"<");
        //log.debug("expandFileName() adir is '"+applicationDirectory+"'");

        String result = null;
        String adir = "";

        if (applicationDirectory != null) 
            adir = applicationDirectory.trim() + java.io.File.separator;

        //log.debug("expandFileName() adir is '"+adir+"'");

        if (java.io.File.separatorChar == '\\')
            //
            // windows
        {
            //if (adir.equals(".\\") || adir.equals(".\\\\")) adir = "";

            if (filename.charAt(1) == ':')
                //
                // absolute
            {
                result = filename;
            }
            else
                //
                // relative
            {
                result = adir + filename;
            }
        }
        else
            //
            // *nix
        {
            //if (adir.equals("./") || adir.equals(".//")) adir = "";

            if (filename.charAt(0) == java.io.File.separatorChar)
                //
                // absolute
            {
                result = filename;
            }
            else
                //
                // relative
            {
                result = adir + filename;
            }
        }

        try
        {
            if (result.endsWith("*"))
            {
                result = 
                    (new java.io.File
                        (result.substring(0, result.length()-1)))
                            .getCanonicalPath() + 
                         java.io.File.separatorChar + 
                         "*";
            }
            else
            {
                result = 
                    (new java.io.File(result)).getCanonicalPath();
            }
        }
        catch (final java.io.IOException ie)
        {
            log.warn("expandFileName() failed to get canonical path", ie);
        }

        if (log.isDebugEnabled())
            log.debug("expandFileName() returning >"+result+"<");

        return result;
    }

    /**
     * builds a list of all the files (their path) that are under
     * the given rootPath
     */
    public static java.util.Set buildFileSet (String rootPath)
    {
        java.util.Set result = new java.util.HashSet();

        buildFileSet(result, new java.io.File(rootPath));

        return result;
    }

    private static void buildFileSet
        (java.util.Set result, 
         java.io.File rootPath)
    {
        java.io.File[] files = rootPath.listFiles();

        for (int i=0; i<files.length; i++)
        {
            java.io.File f = files[i];

            if (f.isDirectory())
            {
                buildFileSet(result, f);
                continue;
            }

            result.add(f.getPath());
        }
    }

    /**
     * Turns a real path into a URL, given the realRootPath.<br>
     * <br>
     * changeToWebPath("/home/jmettraux/", "http://localhost", "/home/jmettraux/nada/file.htm")
     * <br>
     * will return "http://localhost/nada/file.htm".<br>
     * <br>
     * Should also work on windows.
     */
    public static String changeToWebPath
        (String realRootPath, String webRootPath, String pathToChange)
    {
        pathToChange = pathToChange.substring(realRootPath.length());
        pathToChange = pathToChange.replace('\\', '/');

        if ( ! webRootPath.endsWith("/") &&
             ! pathToChange.startsWith("/"))
        {
            webRootPath += "/";
        }

        return 
            webRootPath + pathToChange;
    }

    /**
     * Simply returning the platform dependent temp directory.
     */
    public static String getTempDir ()
    {
        return System.getProperty("java.io.tmpdir");
    }

    /**
     * A method for dumping data in a temp file. Returns the filename.
     */
    public static String dump (final String prefix, final byte[] data)
    {
        try
        {
            final java.io.File tempFile = 
                java.io.File.createTempFile(prefix, ".dmp");

            if (log.isDebugEnabled())
                log.debug("dump() to "+tempFile.getPath());

            final java.io.FileOutputStream fos = 
                new java.io.FileOutputStream(tempFile);

            fos.write(data);
            fos.flush();
            fos.close();

            return tempFile.getPath();
        }
        catch (final Throwable t)
        {
            if (log.isDebugEnabled())
                log.debug("dump() failure "+t);
        }

        return null;
    }

    /**
     * Ensures (return a sure copy) a string for use in a fileName.
     * I.e.  'toto nada' --&gt; 'toto_nada'
     */
    public static String ensureForFileName (final String in)
    {
        return in
            .replaceAll(" ", "_")
            .replaceAll(":", "_")
            .replaceAll(";", "_")
            .replaceAll("\\*", "_")
            .replaceAll("%", "_");
    }

    /**
     * Returns a long indicating when the resource given by its url 
     * was last modified.
     */
    public static long getLastModified (final String url)
    {
        try
        {
            //return getLastModified(new java.net.URL(url));
            return getLastModified(toUrl(url));
        }
        catch (final Throwable t)
        {
            log.info("getLastModified() failure for "+url, t);
        }

        return -1;
    }

    /**
     * Returns a long indicating when the resource given by its url 
     * was last modified.
     */
    public static long getLastModified (final java.net.URL u)
    {
        try
        {
            return u.openConnection().getLastModified();
        }
        catch (final Throwable t)
        {
            log.info("getLastModified() failure", t);
        }

        return -1;
    }

    /**
     * Writes a bunch of bytes to a file.
     */
    public static void writeToFile (final String fileName, final byte[] bytes)
        throws java.io.IOException
    {
        java.io.FileOutputStream fos = null;
        FileChannel fc = null;
        try
        {
            fos = new java.io.FileOutputStream(fileName);
            fc = fos.getChannel();

            final ByteBuffer bb = ByteBuffer.wrap(bytes);

            fc.write(bb);
        }
        finally
        {
            try
            {
                fc.close();
                fos.close();
            }
            catch (final Throwable t)
            {
                // ignore
            }
        }
    }

    /**
     * Turns a path into a URL (basically any path, even those with white 
     * spaces).
     */
    public static java.net.URL toUrl (final String path)
        throws java.net.MalformedURLException
    {
        try
        {
            return new java.net.URL(expandUrl(path));
        }
        catch (final java.net.MalformedURLException e)
        {
            // ignore
        }

        final java.io.File f = new java.io.File(path);

        return f.toURI().toURL();
    }

    /**
     * Turns a relative file url into an absolute one (and makes sure that
     * the "file:" prefix is set).
     * If the url is for another protocol, the output will be the same as the
     * input.
     */
    public static String expandUrl (final String url)
    {
        return expandUrl(null, url);
    }

    /**
     * Makes sure the result is an URL, and if its a path URL, will make
     * sure that it's an absolute and canonical path.
     */
    public static String expandUrl 
        (final String applicationDirectory, final String url)
    {
        if (log.isDebugEnabled())
            log.debug("expandUrl() considering >"+url+"<");

        // openwfe exceptions...
        if (url.startsWith("field:")) return url;
        if (url.startsWith("resource:")) return url;

        if (url.startsWith("file:"))
        {
            final String fn = expandFileName
                (applicationDirectory, removeFileProtocolPrefix(url));

            //final String result = getFileProtocolPrefix() + fn;

            String result = fn;
            if (fn.indexOf(" ") < 0) result = getFileProtocolPrefix() + fn;

            if (log.isDebugEnabled())
                log.debug("expandUrl() 0 returning >"+result+"<");

            return result;
        }

        if ( ! isUrl(url))
            return expandUrl(applicationDirectory, getFileProtocolPrefix()+url);

        if (log.isDebugEnabled())
            log.debug("expandUrl() 1 returning >"+url+"<");

        return url;

        //
        // java.io.File.toUrl() ??
        //
    }

    /*
    public static String pathToUrl (final String path)
    {
    }
    */

    /**
     * Takes as input a String containing an url and makes sure
     * to output the same url with a protocol if missing, the protocol
     * added is "file:".
     */
    public static String ensureProtocol (String url)
    {
        url = url.trim();

        final int colon = url.indexOf(":");
        if (colon < 0 ||
            colon == 1) // probably something like "D:\nada\"
        {
            return getFileProtocolPrefix() + url;
        }

        return url;
    }

    /**
     * Returns true if the given string is an URL.
     */
    public static boolean isUrl (final String s)
    {
        try
        {
            new java.net.URL(s);
        }
        catch (final Throwable t)
        {
            return false;
        }
        return true;
    }

}
