/*
 * Copyright (c) 2001-2006, John Mettraux, OpenWFE.org
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * . Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.  
 * 
 * . Redistributions in binary form must reproduce the above copyright notice, 
 *   this list of conditions and the following disclaimer in the documentation 
 *   and/or other materials provided with the distribution.
 * 
 * . Neither the name of the "OpenWFE" nor the names of its contributors may be
 *   used to endorse or promote products derived from this software without
 *   specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: PolicyService.java 3077 2006-08-30 06:01:05Z jmettraux $
 */

//
// PolicyService.java
//
// jmettraux@openwfe.org
//
// generated with 
// jtmpl 1.0.04 31.10.2002 John Mettraux (jmettraux@openwfe.org)
//

package openwfe.org.auth;

import java.security.Policy;
import java.security.CodeSource;
import java.security.Permissions;
import java.security.PermissionCollection;
import java.security.ProtectionDomain;
import javax.security.auth.Subject;

import openwfe.org.MapUtils;
import openwfe.org.RunnableService;
import openwfe.org.ServiceException;
import openwfe.org.ApplicationContext;
import openwfe.org.state.PausedState;
import openwfe.org.state.RunningState;
import openwfe.org.state.StoppedState;
import openwfe.org.state.ServiceState;


/**
 * Our custom policy. Should be usable out of the box.
 *
 * Important note :
 * Log ouptut for Passwd and PolicyService has been commented out, it
 * induced stack overflow errors when log4j was rotating its log files (and
 * thus requesting this PolicyService for filepermissions).<br>
 * Feel free to comment in log output, but beware to comment it out for
 * production builds.
 * </p>
 *
 * <p><font size=2>CVS Info :
 * <br>$Author: jmettraux $
 * <br>$Date: 2006-08-30 08:01:05 +0200 (Wed, 30 Aug 2006) $
 * <br>$Id: PolicyService.java 3077 2006-08-30 06:01:05Z jmettraux $ </font>
 *
 * @author jmettraux@openwfe.org
 */
public class PolicyService

    extends Policy

    implements RunnableService

{

    private final static org.apache.log4j.Logger log = org.apache.log4j.Logger
        .getLogger(PolicyService.class.getName());

    //
    // CONSTANTS and definitions

    /**
     * This is the classical service name for a policy service.
     * This is the name used in lookupPolicyService() for example.
     */
    public final static String POLICY_SERVICE = "policyService";

    /**
     * This parameter 'passwdCodec' awaits as value the name of a class
     * extending openwfe.org.auth.PasswdCodec. An instance of this class
     * will be created upon initializing this policy service and will
     * be used to authentify and authorize users.
     */
    public final static String P_PASSWD_CODEC = "passwdCodec";

    /**
     * If this parameter 'refreshEachTime' is set to true, the passwd data
     * is reloaded each time.
     * (This parameter will be replaced by a better way of determining 
     * changes in passwd files / databases very soon)
     */
    public final static String P_REFRESH_EACH_TIME = "refreshEachTime";

    //
    // FIELDS

    private String serviceName = null;
    private ApplicationContext applicationContext = null;
    private java.util.Map serviceParams = null;
    private ServiceState serviceState = null;

    private Policy deferredPolicy = null;

    private PasswdCodec passwdCodec = null;

    private Passwd passwd = null;

    private boolean refreshEachTime = false;

    //
    // CONSTRUCTORS

    public void init 
        (final String serviceName, 
         final ApplicationContext context, 
         final java.util.Map serviceParams)
    throws 
        ServiceException
    {
        this.serviceName = serviceName;
        this.applicationContext = context;
        this.serviceParams = serviceParams;

        //
        // prepare passwdCodec

        String passwdCodecClassName = 
            (String)this.serviceParams.get(P_PASSWD_CODEC);
        if (passwdCodecClassName == null)
        {
            passwdCodecClassName = openwfe.org.auth.xml.XmlPasswdCodec.class
                .getName();
            log.info("init() using default PasswdCodec");
        }

        log.info("init() using PasswdCodec '"+passwdCodecClassName+"'");

        try
        {
            Class codecClass = Class.forName(passwdCodecClassName);
            this.passwdCodec = (PasswdCodec)codecClass.newInstance();
        }
        catch (final Exception e)
        {
            throw new ServiceException
                ("Failed to load codec, service '"+this.serviceName+
                 "' cannot continue.", e);
        }

        this.passwdCodec.init(this.applicationContext, this.serviceParams);

        //
        // should we refresh the passwd each time ?
        
        this.refreshEachTime = MapUtils
            .getAsBoolean(serviceParams, P_REFRESH_EACH_TIME, false);

        log.info("init() refresh passwd each time ? "+this.refreshEachTime);

        //
        // load deferredPolicy
        
        this.deferredPolicy = Policy.getPolicy();

        //
        // load passwd

        loadPasswd();

        log.info("init() passwd successfully decoded");

        //
        // set self as system policy !
        
        Policy.setPolicy(this);

        log.info("init() set self as system security Policy.");
    }

    protected void loadPasswd ()
        throws ServiceException
    {
        try
        {
            this.passwd = this.passwdCodec.decodePasswd();
        }
        catch (final AuthException ae)
        {
            throw new ServiceException
                ("Failed to load Passwd", ae);
        }
    }

    //
    // METHODS

    public Principal authentify 
        (final String userName, final Object credentials)
    throws 
        AuthException
    {
        if (this.refreshEachTime) this.refresh();
        //this.refresh();

        return this.passwd.authentify(userName, credentials);
    }

    //
    // METHODS from Service

    public void play ()
        throws ServiceException
    {
        log.info("play() requested for service '"+getName()+"'");

        if ( ! this.isRunning()) this.serviceState = new RunningState();
    }

    public void pause ()
        throws ServiceException
    {
        log.info("pause() requested for service '"+getName()+"'");

        if (this.isRunning()) this.serviceState = new PausedState();
    }

    public void stop ()
        throws ServiceException
    {
        log.info("stop() requested for service '"+getName()+"'");

        this.serviceState = new StoppedState();
    }

    public void update ()
        throws ServiceException
    {
        log.info("update() requested for service '"+getName()+"'");

        // nothing to do...
    }

    public ServiceState getState () 
    {
        return this.serviceState;
    }

    protected void setState (ServiceState state)
    {
        this.serviceState = state;
    }

    public String getName ()
    {
        return this.serviceName;
    }

    public ApplicationContext getContext ()
    {
        return this.applicationContext;
    }

    public java.util.Map getParams ()
    {
        return java.util.Collections.unmodifiableMap(this.serviceParams);
    }

    public org.jdom.Element getStatus ()
    {
        return new org.jdom.Element(getName());
    }

    public boolean isRunning ()
    {
        return getState() instanceof RunningState;
    }

    //
    // METHODS from Policy

    public PermissionCollection getPermissions (final CodeSource cs)
    {
        final PermissionCollection permissions = 
            this.deferredPolicy.getPermissions(cs);

        // debug
        //final StringBuffer sb = new StringBuffer();
        //final java.util.Enumeration en = permissions.elements();
        //int i = 0;
        //while (en.hasMoreElements())
        //{
        //    java.security.Permission p = 
        //      (java.security.Permission)en.nextElement();
        //    sb.append("\n - "+(i++)+": "+p);
        //}
        //log.debug("getPermissions(cs) returning"+sb.toString());

        return permissions;
    }

    public PermissionCollection getPermissions (final ProtectionDomain pd)
    {
        //log.debug("getPermissions()");

        PermissionCollection permissions = 
            this.deferredPolicy.getPermissions(pd);

        PermissionCollection passwdPermissions = null;
        try
        {
            passwdPermissions = this.passwd.getPermissions(pd);
        }
        catch (final Exception e)
        {
            //log.warn
            //    ("getPermissions() "+
            //     "Failed to use 'passwd' to determine permissions, "+
            //     "falling back to system permissions\nCaused by : "+e);

            passwdPermissions = new Permissions();
        }

        java.util.Enumeration en = passwdPermissions.elements();
        while (en.hasMoreElements())
        {
            java.security.Permission p = 
                (java.security.Permission)en.nextElement();
            permissions.add(p);
        }

        // debug
        //StringBuffer sb = new StringBuffer();
        //en = permissions.elements();
        //int i = 0;
        //while (en.hasMoreElements())
        //{
        //    java.security.Permission p = 
        //      (java.security.Permission)en.nextElement();
        //    sb.append("\n - "+(i++)+": "+p);
        //}
        //log.debug("getPermissions(pd) returning"+sb.toString());

        return permissions;
    }

    public PermissionCollection getPermissions (Subject subject)
        throws AuthException
    {
        return this.passwd.getPermissions(subject);
    }

    public void refresh ()
    {
        log.info("refresh() '"+this.serviceName+"'");

        try
        {
            loadPasswd();
        }
        catch (final ServiceException se)
        {
            //log.warn("Passwd reload failed", se);
        }

        this.deferredPolicy.refresh();
    }

    //
    // UMAN METHODS
    // (restricted access)

    public java.util.List getPrincipals ()
    {
        checkAccess();
        return this.passwd.getPrincipals();
    }

    public java.util.Map getGrants ()
    {
        checkAccess();
        return this.passwd.getGrants();
    }

    public void updatePrincipals (final java.util.List principals)
    {
        checkAccess();
        this.passwd.updatePrincipals(principals);
    }

    public void updateGrants (final java.util.Map grants)
    {
        checkAccess();
        this.passwd.updateGrants(grants);
    }

    public void savePasswd ()
        throws AuthException
    {
        checkAccess();

        this.passwdCodec.encodePasswd(this.passwd);
    }

    // 
    // METHODS

    protected void checkAccess ()
    {
        final java.util.Map params = new java.util.HashMap(1);
        params.put(Permission.NAME, this.passwd.getName());

        java.security.AccessController.checkPermission
            (new UmanPermission(params));
    }

    //
    // STATIC METHODS

    public static PolicyService lookupPolicyService 
        (final ApplicationContext context)
    {
        //log.debug
        //    ("lookupPolicyService() in context '"+
        //     context.getApplicationName()+"'");

        final PolicyService ps = (PolicyService)context.lookup(POLICY_SERVICE);

        if (ps != null) return ps;

        if (context.getParentContext() == null) return null;

        return lookupPolicyService(context.getParentContext());
    }

}
