/*
 * Copyright (c) 2001-2006, John Mettraux, OpenWFE.org
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * . Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.  
 * 
 * . Redistributions in binary form must reproduce the above copyright notice, 
 *   this list of conditions and the following disclaimer in the documentation 
 *   and/or other materials provided with the distribution.
 * 
 * . Neither the name of the "OpenWFE" nor the names of its contributors may be
 *   used to endorse or promote products derived from this software without
 *   specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: SocketService.java 3077 2006-08-30 06:01:05Z jmettraux $
 */

//
// SocketService.java
//
// jmettraux@openwfe.org
//
// generated with 
// jtmpl 1.0.04 20.11.2001 John Mettraux (jmettraux@openwfe.org)
//

package openwfe.org.net;

import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.nio.channels.Selector;
import java.nio.channels.SelectionKey;
import java.nio.channels.SocketChannel;
import java.nio.channels.SelectableChannel;
import java.nio.channels.ServerSocketChannel;

import openwfe.org.MapUtils;
import openwfe.org.AbstractService;
import openwfe.org.ServiceException;
import openwfe.org.ApplicationContext;


/**
 * An abstract class for managing services that wait and handle socket
 * connections
 *
 * <p><font size=2>CVS Info :
 * <br>$Author: jmettraux $
 * <br>$Date: 2006-08-30 08:01:05 +0200 (Wed, 30 Aug 2006) $
 * <br>$Id: SocketService.java 3077 2006-08-30 06:01:05Z jmettraux $ </font>
 *
 * @author john.mettraux@openwfe.org
 */
public abstract class SocketService

    extends AbstractService

{

    private final static org.apache.log4j.Logger log = org.apache.log4j.Logger
        .getLogger(SocketService.class.getName());

    //
    // CONSTANTS (definitions)

    /**
     * Use the parameter 'port' to tell this service on which port it should
     * listen.
     * By default, this service will listen on port 7000.
     */
    public final static String P_PORT 
        = "port";

    /**
     * Use the parameter 'inetAddress' to tell the service on which interface
     * it should listen : by default, it listens on all interfaces.
     */
    public final static String P_INET_ADDRESS
        = "inetAddress";

    /* *
     * Use this parameter to specify wich socket factory should be used
     * by this service.
     * /
    public final static String P_SERVER_SOCKET_FACTORY
        = "serverSocketFactory";
     */

    //
    // FIELDS

    private int port = -1;
    private int defaultPort = 7000;

    //
    // CONSTRUCTORS

    public void init 
        (final String serviceName, 
         final ApplicationContext context, 
         final java.util.Map serviceParams)
    throws 
        ServiceException
    {
        super.init(serviceName, context, serviceParams);

        //
        // get port
        
        this.port = MapUtils.getAsInt(serviceParams, P_PORT, getDefaultPort());

        if (log.isDebugEnabled())
            log.debug("init() listening on port "+this.port);

        //
        // get inetAddress (for machines with multiple net interfaces)

        InetAddress inetAddress = null;

        String sInetAddress = 
            MapUtils.getAsString(serviceParams, P_INET_ADDRESS);

        if (sInetAddress != null)
        {
            try
            {
                inetAddress = 
                    java.net.InetAddress.getByName(sInetAddress);
            }
            catch (Exception e)
            {
                log.info
                    ("Cannot resolve '"+sInetAddress+
                     "' to an inet address, listening on all interfaces", e);
            }
        }

        //
        // get serverSocketFactory

        /*
        javax.net.ServerSocketFactory factory = null;

        String sFactoryClass = 
            (String)serviceParams.get(P_SERVER_SOCKET_FACTORY);
        try
        {
            Class factoryClass = Class.forName(sFactoryClass);

            java.lang.reflect.Method getDefault = factoryClass
                .getDeclaredMethod("getDefault", new Class[] {});

            factory = (javax.net.ServerSocketFactory)getDefault
                .invoke(null, new Object[] {});
        }
        catch (Exception e)
        {
            if (sFactoryClass != null)
                log.info("Failed to use socket factory '"+sFactoryClass+"'", e);

            factory = javax.net.ServerSocketFactory.getDefault();
        }
        log.info
            ("Using serverSocketFactory '"+factory.getClass().getName()+"'");
        */

        //
        // listen...
        
        try
        {
            ServerSocketChannel serverSocketChannel = 
                ServerSocketChannel.open();

            serverSocketChannel.socket()
                .bind(new InetSocketAddress(inetAddress, this.port));

            serverSocketChannel.configureBlocking(false);

            final Selector selector = Selector.open();

            serverSocketChannel.register(selector, SelectionKey.OP_ACCEPT);

            (new Thread() 
             {
                 public void run ()
                 {
                    while (SocketService.this.isRunning())
                    {
                        try
                        {
                            int n = selector.select();

                            if (n == 0) continue;

                            java.util.Iterator it = 
                                selector.selectedKeys().iterator();

                            while (it.hasNext())
                            {
                                SelectionKey key = (SelectionKey)it.next();

                                if (key.isAcceptable())
                                {
                                    ServerSocketChannel ssc = 
                                        (ServerSocketChannel)key.channel();

                                    SocketChannel sc = ssc.accept();

                                    if (log.isDebugEnabled())
                                    {
                                        log.debug
                                            ("Incoming connection: "+
                                             prettyPrintSocket(sc.socket()));
                                    }

                                    //registerChannel
                                    //    (selector, 
                                    //     sc, 
                                    //     SelectionKey.OP_READ | 
                                    //     SelectionKey.OP_WRITE);
                                        //
                                        // simply works without the OP_WRITE
                                        //
                                    registerChannel
                                        (selector, 
                                         sc, 
                                         SelectionKey.OP_READ);

                                    log.debug("channel registered.");
                                }

                                if (key.isReadable())
                                {
                                    handle(key);
                                }

                                it.remove();
                            }
                        }
                        catch (final Throwable t)
                        {
                            log.info("Handling of a socket failed", t);
                        }
                    }
                 }
             }).start();
        }
        catch (final Throwable t)
        {
            log.warn("Socket handling failed. Service stopped.", t);
        }
    }

    private static String prettyPrintSocket (final java.net.Socket socket)
    {
        StringBuffer sb = new StringBuffer();

        sb.append("remote ");
        sb.append(socket.getInetAddress());
        sb.append(":");
        sb.append(socket.getPort());
        sb.append(" <--> local ");
        sb.append(socket.getLocalAddress());
        sb.append(":");
        sb.append(socket.getLocalPort());

        return sb.toString();
    }

    private static void registerChannel 
        (Selector selector, SelectableChannel channel, int operations)
    throws
        java.io.IOException
    {
        if (channel == null) return;
        channel.configureBlocking(false);
        channel.register(selector, operations);
    }

    //
    // METHODS

    public int getPort ()
    {
        return this.port;
    }

    public int getDefaultPort ()
    {
        return this.defaultPort;
    }

    public void setDefaultPort (int port)
    {
        this.defaultPort = port;
    }

    //
    // ABSTRACT METHOD

    /**
     * How an incoming socket connection is treated.
     */
    public abstract void handle (SelectionKey key)
        throws ServiceException;

}
