/*
 * Copyright (c) 2001-2006, John Mettraux, OpenWFE.org
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * . Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.  
 * 
 * . Redistributions in binary form must reproduce the above copyright notice, 
 *   this list of conditions and the following disclaimer in the documentation 
 *   and/or other materials provided with the distribution.
 * 
 * . Neither the name of the "OpenWFE" nor the names of its contributors may be
 *   used to endorse or promote products derived from this software without
 *   specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: AbstractEngine.java 2535 2006-05-02 14:46:42Z jmettraux $
 */

//
// AbstractEngine.java
//
// john.mettraux@openwfe.org
//
// generated with 
// jtmpl 1.1.01 2004/05/19 (john.mettraux@openwfe.org)
//

package openwfe.org.embed.impl.engine;

import openwfe.org.Service;
import openwfe.org.OwfeRunnable;
import openwfe.org.AbstractService;
import openwfe.org.ServiceException;
import openwfe.org.ApplicationContext;
import openwfe.org.engine.Definitions;
import openwfe.org.engine.launch.Launcher;
import openwfe.org.engine.launch.LaunchException;
import openwfe.org.engine.expool.ExpressionPool;
import openwfe.org.engine.control.ControlSession;
import openwfe.org.engine.control.local.ControlSessionImpl;
import openwfe.org.engine.workitem.LaunchItem;
import openwfe.org.engine.workitem.InFlowWorkItem;
import openwfe.org.engine.workitem.XmlCoderLoader;
import openwfe.org.engine.expressions.ReplyException;
import openwfe.org.engine.expressions.FlowExpressionId;
import openwfe.org.engine.expressions.map.XmlExpressionMap;
import openwfe.org.engine.participants.ParticipantMap;
import openwfe.org.engine.impl.launch.SimpleXmlLauncher;
import openwfe.org.engine.impl.functions.XmlFunctionMap;
import openwfe.org.embed.engine.Engine;
import openwfe.org.embed.engine.EmbeddedParticipant;
import openwfe.org.embed.impl.engine.EmbeddedParticipantMap;


/**
 * The basics of the implementation of an embedded OpenWFE engine.
 *
 * <p><font size=2>CVS Info :
 * <br>$Author: jmettraux $
 * <br>$Id: AbstractEngine.java 2535 2006-05-02 14:46:42Z jmettraux $ </font>
 *
 * @author john.mettraux@openwfe.org
 */
public abstract class AbstractEngine

    extends AbstractService

    implements Engine

{

    private final static org.apache.log4j.Logger log = org.apache.log4j.Logger
        .getLogger(AbstractEngine.class.getName());

    //
    // CONSTANTS & co

    /**
     * By default, this engine is named 'embedded-engine'.
     */
    public final static String DEFAULT_ENGINE_NAME
        = "embedded-engine";

    private final static String S_CONSUMER 
        = "consumer";

    //
    // FIELDS

    //
    // CONSTRUCTORS

    /**
     * Builds an embedded engine. 
     */
    public AbstractEngine ()
        throws ServiceException
    {
        this(null);
    }

    /**
     * Builds an embedded engine with a given name.
     * The necessary engine services are prepared here.
     */
    public AbstractEngine (String engineName)
        throws ServiceException
    {
        if (engineName == null)
            engineName = DEFAULT_ENGINE_NAME;

        final ApplicationContext ac = new ApplicationContext();
        ac.setApplicationName(engineName);

        this.init(engineName, ac, new java.util.HashMap(0));

        //
        // launcher

        final SimpleXmlLauncher launcher = new SimpleXmlLauncher();

        launcher.init(Definitions.S_LAUNCHER, ac, new java.util.HashMap(0));

        ac.add(launcher);

        //
        // expression pool
        //
        // (done in further implementations)

        //
        // participant map

        final EmbeddedParticipantMap pMap = new EmbeddedParticipantMap();

        pMap.init(Definitions.S_PARTICIPANT_MAP, ac, new java.util.HashMap(0));

        ac.add(pMap);

        //
        // expression map

        final XmlExpressionMap eMap = new XmlExpressionMap();

        final java.util.Map eMapParams = new java.util.HashMap(1);

        eMapParams.put
            (XmlExpressionMap.P_EXPRESSION_MAP_FILE, 
             "resource:/expression-map.xml");

        eMap.init(Definitions.S_EXPRESSION_MAP, ac, eMapParams);

        ac.add(eMap);

        //
        // history

        //
        // function map

        final XmlFunctionMap fMap = new XmlFunctionMap();

        final java.util.Map fMapParams = new java.util.HashMap(1);

        fMapParams.put
            (XmlFunctionMap.P_FUNCTION_MAP_FILE,
             "resource:/function-map.xml");

        fMap.init(Definitions.S_FUNCTION_MAP, ac, fMapParams);

        ac.add(fMap);

        //
        // WorkItemCoderLoader

        final XmlCoderLoader coderLoader = new XmlCoderLoader();

        final java.util.Map clParams = new java.util.HashMap(1);

        clParams.put(CONFIGURATION_FILE, "resource:/coder-configuration.xml");

        coderLoader.init
            (Definitions.S_WORKITEM_CODER_LOADER, ac, clParams);

        ac.add(coderLoader);
    }

    //
    // METHODS from Engine
    
    /**
     * Given a launch item, launches a flow in the embedded engine.
     */
    public FlowExpressionId launch (final LaunchItem li, final boolean async)
        throws LaunchException
    {
        final Launcher l = Definitions.getLauncher(getContext());

        return l.launch(li, async);
    }

    /**
     * Given the URL of a process definition, launches an instance of
     * it in the embedded engine.
     */
    public FlowExpressionId launch (final String flowUrl, final boolean async)
        throws LaunchException
    {
        final LaunchItem li = new LaunchItem();

        li.setWorkflowDefinitionUrl(flowUrl);

        return launch(li, async);
    }

    /**
     * Registers a participant in the participant map of the embedded engine.
     */
    public void registerParticipant (final EmbeddedParticipant p)
    {
        log.debug("registerParticipant() '"+p.getRegex()+"'");

        p.setEngine(this);

        final ParticipantMap pm = Definitions.getParticipantMap(getContext());

        p.init(p.getRegex(), new java.util.HashMap());

        pm.add(p);
    }

    /**
     * When a participant has finished working with a workitem, it uses this
     * method to give it back to the engine.
     */
    public void reply (final InFlowWorkItem wi)
        throws ReplyException
    {
        log.debug("reply() to "+wi.getLastExpressionId());

        getExpressionPool().reply(wi.getLastExpressionId(), wi);
    }

    /**
     * If daemon is set to true (the default value), the embedded engine will
     * allow the VM to exit once there are no more pending jobs in its 
     * scheduler.
     */
    public void setDaemon (boolean b)
    {
        getExpressionPool().getScheduler().setDaemonIfIdle(b);
    }

    /**
     * @see #setDaemon
     */
    public boolean isDaemon ()
    {
        return getExpressionPool().getScheduler().isDaemonIfIdle();
    }

    /**
     * Stops the engine and makes sure the underlying resources are properly
     * disposed of.
     */
    public void stop ()
        throws ServiceException
    {
        ((OwfeRunnable)getExpressionPool()).stop();

        super.stop();
    }
    
    /**
     * Returns a control session on this engine.
     */
    public ControlSession getControlSession ()
    {
        try
        {
            final ControlSession cs = new ControlSessionImpl();

            //cs.init(getContext(), new java.util.HashMap(0), null);

            final String sessionServiceName =
                getName() + "_session_" + cs.toString();

            // no subject to store in the params.

            ((Service)cs).init
                (sessionServiceName, getContext(), new java.util.HashMap(0));

            return cs;
        }
        catch (final Throwable t)
        {
            log.warn
                ("getControlSession() failed to establish control session", t);
        }
        
        return null;
    }

    //
    // METHODS
    
    /**
     * Returns the expression pool servicing this embedded engine.
     */
    protected ExpressionPool getExpressionPool ()
    {
        return Definitions.getExpressionPool(getContext());
    }

    //
    // STATIC METHODS

}
