/*
 * Copyright (c) 2001-2006, John Mettraux, OpenWFE.org
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * . Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.  
 * 
 * . Redistributions in binary form must reproduce the above copyright notice, 
 *   this list of conditions and the following disclaimer in the documentation 
 *   and/or other materials provided with the distribution.
 * 
 * . Neither the name of the "OpenWFE" nor the names of its contributors may be
 *   used to endorse or promote products derived from this software without
 *   specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: AbstractWorklist.java 2494 2006-04-15 21:15:08Z jmettraux $
 */

//
// AbstractWorklist.java
//
// john.mettraux@openwfe.org
//
// generated with 
// jtmpl 1.1.01 2004/05/19 (john.mettraux@openwfe.org)
//

package openwfe.org.embed.impl.worklist;

import openwfe.org.ApplicationContext;
import openwfe.org.engine.workitem.WorkItem;
import openwfe.org.engine.workitem.InFlowWorkItem;
import openwfe.org.engine.dispatch.DispatchingException;
import openwfe.org.engine.expressions.ReplyException;
import openwfe.org.engine.expressions.FlowExpressionId;
import openwfe.org.engine.participants.AbstractParticipant;
import openwfe.org.worklist.WorkListException;
import openwfe.org.worklist.store.WorkItemStorage;
import openwfe.org.embed.engine.Engine;
import openwfe.org.embed.worklist.Worklist;


/**
 * Half an implementation of an embedded OpenWFE worklist.
 *
 * <p><font size=2>CVS Info :
 * <br>$Author: jmettraux $
 * <br>$Id: AbstractWorklist.java 2494 2006-04-15 21:15:08Z jmettraux $ </font>
 *
 * @author john.mettraux@openwfe.org
 */
public abstract class AbstractWorklist

    extends AbstractParticipant

    implements Worklist

{

    private final static org.apache.log4j.Logger log = org.apache.log4j.Logger
        .getLogger(AbstractWorklist.class.getName());

    //
    // CONSTANTS & co

    //
    // FIELDS

    private Engine engine = null;

    private String listName = null;

    //
    // CONSTRUCTORS

    /**
     * Builds a worklist for a participant name that is not a regular expression
     * (thus, the participant name and the list name will be the same).
     */
    public AbstractWorklist 
        (final String participantName)
    {
        this(participantName, participantName);
    }

    /**
     * Builds a worklist where the participant name is a regular expression and
     * thus cannot be employed as a list name : the list name is given as
     * a parameter of this constructor.
     */
    public AbstractWorklist 
        (final String participantRegex, final String listName)
    {
        super();

        this.setRegex(participantRegex);

        this.listName = listName;
    }

    //
    // METHODS

    public String getListName ()
    {
        return this.listName;
    }

    //
    // ABSTRACT METHODS

    /**
     * Returns the storage attached to this embedded worklist. 
     * Will throw a WorkListException if no storage could be found.
     */
    public abstract WorkItemStorage getStorage ()
        throws WorkListException;

    //
    // METHODS from Participant and EmbeddedParticipant

    /**
     * This method is used by the registerParticipant() method of the embedded
     * engine (so that the participant knows which is its engine).
     */
    public void setEngine (final Engine e)
    {
        this.engine = e;
    }

    /**
     * A shortcut method, allowing the embedded participants full implementation
     * to return their workitem to the emitting engine after consumption.
     */
    public void replyToEngine (final InFlowWorkItem wi)
        throws ReplyException
    {
        this.engine.reply(wi);
    }

    /**
     * This method is used by the embedded engine to transmit the workitems
     * to the embedded participant instance.
     */
    public void consume (final WorkItem wi)
        throws Exception
    {
        this.save((InFlowWorkItem)wi);
    }

    /**
     * This is a method from the engine/Participant class : a regular
     * OpenWFE engines doesn't know about embedded participants, but it
     * knows about [plain] participants.
     * This method wraps the consume() call and thus bridges the regular
     * participant that the classical engine manipulates to the 
     * embedded participant.
     */
    public Object dispatch 
        (final ApplicationContext context, final WorkItem wi)
    throws 
        DispatchingException
    {
        try
        {
            consume(wi);
            return null;
        }
        catch (final Throwable t)
        {
            log.debug
                ("embedded participant failure", t);

            throw new DispatchingException
                ("embedded participant failure", t);
        }
    }

    //
    // METHODS from Worklist

    /**
     * Returns how much workitems are stored in this list.
     */
    public int countWorkItems ()
        throws WorkListException
    {
        return getStorage().countWorkItems(this.listName);
    }

    /**
     * An iterator on the stored workitems themselves.
     */
    public java.util.Iterator iterator ()
        throws WorkListException
    {
        return getStorage().listWorkItems(this.listName, -1).iterator();
    }

    /**
     * Returns a workitem from its given id.
     */
    public InFlowWorkItem get (final FlowExpressionId fei)
        throws WorkListException
    {
        return getStorage().retrieveWorkItem(this.listName, fei);
    }

    /**
     * Saves a workitem.
     */
    public void save (final InFlowWorkItem wi)
        throws WorkListException
    {
        getStorage().storeWorkItem(this.listName, (InFlowWorkItem)wi.clone());
    }

    /**
     * Forwards a workitem (make it proceed in its flow).
     */
    public void forward (final InFlowWorkItem wi)
        throws WorkListException
    {
        getStorage().removeWorkItem(this.listName, wi.getLastExpressionId());

        try
        {
            replyToEngine(wi);
        }
        catch (final ReplyException e)
        {
            throw new WorkListException
                ("Forward (proceed) failure", e);
        }
    }

    //
    // METHODS from Service

    //
    // METHODS

    //
    // STATIC METHODS

}
