/*
 * Copyright (c) 2005, John Mettraux, OpenWFE.org
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * . Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.  
 * 
 * . Redistributions in binary form must reproduce the above copyright notice, 
 *   this list of conditions and the following disclaimer in the documentation 
 *   and/or other materials provided with the distribution.
 * 
 * . Neither the name of the "OpenWFE" nor the names of its contributors may be
 *   used to endorse or promote products derived from this software without
 *   specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: AbstractFlowExpression.java 2466 2006-04-12 20:10:40Z jmettraux $
 */

//
// FlowExpression.java
//
// jmettraux@openwfe.org
//
// generated with 
// jtmpl 1.0.04 20.11.2001 John Mettraux (jmettraux@openwfe.org)
//

package openwfe.org.engine.expressions;

import openwfe.org.Utils;
import openwfe.org.ReflectionUtils;
import openwfe.org.ApplicationContext;
import openwfe.org.time.Time;
import openwfe.org.misc.Text;
import openwfe.org.engine.Definitions;
import openwfe.org.engine.launch.Launcher;
import openwfe.org.engine.expool.PoolException;
import openwfe.org.engine.expool.ExpressionPool;
import openwfe.org.engine.history.History;
import openwfe.org.engine.workitem.WorkItem;
import openwfe.org.engine.workitem.InFlowWorkItem;
import openwfe.org.engine.expressions.raw.RawExpression;
import openwfe.org.engine.expressions.map.ExpressionMap;
import openwfe.org.engine.expressions.state.ExpressionState;
import openwfe.org.engine.participants.ParticipantMap;


/**
 * For each XML tag in an OpenWFE process definition, there is a FlowExpression
 * instance.
 * Each expression is tagged with (designated by) a flowExpressionId. You may
 * think of expression as pieces of workflow instance pointing to each other
 * through references (FlowExpressionId instances).
 *
 * <p><font size=2>CVS Info :
 * <br>$Author: jmettraux $
 * <br>$Date: 2006-04-12 22:10:40 +0200 (Wed, 12 Apr 2006) $
 * <br>$Id: AbstractFlowExpression.java 2466 2006-04-12 20:10:40Z jmettraux $ </font>
 *
 * @author john.mettraux@openwfe.org
 */
public abstract class AbstractFlowExpression

    implements FlowExpression

{

    private final static org.apache.log4j.Logger log = org.apache.log4j.Logger
        .getLogger(AbstractFlowExpression.class.getName());

    //
    // FIELDS

    private FlowExpressionId id = null;

    private FlowExpressionId parent = null;

    //private FlowExpressionId next = null;
    //private FlowExpressionId previous = null;
    //private FlowExpressionId behalf = null;

    private String applyTime = null;

    private transient ApplicationContext applicationContext = null;

    private java.util.Map attributes = null;

    private FlowExpressionId envId = null;

    private ExpressionState state = null;

    //
    // CONSTRUCTORS

    /**
     * The constructor for a flow expression. The bulk work of initialization
     * is though done in the 'build()' method.
     */ 
    public AbstractFlowExpression ()
    {
        super();

        this.attributes = new java.util.HashMap(7);
    }

    //
    // BEAN METHODS

    /**
     * Returns the FlowExpressionId of this expression.
     */
    public FlowExpressionId getId () { return this.id; }

    /**
     * Returns the FlowExpressionId of the parent expression of this
     * expression.
     */
    public FlowExpressionId getParent () { return this.parent; }

    /* *
     * Sometimes (especially in the case of a sequence), having children
     * expressions always replying to the parent is not necessary, they
     * can directly apply their neighbor, this 'next' member is here to
     * allow such behaviour.
     * /
    public FlowExpressionId getNext () { return this.next; }

    /* *
     * A parallel to the 'next' field : this 'previous' points to the
     * previous expression in the flat sequence, this is used mainly
     * in cancel() where a parent knows only about the last expression
     * in a sequence and thus calls cancel on it, this last expression then
     * calls cancel() on its previous expression (if they are still around).
     * /
    public FlowExpressionId getPrevious () { return this.previous; }

    /* *
     * See javadoc in father class.
     * /
    public FlowExpressionId getBehalf () { return this.behalf; }
     */

    /**
     * Returns the string representing the moment at which this expression
     * got applied (or null if it was not yet applied).
     */
    public String getApplyTime () { return this.applyTime; }

    /**
     * Returns the attribute of this expression (they tune the behaviour
     * of the expression).
     */
    public java.util.Map getAttributes () { return this.attributes; }

    public FlowExpressionId getEnvironmentId () { return this.envId; }


    public void setId (FlowExpressionId fei) { this.id = fei; }
    public void setParent (FlowExpressionId fei) { this.parent = fei; }
    //public void setNext (FlowExpressionId fei) { this.next = fei; }
    //public void setPrevious (FlowExpressionId fei) { this.previous = fei; }
    //public void setBehalf (FlowExpressionId fei) { this.behalf = fei; }
    public void setApplyTime (String isoDate) { this.applyTime = isoDate; }
    public void setAttributes (java.util.Map m) { this.attributes = m; }
    public void setEnvironmentId (FlowExpressionId fei) { this.envId = fei; }

    public void setApplicationContext (ApplicationContext ac) { this.applicationContext = ac; }

    //
    // METHODS

    /**
     * For a regular expression, will be equivalent to getClass(), for
     * a RawExpression, will return the 'encapsulated' (non-compiled) 
     * expression class.
     */
    public Class getExpressionClass ()
    {
        return this.getClass();
    }

    /**
     * This method returns the applicationContext currently attached to this 
     * FlowExpression.
     */
    public ApplicationContext context ()
    {
        return this.applicationContext;
    }

    /*
     * determines the behalf variable name
     * /
    private String behalfVariableName (final FlowExpressionId behalf)
    {
        return V_BEHALF + behalf.hashCode();
    }

    /* *
     * SetBehalf is a bean method, it just sets the behalf, this method
     * takes care of also registering the behalf in the environment.
     * /
    public void placeBehalf (final FlowExpressionId behalf)
    {
        this.behalf = behalf;

        //this.bindVariable
        //    (behalfVariableName(behalf), this.getId().toParseableString());

        final Behalf b = new Behalf(behalf, this.getId());

        try
        {
            getExpressionPool().add(b);

            log.debug
                ("placeBehalf() behalf : "+this.behalf+
                 " -->  "+this.getId());
        }
        catch (final PoolException e)
        {
            log.warn
                ("placeBehalf() failed to place behalf : "+this.behalf+
                 " -->  "+this.getId());
        }
    }
     */

    /**
     * This method will return false if the given String contains some
     * $ notation element, it's generally used from inside lookupAttribute().
     * For instance <pre>referencesNoVariable("field-${name}")</pre> will yield
     * false.
     */
    protected static boolean referencesNoVariable (final String attributeValue)
    {
        return (attributeValue == null || attributeValue.indexOf("${") < 0);
    }

    /**
     * Looks a process definition attribute ie in the example<br>
     * &lt;participant ref="toto" /&gt;<br>
     * lookupAttribute("ref");<br>
     * will return "toto".
     * If it were<br>
     * &lt;participant ref="toto_${index}" /&gt;<br>
     * and the variable "index" would be set to "2",<br>
     * lookupAttribute("ref");<br>
     * would have returned "toto_2".
     */
    public String lookupAttribute 
        (final String attributeName,
         final InFlowWorkItem wi)
    {
        //log.debug("lookupAttribute() '"+attributeName+"'");

        final String attributeValue = 
            (String)this.attributes.get(attributeName);

        //log.debug("lookupAttribute() value is '"+attributeValue+"'");

        return substituteText(attributeValue, wi);
    }

    /**
     * This variant of lookupAttribute accept as last argument a default
     * value.
     */
    public String lookupAttribute 
        (final String attributeName,
         final InFlowWorkItem wi,
         final String defaultValue)
    {
        final String s = lookupAttribute(attributeName, wi);
        
        if (s != null) return s;

        return defaultValue;
    }

    /**
     * Given a text, will substitute any dollar notations.
     */
    public String substituteText (final String text, final InFlowWorkItem wi)
    {
        log.debug("substituteText() text is    >"+text+"<");

        if (referencesNoVariable(text)) return text;

        //log.debug("substituteText() doing substitution...");

        final VariableMap vMap = new VariableMap(this, wi);

        final String s = Text.substitute(text, vMap);

        log.debug("substituteText() result is  >"+s+"<");

        return s;
    }

    /**
     * sets the apply time to now
     */
    public void touchApplyTime ()
    {
        this.setApplyTime(Time.toIsoDate());
    }

    /**
     * Takes care of looking up a variable (locally or from the expression 
     * pool).
     */
    public Object lookupVariable (final String variableName)
    {
        if (getExpressionPool() == null) return null;

        return getExpressionPool().lookupVariable(this, variableName);
    }

    /**
     * A shortcut method for setting a variable from an expression.
     */
    public void bindVariable (final String variableName, final Object value)
    {
        log.debug("bindVariable() '"+variableName+"' -> '"+value+"'");
        log.debug("bindVariable() for "+getClass().getName()+"  "+getId());

        if (getExpressionPool() == null) return;

        getExpressionPool().setVariable(this, variableName, value);
    }

    /**
     * Sets the workflowInstanceId of this expression (used when
     * initing a flow)
     */
    public void initWorkflowInstanceId (final String workflowInstanceId)
    {
        if (this.parent != null)
            this.parent.setWorkflowInstanceId(workflowInstanceId);

        this.id.setWorkflowInstanceId(workflowInstanceId);
    }
    
    public String getAttributeValue (final String attributeName)
    {
        return (String)this.attributes.get(attributeName);
    }

    /**
     * Sets the lastExpressionId of the workItem as 'this' id
     */
    public void tag (final InFlowWorkItem wi)
    {
        if (wi == null)
        {
            log.warn("tag() wi is null ??? for "+this.getId());
            return;
        }

        //log.debug("tag() was "+wi.getLastExpressionId());

        wi.setId(this.getId());

        //log.debug("tag() is  "+wi.getLastExpressionId());
    }

    /**
     * Use this method when you want to ensure that the changes in the
     * fields of your expression will be kept, even if the expression
     * gets swapped to disk (to the ExpressionStore).
     * Expressions decide by themselves when they have to be stored.
     */
    public void storeItself ()
    {
        log.debug
            ("storeItself() invoked for "+this.getId()+
             " ("+this.getClass().getName()+")");
        log.debug
            ("storeItself() parent is "+this.getParent());

        //Utils.logStackTrace(log, "storeItself");
            //
            // sometimes, a bughunt requires us to know who
            // ordered the storeItself()

        if (this.applicationContext == null ||
            getExpressionPool() == null) 
        {
            log.debug
                ("storeItself() no applicationContext or expressionPool, "+
                 "cannot store.");
            return;
        }

        try
        {
            getExpressionPool().update(this);
        }
        catch (final PoolException pe)
        {
            log.warn("storeItself() failure", pe);
        }
    }

    /**
     * A shortcut to fetch the expression pool from this application context
     */
    public ExpressionPool getExpressionPool ()
    {
        return Definitions.getExpressionPool(this.applicationContext);
    }

    /**
     * A shortcut to fetch the expression map from this application context
     */
    public ExpressionMap getExpressionMap ()
    {
        return Definitions.getExpressionMap(this.applicationContext);
    }

    /**
     * A shortcut to fetch the participant map from this application context
     */
    public ParticipantMap getParticipantMap ()
    {
        return Definitions.getParticipantMap(this.applicationContext);
    }

    /**
     * A shortcut to fetch the launcher from this application context
     */
    public Launcher getLauncher ()
    {
        return Definitions.getLauncher(this.applicationContext);
    }

    /* *
     * Returns the history service
     */
    private History getHistory ()
    {
        return Definitions.getHistory(this.applicationContext);
    }

    /**
     * This method enables child expressions to log events.
     * If the history component is not found, nothing will get logged.
     */
    public void historyLog
        (final WorkItem wi,
         final String eventCode,
         final String participantName,
         final String message)
    {
        final History h = getHistory();

        if (h == null)
        {
            //log.debug
            //    ("historyLog() cannot log as there is no History service.");
            return;
        }

        h.log
            (this.getId(),
             wi,
             eventCode,
             participantName,
             message);
    }

    /**
     * Cancels this expression.
     */
    public InFlowWorkItem cancel () 
        throws ApplyException
    {
        // nothing special to do
        // children will probably need an implementation though...

        return null;
    }

    /**
     * replies to this FlowExpression parent
     */
    public void replyToParent (final InFlowWorkItem wi)
        throws ReplyException
    {
        //log.debug("replyToParent()");
        
        this.tag(wi);

        getExpressionPool().replyToParent(this, wi);
    }

    /**
     * This method may be a bit misnamed, it's simply a replyToParent()
     * wrapped so that it throws an ApplyException instead of a 
     * ReplyException
     */
    public void applyToParent (final InFlowWorkItem wi)
        throws ApplyException
    {
        //log.debug("applyToParent()");

        try
        {
            replyToParent(wi);
        }
        catch (ReplyException re)
        {
            throw new ApplyException
                ("replyToParent() failed", re);
        }
    }

    /**
     * Returns the number of milliseconds elapsed since the expression
     * was applied
     */
    public long getTimeSinceApplied ()
    {
        try
        {
            return 
                System.currentTimeMillis() - Time.fromIsoDate(this.applyTime);
        }
        catch (java.text.ParseException pe)
        {
            return 0L;
        }
    }

    //
    // equality methods

    /**
     * Checks whether a flowExpression is equal to another. This is mainly
     * done based on the hashcode.
     */
    public boolean equals (final Object o)
    {
        if (o == null || ! (o.getClass().equals(this.getClass())))
            return false;

        return o.hashCode() == this.hashCode();
    }

    /**
     * Computes the hashCode for the flowExpression. This is simply the 
     * hashCode of the flowExpressionId designating the flowExpression.
     */
    public int hashCode ()
    {
        if (this.getId() != null)
            return this.getId().hashCode();
        return super.hashCode();
    }

    /**
     * A deep copy of this expression (has probably to be extended by
     * extending child classes).
     */
    public Object clone ()
    {
        final AbstractFlowExpression clone = 
            (AbstractFlowExpression)ReflectionUtils.newInstance(this);

        clone.setId(getId().copy());
        
        if (getEnvironmentId() != null)
            clone.setEnvironmentId(getEnvironmentId().copy());

        if (getParent() != null)
            clone.setParent(getParent().copy());

        //if (getNext() != null)
        //    clone.setNext(getNext().copy());

        clone.applicationContext = context();

        clone.setAttributes(Utils.copyHashMap(getAttributes()));

        // (variables are not cloned.)

        return clone;
    }

    public void init 
        (final ApplicationContext context,
         final FlowExpressionId environmentId,
         final FlowExpressionId parentId, 
         final FlowExpressionId id, 
         final RawExpression generatingExpression,
         final Object raw,
         final InFlowWorkItem currentWi)
    throws 
        BuildException
    {
        setApplicationContext(context);

        this.setId(id);
        this.setParent(parentId);
        this.setEnvironmentId(environmentId);

        this.setAttributes(getLauncher().fetchAttributes(this, raw));

        //this.debugAttributes("init()");
    }

    /**
     * If an ExpressionState is present, the expression pool will use it
     * for any apply() or reply() instead of directly working with
     * the apply() and reply() of the expression.
     */
    public ExpressionState getState ()
    {
        return this.state;
    }

    /**
     * Sets the ExpressionState of the FlowExpression (ie FrozenState, 
     * PausedState or null).
     */
    public void setState (final ExpressionState es)
    {
        this.state = es;
        
        //this.storeItself();
            //
            // No, not here ! 
            // This is a bean method, it's used when the expression
            // gets deserialized.
    }

    /**
     * Returns the environment that this expression is tied to.
     */
    public Environment fetchEnvironment ()
    {
        return (Environment)getExpressionPool().fetch(this.getEnvironmentId());
    }

    /**
     * This method is used by ExpressionPool.dump().
     */
    public org.jdom.Element dump ()
    {
        final org.jdom.Element elt = new org.jdom.Element("expression");

        elt.setAttribute("class", this.getClass().getName());
        elt.setAttribute("id", this.getId().toString());

        /*
        if (this.behalf != null)
        {
            final org.jdom.Element eBehalf = new org.jdom.Element("behalf");
            eBehalf.setText(this.behalf.toString());
            elt.addContent(eBehalf);
        }
        */

        // no treatment on previous

        /*
        if (this.next != null)
        {
            final org.jdom.Element eNext = new org.jdom.Element("next");

            //final FlowExpression feNext = 
            //    getExpressionPool().fetch(this.next);
            //eNext.addContent(feNext.dump());

            eNext.setText(this.next.toString());

            elt.addContent(eNext);
        }
        */

        return elt;
    }

    public void debugDump ()
    {
        final StringBuffer sb = new StringBuffer();

        sb
            .append("debugDump()  ")
            .append(this.getClass().getName())
            .append("\n")
            .append("   ")
            .append(this.getId().toString())
            .append("\n")
            .append("   attributes :\n");

        final java.util.Iterator it = this.getAttributes().keySet().iterator();
        while (it.hasNext())
        {
            final String k = (String)it.next();
            final String v = this.getAttributes().get(k).toString();

            sb
                .append("     - '")
                .append(k)
                .append("' -->  >")
                .append(v)
                .append("<\n");
        }

        log.debug(sb.toString());
    }

    /**
     * log.debug() an XML string representation of the expression.
     */
    public void xmlDebugDump (final String methodName)
    {
        try
        {
            log.debug
                (methodName+"()\n"+
                 openwfe.org.xml.XmlCoder.encodeToString(this, null));
        }
        catch (final Throwable t)
        {
            log.debug
                (methodName+"() failure in xmlDebugDump()", t);
        }
    }

    //
    // SOME STATIC METHODS

    /**
     * returns a deep copy of the source list, each flowExpressionId in the
     * source list is cloned.
     */
    public static java.util.List deepCopy (java.util.List source)
    {
        if (source == null) return new java.util.ArrayList(0);

        java.util.List result = new java.util.ArrayList(source.size());

        java.util.Iterator it = source.iterator();
        while (it.hasNext())
            result.add(((FlowExpressionId)it.next()).clone());

        return result;
    }
    
}
