/*
 * Copyright (c) 2005-2006, John Mettraux, OpenWFE.org
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * . Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.  
 * 
 * . Redistributions in binary form must reproduce the above copyright notice, 
 *   this list of conditions and the following disclaimer in the documentation 
 *   and/or other materials provided with the distribution.
 * 
 * . Neither the name of the "OpenWFE" nor the names of its contributors may be
 *   used to endorse or promote products derived from this software without
 *   specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: CronExpression.java 2522 2006-04-27 14:43:51Z jmettraux $
 */

//
// CronExpression.java
//
// john.mettraux@openwfe.org
//
// generated with 
// jtmpl 1.1.01 2004/05/19 (john.mettraux@openwfe.org)
//

package openwfe.org.engine.expressions;

import openwfe.org.time.Scheduler;
import openwfe.org.time.Schedulable;
import openwfe.org.engine.expool.PoolException;
import openwfe.org.engine.workitem.InFlowWorkItem;
import openwfe.org.engine.expressions.raw.RawExpression;


/**
 * An expression triggering its child expression from time to time
 *
 * <p><font size=2>CVS Info :
 * <br>$Author: jmettraux $
 * <br>$Id: CronExpression.java 2522 2006-04-27 14:43:51Z jmettraux $ </font>
 *
 * @author john.mettraux@openwfe.org
 */
public class CronExpression

    extends OneRawChildExpression

    implements Schedulable, ExpressionToUnbind

{

    private final static org.apache.log4j.Logger log = org.apache.log4j.Logger
        .getLogger(CronExpression.class.getName());

    //
    // CONSTANTS & co

    /**
     * The cron expression awaits a 'tab' attribute indicated the frequency
     * of occurence of the cron jobs.
     * See 'man 5 crontab' for an explanation of the 'tab' attribute format.
     */
    public final static String A_TAB 
        = "tab";

    /**
     * The default value for this 'level' attribute is the empty string, 
     * meaning that the cron is to be bound in the local environment.
     * Other possible values are "/" and "//" for binding the cron at flow
     * level or at engine level respectively.<br>
     * A cron bound at engine level only dies when the engine's VM exits.<br>
     * Other cron do die when their environment is removed.
     */
    public final static String A_LEVEL
        = "level";

    /**
     * A cron may be named, so that a subsequent cron with the same name
     * could unschedule this one and override it.
     */
    public final static String A_NAME
        = "name";

    /*
     * This is the prefix used for building the variable name under
     * which the cronId will be saved in the environment.
     */
    private final static String V_CRON
        = "__cron__";

    /**
     * The history event tag for a cron getting triggered.
     */
    public final static String EVT_CRON
        = "-cro";

    //
    // FIELDS

    private String tab = null;

    private String name = null;
    private String level = null;

    private Long cronId = null;

    private InFlowWorkItem appliedItem = null;

    //
    // CONSTRUCTORS

    /*
    public CronExpression ()
    {
        super();

        log.debug("<init> ...");
    }
    */

    //
    // METHODS

    //
    // BEAN METHODS

    /**
     * Returns the string representation of the frequency set for this
     * Schedulable expression.
     */
    public String getTab ()
    {
        return this.tab;
    }

    /**
     * Returns '', '/' or '//', meaning 'local', 'process' or 'engine' level
     * for the cron.
     */
    public String getLevel ()
    {
        return this.level;
    }

    /**
     * Returns the name of this cron.
     */
    public String getName ()
    {
        return this.name;
    }

    /**
     * Once this expression has been applied, this cronId will be set.
     */
    public Long getCronId () 
    {
        return this.cronId;
    }

    /**
     * Keeping here the workitem as applied when the cron was first scheduled.
     */
    public InFlowWorkItem getAppliedItem ()
    {
        return this.appliedItem;
    }

    public void setTab (final String s)
    {
        this.tab = s;
    }

    public void setLevel (final String s)
    {
        this.level = s;
    }

    public void setName (final String s)
    {
        this.name = s;
    }

    public void setCronId (final Long l)
    {
        this.cronId = l;
    }

    public void setAppliedItem (final InFlowWorkItem wi)
    {
        this.appliedItem = wi;
    }

    //
    // METHODS from Schedulable

    /**
     * This is the method called by the scheduler (in the expool) when 
     * this cron job has to execute.
     * Params are irrelevent (null).
     */
    public void trigger (final Object[] params)
    {
        //
        // no synchronization as the scheduler is currently sequential
        //

        try
        {
            log.debug("trigger() cron.id is  "+getId());
            log.debug("trigger() cron.env is  "+this.getEnvironmentId());

            if (this.appliedItem == null) return;

            final RawExpression child = 
                (RawExpression)getRawChild().clone();

            child.getId().setWorkflowInstanceId
                (RawExpression.determineNewWorkflowInstanceId());

            child.setEnvironmentId(this.getEnvironmentId());

            getExpressionPool().add(child);

            this.historyLog
                (this.appliedItem,
                 EVT_CRON,
                 null,
                 "triggered");

            //FlowExpressionId parentId = this.getId().copy();
            //if (this.level.equals("//")) parentId = null;

            log.debug("trigger() child is  "+child.getId());
            
            getLauncher().launchSub
                (this.appliedItem,
                 null,      // parentId
                 //null,      // no 'on behalf of'
                 child.getId(),
                 null,
                 true);     // async call (will return immediately)
        }
        catch (final Throwable t)
        {
            log.warn("trigger() problem", t);
        }
    }

    public Long reschedule (final Scheduler s)
    {
        return 
            s.schedule(this.tab, this, new Object[] { });
    } 

    //
    // METHODS from ExpressionToUnbind

    public void unbind ()
        throws PoolException
    {
        if (this.cronId == null) return;

        getExpressionPool().getScheduler().unschedule(this.cronId);
    }

    //
    // METHODS from FlowExpression

    /*
     * Determines at which level the cron will be bound.
     */
    private void determineNameAndLevel (final InFlowWorkItem wi)
    {
        final StringBuffer sb = new StringBuffer();

        final String sLevel = lookupAttribute(A_LEVEL, wi);
        final String sName = lookupAttribute(A_NAME, wi);

        if (sLevel != null)
        {
            for (int i=0; i<sLevel.length(); i++)
            {
                if (sb.length() >= 2) break;
                if (sLevel.charAt(i) != '/') break;
                sb.append('/');
            }
        }
        else // sLevel is null
        {
            if (sName != null)
            {
                if (sName.startsWith("//")) 
                {
                    sb.append("//");
                }
                else if (sName.startsWith("/")) 
                {
                    sb.append("/");
                }
            }
        }

        this.name = removeSlashes(sName);
        this.level = sb.toString();

        log.debug("determineNameAndLevel() '"+this.name+"' '"+this.level+"'");
    }

    /*
     * Removes initial slashes out of a string.
     */
    private String removeSlashes (final String s)
    {
        if (s == null) return null;
        if (s.startsWith("/")) return removeSlashes(s.substring(1));
        return s;
    }

    public void apply (final InFlowWorkItem wi)
        throws ApplyException
    {
        if (this.getChildExpressionId() == null) return;

        fetchRawChild();

        getExpressionPool().removeExpression(this.getChildExpressionId());

        log.debug
            ("apply() rawChild : \n"+
             ((openwfe.org.engine.expressions.raw.XmlRawExpression)
                  getRawChild()).getXmlBranch());

        //
        // keep applied item

        this.appliedItem = (InFlowWorkItem)wi.clone();

        //
        // schedule self

        this.tab = lookupAttribute(A_TAB, wi);

        if (this.tab == null) 
        {
            throw new ApplyException
                ("Missing attribute '"+A_TAB+"'");
        }

        this.cronId = this.reschedule(getExpressionPool().getScheduler());

        //
        // change environment for self

        determineNameAndLevel(wi);

        Environment env = null;

        if (this.level.equals(""))
            env = fetchEnvironment();
        else if (this.level.equals("/"))
            env = getExpressionPool().fetchRootEnvironment(this);
        else if (this.level.equals("//"))
            env = getExpressionPool().fetchEngineEnvironment();

        this.getRawChild().setEnvironmentId(env.getId());

        log.debug("apply() set rawChild.envId to "+env.getId());

        // 
        // determine variable name

        String variableName = null;

        if (this.name == null)
            variableName = this.level + V_CRON + cronId.toString();
        else
            variableName = this.level + V_CRON + name;

        log.debug("apply() variableName >"+variableName+"<");

        // 
        // is there a previous cron with the same name ?

        final Object o = this.lookupVariable(variableName);

        if (o != null && (o instanceof CronExpression))
            //
            // seems so
        {
            log.debug
                ("apply() have to unschedule previous cron '"+variableName+"'");

            final CronExpression ce = (CronExpression)o;

            getExpressionPool().getScheduler().unschedule(ce.getCronId());
        }

        //
        // bind self in environment

        this.bindVariable(variableName, this);

        //this.storeItself();
            //
            // not necessary : stored self in environment

        //
        // apply to parent

        applyToParent(wi);
    }

    /**
     * This method is never used, so it's empty.
     */
    public void reply (final InFlowWorkItem wi)
        throws ReplyException
    {
        // nada
    }

    //
    // STATIC METHODS

}
