/*
 * Copyright (c) 2001-2006, John Mettraux, OpenWFE.org
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * . Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.  
 * 
 * . Redistributions in binary form must reproduce the above copyright notice, 
 *   this list of conditions and the following disclaimer in the documentation 
 *   and/or other materials provided with the distribution.
 * 
 * . Neither the name of the "OpenWFE" nor the names of its contributors may be
 *   used to endorse or promote products derived from this software without
 *   specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: SleepExpression.java 2575 2006-05-08 11:52:39Z jmettraux $
 */

//
// SleepExpression.java
//
// jmettraux@openwfe.org
//
// generated with 
// jtmpl 1.0.04 31.10.2002 John Mettraux (jmettraux@openwfe.org)
//

package openwfe.org.engine.expressions;

import openwfe.org.OpenWfeException;
import openwfe.org.time.Time;
import openwfe.org.time.Scheduler;
import openwfe.org.time.Schedulable;
import openwfe.org.engine.workitem.InFlowWorkItem;


/**
 * This allows allows a process to sleep for a period of time or to sleep 
 * until a certain moment.
 *
 * <p><font size=2>CVS Info :
 * <br>$Author: jmettraux $
 * <br>$Date: 2006-05-08 13:52:39 +0200 (Mon, 08 May 2006) $
 * <br>$Id: SleepExpression.java 2575 2006-05-08 11:52:39Z jmettraux $ </font>
 *
 * @author john.mettraux@openwfe.org
 */
public class SleepExpression

    extends ZeroChildExpression

    implements Schedulable

{

    private final static org.apache.log4j.Logger log = org.apache.log4j.Logger
        .getLogger(SleepExpression.class.getName());

    //
    // CONSTANTS

    /**
     * When this attribute is present, its value is used to determine for how
     * long the expression should make its process sleep.
     * 'for' takes precedence over 'until'.
     */
    public final static String FOR = "for";

    /**
     * As 'for' was used to indicate how long the process had to sleep, this 
     * attribute tells at which point the flow should wake up.
     */
    public final static String UNTIL = "until";

    //
    // FIELDS

    private InFlowWorkItem sleepingWorkitem = null;
    private long awakeningTime = -1;

    //
    // CONSTRUCTORS

    //
    // BEAN METHODS

    /**
     * Returns the workitem that is sleeping here.
     */
    public InFlowWorkItem getSleepingWorkitem ()
    {
        return this.sleepingWorkitem;
    }

    /**
     * Returns the point in time (expressed as a long time since epoch) when
     * the sleep should terminate.
     * If -1 is returned, it means the sleep has not begun.
     */
    public long getAwakeningTime ()
    {
        return this.awakeningTime;
    }

    public void setSleepingWorkitem (final InFlowWorkItem wi)
    {
        this.sleepingWorkitem = wi;
    }

    public void setAwakeningTime (final long l)
    {
        this.awakeningTime = l;
    }

    //
    // METHODS

    public void trigger (final Object[] params)
        //throws OpenWfeException
    {
        if (this.sleepingWorkitem == null) return;
            // not applied
        
        //if (System.currentTimeMillis() < this.awakeningTime) return;
            // sleep is not over

        // 
        // sleep time is over

        try
        {
            applyToParent(this.sleepingWorkitem);
        }
        catch (final ApplyException e)
        {
            log.warn("trigger() couldn't reply to parent expression !!!", e);
        }
    }

    public Long reschedule (final Scheduler s)
    {
        getExpressionPool().scheduleAt(this.awakeningTime, this.getId());

        return new Long(-1);
    }

    public void apply (final InFlowWorkItem wi)
        throws ApplyException
    {
        final String sFor = lookupAttribute(FOR, wi);
        final String sUntil = lookupAttribute(UNTIL, wi);

        log.debug("apply() sFor   >"+sFor+"<");
        log.debug("apply() sUntil >"+sUntil+"<");

        if (sFor != null)
            // 'for' takes precedence over 'until'
        {
            //
            // FOR

            final long duration = Time.parseTimeString(sFor);

            this.awakeningTime = System.currentTimeMillis() + duration;
        }
        else
        {
            if (sUntil == null)
            {
                throwSleepException
                    (wi, "no 'for' or 'until' found, no sleep.");

                return;
            }

            //
            // UNTIL

            try
            {
                //this.awakeningTime = Time.fromIsoDate(sUntil);
                this.awakeningTime = Time.parseDate(sUntil).getTime();
            }
            catch (final Throwable t)
            {
                throwSleepException
                    (wi, "Couldn't parse '"+sUntil+"' : "+t);

                return;
            }
        }

        //
        // do sleep

        this.sleepingWorkitem = wi;

        this.storeItself();

        log.debug
            ("apply() now is "+Time.toIsoDate());
        log.debug
            ("apply() entered sleep 'til "+Time.toIsoDate(this.awakeningTime));

        //getExpressionPool().scheduleAt(this.awakeningTime, this.getId());
        reschedule(null);
    }

    /**
     * This method is used when this sleep cannot handle the given 'for' or 
     * 'until' value.
     * It has to be followed by 'return;'. It logs both in the wi and in the
     * history and also in the engine log.
     */
    protected void throwSleepException 
        (final InFlowWorkItem wi, final String message)
    throws
        ApplyException
    {
        wi.addHistoryItem
            (this.getId(), 
             this.getClass().getName(), 
             message);

        historyLog(wi, "-err", null, message);

        log.debug(message);

        applyToParent(wi);
    }
    
}
