/*
 * Copyright (c) 2005, John Mettraux, OpenWFE.org
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * . Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.  
 * 
 * . Redistributions in binary form must reproduce the above copyright notice, 
 *   this list of conditions and the following disclaimer in the documentation 
 *   and/or other materials provided with the distribution.
 * 
 * . Neither the name of the "OpenWFE" nor the names of its contributors may be
 *   used to endorse or promote products derived from this software without
 *   specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: VariableMap.java 2161 2005-10-11 18:52:38Z jmettraux $
 */

//
// VariableMap.java
//
// john.mettraux@openwfe.org
//
// generated with 
// jtmpl 1.1.01 2004/05/19 (john.mettraux@openwfe.org)
//

package openwfe.org.engine.expressions;

import openwfe.org.engine.workitem.Attribute;
import openwfe.org.engine.workitem.CollectionAttribute;
import openwfe.org.engine.workitem.InFlowWorkItem;


/**
 * A wrapper around an expression to confer it the Map methods for
 * its variables. Only the Map get() method is implemented.
 * This class is intended to use for variable substitution in flow
 * definition tag attributes.
 *
 * <p><font size=2>CVS Info :
 * <br>$Author: jmettraux $
 * <br>$Id: VariableMap.java 2161 2005-10-11 18:52:38Z jmettraux $ </font>
 *
 * @author john.mettraux@openwfe.org
 */
public class VariableMap

    extends java.util.AbstractMap

{

    private final static org.apache.log4j.Logger log = org.apache.log4j.Logger
        .getLogger(VariableMap.class.getName());

    //
    // CONSTANTS & co

    /**
     * A lookup for ${field:toto} will trigger a search
     * in the workitem instead of in the variable scope of the expression.
     */
    public final static String FIELD_PREFIX
        = "field:";

    /**
     * A lookup for ${f:toto} will trigger a search
     * in the workitem instead of in the variable scope of the expression.
     */
    public final static String FIELD_PREFIX_2
        = "f:";

    public final static String VAR_PREFIX
        = "v:";
    public final static String VAR_PREFIX_2
        = "var:";
    public final static String VAR_PREFIX_3
        = "variable:";

    /**
     * A lookup for ${fv:something} will at first search
     * for a field named 'something' then, if not found, will look for
     * a variable named 'something'.
     */
    public final static String FV_PREFIX
        = "fv:";

    /**
     * A lookup for ${fv:something} will at first search
     * for a variable named 'something' then, if not found, will look for
     * a field named 'something'.
     */
    public final static String VF_PREFIX
        = "vf:";

    /**
     * ${call:now()} will insert the current time : this "call:" prefix
     * triggers a function call.
     */
    public final static String CALL_PREFIX
        = "call:";

    /**
     * ${c:fieldCount()} is a short version of ${call:fieldCount}.
     */
    public final static String CALL_PREFIX_2
        = "c:";

    /**
     * This prefix allows for insertion of java constants in the workflow
     * definitions.
     * ${const:java.io.File.separator} will, for example, return "/" or "\\" 
     * (windows platform).
     */
    public final static String CONST_PREFIX
        = "const:";

    //
    // FIELDS

    private java.util.Map map = null;
    private FlowExpression flowExpression = null;
    private InFlowWorkItem workitem = null;

    //
    // CONSTRUCTORS

    /**
     * Builds a VariableMap instance wrapped around the given flowExpression,
     * and around a workitem.
     */
    public VariableMap 
        (final FlowExpression fe, 
         final InFlowWorkItem wi) 
    {
        this(null, fe, wi);
    }

    public VariableMap
        (final java.util.Map m,
         final FlowExpression fe,
         final InFlowWorkItem wi)
    {
        this.map = m;
        this.flowExpression = fe;
        this.workitem = wi;
    }

    //
    // METHODS from java.util.Map

    public java.util.Set entrySet () 
    { 
        return null; 
    }

    public boolean containsKey (final Object o)
    {
        return (this.get(o) != null);
    }

    public Object get (final Object o)
    {
        //log.debug("get() '"+o.toString()+"'");

        final String rawKey = o.toString();

        log.debug("get() '"+rawKey+"'");

        //if (isEscaped(rawKey)) return unEscape(rawKey);


        final String[] spk = determinePrefixAndKey(o);

        final String prefix = spk[0];
        final String key = spk[1];

        log.debug("get() prefix >"+prefix+"<");
        log.debug("get() key    >"+key+"<");

        //
        // only variable lookup
        //
        if (prefix == null) return lookupVariableValue(key);

        //
        // function call
        //
        if (prefix.equals(CALL_PREFIX))
        {
            return ValueUtils.executeFunction
                (this.flowExpression, this.workitem, key);
        }

        //
        // const lookup
        //
        if (prefix.equals(CONST_PREFIX))
        {
            return ValueUtils.lookupConstant
                (this.flowExpression, this.workitem, key);
        }

        //
        // only field lookup
        //
        if (prefix.equals(FIELD_PREFIX)) return lookupFieldValue(key);

        //
        // field then variable lookup
        //
        if (prefix.equals(FV_PREFIX))
        {
            final String fieldValue = lookupFieldValue(key);

            if (fieldValue != null) return fieldValue;

            return lookupVariableValue(key);
        }

        //
        // variable then field lookup
        //
        if (prefix.equals(VF_PREFIX))
        {
            final String varValue = lookupVariableValue(key);

            if (varValue != null) return varValue;

            return lookupFieldValue(key);
        }

        return null;
    }

    //
    // METHODS

    /**
     * Could get useful in case of reuse of the VariableMap
     */
    public void setFlowExpression (final FlowExpression fe)
    {
        this.flowExpression = fe;
    }

    /**
     * Could get useful in case of reuse of the VariableMap
     */
    public void setWorkitem (final InFlowWorkItem wi)
    {
        this.workitem = wi;
    }

    //
    // PRIVATE METHODS

    private String matchesPrefix (final String rawKey, final String prefix)
    {
        if (rawKey.startsWith(prefix))
            return rawKey.substring(prefix.length());

        return null;
    }

    /*
    private boolean isEscaped (final String rawKey)
    {
        if (rawKey.charAt(0) != 'e') return false;
        return (rawKey.indexOf(":") > 0);
    }

    private String unEscape (final String rawKey)
    {
        String s = rawKey.substring(1);

        if (s.charAt(0) == ':') s = s.substring(1);

        return "${"+s+"}";
    }
    */

    private String[] determinePrefixAndKey (final Object o)
    {
        final String rawKey = o.toString();

        String key = null;

        //
        // call prefix

        key = matchesPrefix(rawKey, CALL_PREFIX);
        if (key != null) return new String[] { CALL_PREFIX, key };

        key = matchesPrefix(rawKey, CALL_PREFIX_2);
        if (key != null) return new String[] { CALL_PREFIX, key };

        //
        // field prefix

        key = matchesPrefix(rawKey, FIELD_PREFIX);
        if (key != null) return new String[] { FIELD_PREFIX, key };

        key = matchesPrefix(rawKey, FIELD_PREFIX_2);
        if (key != null) return new String[] { FIELD_PREFIX, key };
            // return FIELD_PREFIX as key and not FIELD_PREFIX_2
            // makes it easier afterwards !

        //
        // const prefix

        key = matchesPrefix(rawKey, CONST_PREFIX);
        if (key != null) return new String[] { CONST_PREFIX, key };
        
        //
        // other

        key = matchesPrefix(rawKey, FV_PREFIX);
        if (key != null) return new String[] { FV_PREFIX, key };

        key = matchesPrefix(rawKey, VF_PREFIX);
        if (key != null) return new String[] { VF_PREFIX, key };

        //
        // variable prefix

        key = matchesPrefix(rawKey, VAR_PREFIX);
        if (key != null) return new String[] { null, key };

        key = matchesPrefix(rawKey, VAR_PREFIX_2);
        if (key != null) return new String[] { null, key };

        key = matchesPrefix(rawKey, VAR_PREFIX_3);
        if (key != null) return new String[] { null, key };

        //
        // variable (no prefix)

        return new String[] { null, rawKey };
    }

    private String lookupFieldValue (final String key)
    {
        if (this.workitem == null) return null;

        final Attribute a = CollectionAttribute
            .lookupAttribute(key, this.workitem.getAttributes());

        if (a == null) return null;

        return a.toString();
    }

    private String lookupVariableValue (final String key)
    {
        log.debug("lookupVariableValue() key is >"+key+"<");

        Object value = null;

        if (this.map != null)
            value = this.map.get(key);

        if (value != null) return value.toString();

        if (this.flowExpression != null)
            value = this.flowExpression.lookupVariable(key);

        if (value != null) return value.toString();

        return null;
    }

    //
    // STATIC METHODS

}
