/*
 * Copyright (c) 2005, John Mettraux, OpenWFE.org
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * . Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.  
 * 
 * . Redistributions in binary form must reproduce the above copyright notice, 
 *   this list of conditions and the following disclaimer in the documentation 
 *   and/or other materials provided with the distribution.
 * 
 * . Neither the name of the "OpenWFE" nor the names of its contributors may be
 *   used to endorse or promote products derived from this software without
 *   specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: InMemoryExpressionPool.java 2321 2006-02-03 20:59:14Z jmettraux $
 */

//
// InMemoryExpressionPool.java
//
// jmettraux@openwfe.org
//
// generated with 
// jtmpl 1.0.04 20.11.2001 John Mettraux (jmettraux@openwfe.org)
//

package openwfe.org.engine.impl.expool;

import openwfe.org.Application;
import openwfe.org.ReflectionUtils;
import openwfe.org.ServiceException;
import openwfe.org.OpenWfeException;
import openwfe.org.ApplicationContext;
import openwfe.org.xml.XmlUtils;
import openwfe.org.time.Time;
import openwfe.org.engine.Definitions;
import openwfe.org.engine.control.auth.ControlPermission;
import openwfe.org.engine.expool.PoolException;
import openwfe.org.engine.expool.ExpressionPool;
import openwfe.org.engine.expool.ExpressionStore;
import openwfe.org.engine.history.History;
import openwfe.org.engine.workitem.InFlowWorkItem;
import openwfe.org.engine.dispatch.DispatchingException;
import openwfe.org.engine.expressions.EngineEnvironmentId;
import openwfe.org.engine.expressions.FlowExpression;
import openwfe.org.engine.expressions.WhenExpression;
import openwfe.org.engine.expressions.ApplyException;
import openwfe.org.engine.expressions.ReplyException;
import openwfe.org.engine.expressions.FlowExpressionId;
import openwfe.org.engine.expressions.Environment;
import openwfe.org.engine.expressions.ExpressionWithTimeOut;
import openwfe.org.engine.expressions.CompositeFlowExpression;
import openwfe.org.engine.participants.Participant;
import openwfe.org.engine.participants.ParticipantMap;


/**
 * The pool of expressions : it stores workflow instances as expression bits.
 * In fact, it merely orchestrates this, the real storage work is done
 * by ExpressionStore implementations.
 *
 * <p><font size=2>CVS Info :
 * <br>$Author: jmettraux $
 * <br>$Date: 2006-02-03 21:59:14 +0100 (Fri, 03 Feb 2006) $
 * <br>$Id: InMemoryExpressionPool.java 2321 2006-02-03 20:59:14Z jmettraux $ </font>
 *
 * @author jmettraux@openwfe.org
 */
public class InMemoryExpressionPool

    extends SimpleExpressionPool

{

    private final static org.apache.log4j.Logger log = org.apache.log4j.Logger
        .getLogger(InMemoryExpressionPool.class.getName());

    //
    // INNER CLASSES

    //
    // CONSTANTS

    //
    // FIELDS

    private java.util.Map expressionMap = null;

    //
    // CONSTRUCTORS

    public void init 
        (final String serviceName, 
         final ApplicationContext context, 
         final java.util.Map serviceParams)
    throws 
        ServiceException
    {
        super.init(serviceName, context, serviceParams);

        try
        {
            reload();
        }
        catch (PoolException pe)
        {
            log.warn
                ("init() reload() failed, "+
                 "resuming with an empty expression pool. No flow instance "+
                 "could be rebuilt", pe);
        }
    }

    //
    // METHODS

    /**
     * Usually called by WorkflowInstanceBuilders to add a freshly created
     * expression to the pool.
     */
    public void add (final FlowExpression fe)
        throws PoolException
    {
        log.debug("add() "+fe.getId());

        fe.setApplicationContext(getContext());
        this.expressionMap.put(fe.getId(), fe);

        //doStoreExpression(fe);
        if (getStore() != null)
            getStore().storeExpression(fe);
    }

    /**
     * Returns the count of expressions stored here.
     */
    public int size ()
    {
        return this.expressionMap.size();
    }

    /**
     * Stores the flow expression (as it may have changed).
     */
    public void update (final FlowExpression fe)
        throws PoolException
    {
        add(fe);
    }

    public Environment fetchEngineEnvironment ()
    {
        Environment result = (Environment)fetch(EngineEnvironmentId.ID);

        if (result == null)
        {
            if (getStore() == null)
            {
                final Environment ee = new Environment();
                ee.setId(EngineEnvironmentId.ID);
                ee.setApplicationContext(this.getContext());
                this.expressionMap.put(ee.getId(), ee);

                return ee;
            }

            result = getStore().loadEngineEnvironment();
            try
            {
                this.update(result);
            }
            catch (final PoolException pe)
            {
                log.warn("Failed to store Engine Environment.", pe);
            }
        }

        return result;
    }

    /**
     * Retrieves an expression from the pool.
     * If the expression cannot be retrieved from the pool, null will be
     * returned.
     */
    public FlowExpression fetch (final FlowExpressionId fei)
    {
        log.debug("fetch() for "+fei);

        return (FlowExpression)this.expressionMap.get(fei);
    }

    /**
     * Releases a flow expression.
     * This method is called by the replyToFather method, as the
     * expression tree gets diminished.
     */
    public void removeExpression (final FlowExpression fe)
    {
        //log.debug("removeExpression() "+fe.getId());

        this.expressionMap.remove(fe.getId());

        super.removeExpression(fe);
    }

    /**
     * Will return true if there is still a FlowExpression stored for the
     * given FlowExpressionId.
     * Might get useful for a UI that has to determine if a workitem belongs
     * to an active or a dead branch of a workflow instance.
     */
    public boolean isActive (final FlowExpressionId fei)
    {
        return (this.expressionMap.get(fei) != null);
    }

    /**
     * A fairly simple implementation of a contentIterator as all the 
     * expressions are held in memory.
     */
    public java.util.Iterator contentIterator (final Class assignClass)
    {
        final java.util.List result = 
            new java.util.ArrayList(this.expressionMap.size());

        final java.util.Iterator it = this.expressionMap.values().iterator();
        while (it.hasNext())
        {
            final FlowExpression fe = (FlowExpression)it.next();

            //if (assignClass != null &&
            //    ! assignClass.isAssignableFrom(fe.getClass()))
            if ( ! ExpoolUtils.isAssignableFromClass(fe, assignClass))
            {
                continue;
            }

            result.add(fe);

            //log.debug("contentIterator() took "+fe.getId());
        }

        log.debug
            ("contentIterator() found "+result.size()+" matching expressions");

        return result.iterator();
    }

    /**
     * Fills in the memory storage with what comes from the physical store.
     * If there is no physical store set, this method will return 
     * immediately.
     */
    protected void reload ()
        throws PoolException
    {
        this.expressionMap = new java.util.HashMap(10000);

        if (getStore() != null) 
        {
            getStore().loadAll(this.expressionMap);
        }
        else
        {
            log.info
                ("reload() no store attached to "+this.getClass().getName());
        }

        log.info
            ("reload() done. Reloaded "+this.expressionMap.size()+
             " expressions.");
    }

    //
    // STATUS

    public org.jdom.Element getStatus ()
    {
        org.jdom.Element result = new org.jdom.Element(getName());

        result.addContent(XmlUtils.getClassElt(this));
        result.addContent(XmlUtils.getRevisionElt("$Id: InMemoryExpressionPool.java 2321 2006-02-03 20:59:14Z jmettraux $"));

        //
        // pooled expression count
        
        org.jdom.Element pecElt = new org.jdom.Element("pooledExpressionCount");
        pecElt.addContent(""+this.expressionMap.size());

        result.addContent(pecElt);

        //
        // debug
        //
        //final org.jdom.Element expsElt = new org.jdom.Element("expressions");
        //final java.util.Iterator it = this.expressionMap.keySet().iterator();
        //while (it.hasNext())
        //{
        //    final FlowExpressionId fei = (FlowExpressionId)it.next();
        //    final org.jdom.Element expElt = new org.jdom.Element("exp");
        //    expElt.addContent(new org.jdom.CDATA(fei.toString()));
        //    expsElt.addContent(expElt);
        //}
        //result.addContent(expsElt);

        return result;
    }

}
