/*
 * Copyright (c) 2005, John Mettraux, OpenWFE.org
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * . Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.  
 * 
 * . Redistributions in binary form must reproduce the above copyright notice, 
 *   this list of conditions and the following disclaimer in the documentation 
 *   and/or other materials provided with the distribution.
 * 
 * . Neither the name of the "OpenWFE" nor the names of its contributors may be
 *   used to endorse or promote products derived from this software without
 *   specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: FileListener.java 2414 2006-03-10 22:15:03Z jmettraux $
 */

//
// FileListener.java
//
// jmettraux@openwfe.org
//
// generated with 
// jtmpl 1.0.04 20.11.2001 John Mettraux (jmettraux@openwfe.org)
//

package openwfe.org.engine.impl.listen;

import openwfe.org.Utils;
import openwfe.org.MapUtils;
import openwfe.org.Application;
import openwfe.org.ServiceException;
import openwfe.org.ApplicationContext;
import openwfe.org.engine.listen.WorkItemListener;
import openwfe.org.engine.workitem.WorkItemCoder;


/**
 * A daemon that listens for file appearing in a given directory (not very much 
 * used as of now, but could still interest some people).
 * A work item is extracted from each of these files.
 *
 * <p><font size=2>CVS Info :
 * <br>$Author: jmettraux $
 * <br>$Date: 2006-03-10 23:15:03 +0100 (Fri, 10 Mar 2006) $
 * <br>$Id: FileListener.java 2414 2006-03-10 22:15:03Z jmettraux $ </font>
 *
 * @author john.mettraux@openwfe.org
 */
public class FileListener

    extends WorkItemListener

{

    private final static org.apache.log4j.Logger log = org.apache.log4j.Logger
        .getLogger(FileListener.class.getName());

    //
    // CONSTANTS (definitions)

    /**
     * Use the 'directory' parameter to tell this listener where
     * it can find incoming workitems.
     */
    public final static String P_DIRECTORY
        = "directory";

    /**
     * Use the 'frequency' parameter to tell this listener how frequently
     * it should poll for new workitem files.
     * The default time is '10s'.
     */
    public final static String P_FREQUENCY
        = "frequency";

    private final static String DEFAULT_FREQUENCY
        = "10s"; // every 10 seconds

    //
    // FIELDS

    private String directory = null;

    private java.util.TimerTask listenTask = null;

    //
    // CONSTRUCTORS

    public void init 
        (final String serviceName, 
         final ApplicationContext context, 
         final java.util.Map serviceParams)
    throws 
        ServiceException
    {
        // using the super init method is mandatory as it instatiates the
        // work item consumer that the listener will use.

        super.init(serviceName, context, serviceParams);

        //
        // directory

        this.directory = 
            MapUtils.getAsString(serviceParams, P_DIRECTORY, "in/");

        if ( ! this.directory.startsWith(java.io.File.separator))
        {
            this.directory = 
                getContext().getApplicationDirectory()+ 
                this.directory;
        }

        log.info("Input directory set to "+this.directory);

        //
        // frequency

        long frequency = MapUtils
            .getAsTime(serviceParams, P_FREQUENCY, DEFAULT_FREQUENCY);

        //
        // start the service !

        this.listenTask = new java.util.TimerTask()
        {
             public void run ()
             {
                 load();
             }
        };

        Application.getTimer().schedule(this.listenTask, 10, frequency);
    }

    //
    // METHODS

    /**
     * Takes care of stopping the polling.
     */
    public void stop ()
        throws ServiceException
    {
        super.stop();

        this.listenTask.cancel();

        log.info("'"+getName()+"' Stopped.");
    }

    protected void load ()
    {
        try
        {
            java.io.File dir = new java.io.File(this.directory);

            java.io.File[] files = dir.listFiles();

            for (int i=0; i<files.length; i++)
            {
                loadFile(files[i]);
            }
        }
        catch (final Throwable t)
        {
            log.warn("load() failure", t);
        }
    }

    protected void loadFile (java.io.File f)
    {
        if (f.isDirectory()) return;

        try
        {
            final java.io.InputStream is = new java.io.FileInputStream(f);

            final WorkItemCoder coder = 
                (WorkItemCoder)determineCoderAndLength(is)[0];

            getConsumer().use(coder.decode(is, getContext(), getParams()));
                // doesn't care about the result of the consumer...

            f.delete();
        }
        catch (Exception e)
        {
            log.info("Rejecting item "+f.getPath(), e);

            Utils.move(f, this.directory+"rejected/"+f.getName());
        }
    }

}
