/*
 * Copyright (c) 2006, John Mettraux, OpenWFE.org
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * . Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.  
 * 
 * . Redistributions in binary form must reproduce the above copyright notice, 
 *   this list of conditions and the following disclaimer in the documentation 
 *   and/or other materials provided with the distribution.
 * 
 * . Neither the name of the "OpenWFE" nor the names of its contributors may be
 *   used to endorse or promote products derived from this software without
 *   specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: XmlParticipantMapFactory.java 2385 2006-03-04 09:30:22Z jmettraux $
 */

//
// XmlParticipantMapFactory.java
//
// jmettraux@openwfe.org
//
// generated with 
// jtmpl 1.0.04 20.11.2001 John Mettraux (jmettraux@openwfe.org)
//

package openwfe.org.engine.impl.participants;

import openwfe.org.Utils;
import openwfe.org.MapUtils;
import openwfe.org.Parameters;
import openwfe.org.AbstractService;
import openwfe.org.ApplicationContext;
import openwfe.org.ServiceException;
import openwfe.org.FactoryException;
import openwfe.org.xconf.XconfBuilder;
import openwfe.org.xconf.XconfElementBuilder;
import openwfe.org.engine.participants.Participant;
import openwfe.org.engine.participants.RefParticipant;
import openwfe.org.engine.participants.LeafParticipant;
import openwfe.org.engine.participants.CompositeParticipant;
import openwfe.org.engine.participants.ParticipantMap;
import openwfe.org.engine.participants.ParticipantMapFactory;


/**
 * A factory for participant enumerated in an XML document accessible locally
 * or through a URL.
 *
 * <p><font size=2>CVS Info :
 * <br>$Author: jmettraux $
 * <br>$Date: 2006-03-04 10:30:22 +0100 (Sat, 04 Mar 2006) $
 * <br>$Id: XmlParticipantMapFactory.java 2385 2006-03-04 09:30:22Z jmettraux $ </font>
 *
 * @author john.mettraux@openwfe.org
 */
public class XmlParticipantMapFactory

    extends AbstractParticipantMapFactory

    implements XconfElementBuilder

{

    private final static org.apache.log4j.Logger log = org.apache.log4j.Logger
        .getLogger(XmlParticipantMapFactory.class.getName());

    //
    // CONSTANTS (definitions)

    private final static String E_PARTICIPANT
        = "participant";
    private final static String A_NAME
        = "name";
    private final static String A_REGEX
        = "regex";
    private final static String A_REF
        = "ref";
    private final static String A_CLASS
        = "class";

    /**
     * The parameter named 'participantMapFile' indicates to this
     * participant map factory where it can locate its XML
     * representation.
     * The value of this parameter may be simply the path to a local
     * file or a complete URL. A URL could be shared by many OpenWFE
     * components distributed across an organization and its content
     * be fetched from the intranet server (of course, you can map
     * this scenario to multiple [cross] organizations).
     */
    public final static String P_PARTICIPANT_MAP_FILE
        = "participantMapFile";

    //
    // FIELDS

    private XconfBuilder builder = null;

    private java.util.List participants = null;

    //
    // CONSTRUCTORS

    /**
     * A no parameter constructor for using this class as an OpenWFE service.
     */
    public XmlParticipantMapFactory ()
    {
        super();

        this.participants = new java.util.ArrayList(35);
    }

    /**
     * This constructor should be used when XmlParticipantMapFactory is 
     * not used as an OpenWFE service; you can thus directly give this
     * constructor the URL of its XML source.
     */
    public XmlParticipantMapFactory (final String pmapUrl)
        throws ServiceException
    {
        this();

        this.builder = new XconfBuilder(pmapUrl, this);
    }

    /**
     * Initialization method of this OpenWFE service.
     */
    public void init 
        (final String serviceName, 
         final ApplicationContext context, 
         final java.util.Map serviceParams)
    throws 
        ServiceException
    {
        super.init(serviceName, context, serviceParams);

        String url = MapUtils.getAsString
            (serviceParams, 
             P_PARTICIPANT_MAP_FILE, 
             "etc/engine/participant-map.xml");

        url = Utils.expandUrl(context.getApplicationDirectory(), url);

        log.debug("init() url is "+url);

        this.builder = new XconfBuilder(url, this);
    }

    //
    // METHODS from XconfElementBuilder

    /**
     * Does nothing.
     * (Clears the participant map. This method should and is only called by
     * the XconfBuilder helper class)
     */
    public void clearConfig ()
    {
        this.participants.clear();
    }

    /**
     * XconfBuilder calls this method when parsing a config file. It should
     * only be used by XconfBuilder.
     */
    public void parseElement 
        (final XconfBuilder builder, final org.jdom.Element elt)
    throws 
        ServiceException
    {
        buildParticipant(elt);
    }

    //
    // METHODS from ParticipantMapFactory

    /**
     * Builds a list of participant that will be turned into 
     * a participant map (by the participant map implementation).
     */
    public java.util.List createParticipants (final ParticipantMap m)
        throws FactoryException
    {
        try
        {
            this.builder.buildConfig();
        }
        catch (final Throwable t)
        {
            throw new FactoryException
                ("failed to create participants", t);
        }

        return this.participants;
    }

    /**
     * Will return true if the 'participant source' behind the factory changed,
     * the participant map may then call createParticipants() to 
     * get a complete and fresh list of participants.
     */
    public boolean hasSourceChanged ()
        throws FactoryException
    {
        return this.builder.hasSourceBeenModified();
    }

    //
    // METHODS from Service

    //
    // METHODS

    /**
     * Turns an xml element into the participant it represents
     */
    protected Participant buildParticipant (final org.jdom.Element elt)
    {
        if ( ! elt.getName().equals(E_PARTICIPANT)) return null;

        String regex = elt.getAttributeValue(A_REGEX);
        if (regex == null) regex = elt.getAttributeValue(A_NAME);

        final java.util.Map params = Parameters.extractParamsAndAttributes(elt);

        log.debug("buildParticipant() building '"+regex+"'");

        if (elt.getChild(E_PARTICIPANT) != null)
        {
            return buildCompositeParticipant(elt, regex, params);
        }

        //
        // determine participant class

        String className = null;

        final String ref = elt.getAttributeValue(A_REF);

        if (ref != null) 
            className = RefParticipant.class.getName();

        final String specifiedClassName = (String)params.get(A_CLASS);

        if (specifiedClassName != null)
            className = specifiedClassName;

        //
        // instantiate participant

        Participant p = null;
        try
        {
            Class participantClass = LeafParticipant.class;

            if (className != null)
                participantClass = Class.forName(className);

            p = (Participant)participantClass.newInstance();
        }
        catch (final Throwable t)
        {
            log.warn
                ("buildParticipant() cannot build from class "+className+
                 ". falling back to "+LeafParticipant.class.getName());
            log.debug
                ("buildParticipant() cannot build from class "+className+
                 ". falling back to "+LeafParticipant.class.getName(),
                 t);

            p = new LeafParticipant();
        }

        log.debug
            ("buildParticipant() of class '"+p.getClass().getName()+"'");

        p.init(regex, params);

        this.participants.add(p);

        return p;
    }

    /**
     * Parses a composite participant
     */
    protected CompositeParticipant buildCompositeParticipant 
        (final org.jdom.Element elt,
         final String regex,
         final java.util.Map params)
    {
        final CompositeParticipant cp = new CompositeParticipant();
        cp.init(regex, params);

        final java.util.Iterator it = 
            elt.getChildren(E_PARTICIPANT).iterator();

        while (it.hasNext())
            cp.add(buildParticipant((org.jdom.Element)it.next()));

        this.participants.add(cp);

        return cp;
    }

    //
    // STATUS

    /**
     * Returns the current status of this participant map.
     */
    public org.jdom.Element getStatus ()
    {
        String name = this.getName();
        if (name == null) name = this.getClass().getName();

        final org.jdom.Element result = new org.jdom.Element(name);

        result.setAttribute
            ("source-url", 
             this.builder.getSourceUrl().toString());

        return result;
    }

}
