/*
 * Copyright (c) 2005, John Mettraux, OpenWFE.org
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * . Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.  
 * 
 * . Redistributions in binary form must reproduce the above copyright notice, 
 *   this list of conditions and the following disclaimer in the documentation 
 *   and/or other materials provided with the distribution.
 * 
 * . Neither the name of the "OpenWFE" nor the names of its contributors may be
 *   used to endorse or promote products derived from this software without
 *   specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: Filter.java 2030 2005-08-13 08:59:16Z jmettraux $
 */

//
// Filter.java
//
// jmettraux@openwfe.org
//
// generated with 
// jtmpl 1.0.04 31.10.2002 John Mettraux (jmettraux@openwfe.org)
//

package openwfe.org.engine.participants;

import openwfe.org.engine.workitem.InFlowWorkItem;
import openwfe.org.engine.workitem.Attribute;
import openwfe.org.engine.workitem.StringAttribute;
import openwfe.org.engine.workitem.StringMapAttribute;


/**
 * Participants may see the workitems they receive through a filter.
 *
 * <p><font size=2>CVS Info :
 * <br>$Author: jmettraux $
 * <br>$Date: 2005-08-13 10:59:16 +0200 (Sat, 13 Aug 2005) $
 * <br>$Id: Filter.java 2030 2005-08-13 08:59:16Z jmettraux $ </font>
 *
 * @author jmettraux@openwfe.org
 */
public class Filter

    implements java.io.Serializable

{

    static final long serialVersionUID = -1652507046784415271L;


    private final static org.apache.log4j.Logger log = org.apache.log4j.Logger
        .getLogger(Filter.class.getName());

    //
    // INNER CLASSES

    private final static Filter noFilter = new NoFilter();

    final static class NoFilter extends Filter
    {
        public NoFilter ()
        {
            this.setName(this.getClass().getName());
            this.setType(Filter.TYPE_OPEN);
            this.setAddAllowed(true);
            this.setRemoveAllowed(true);
            this.setEntries(new java.util.ArrayList(0));
        }
    }

    //
    // CONSTANTS

    /**
     * (0)
     */
    public final static int TYPE_OPEN = 0;

    /**
     * (1)
     */
    public final static int TYPE_CLOSED = 1;

    //
    // FIELDS

    private String name = null;
    private int type = -1;
    private boolean addAllowed = true;
    private boolean removeAllowed = false;
    private java.util.List entries = null;

    //
    // CONSTRUCTORS

    //
    // BEAN METHODS

    public String getName () { return this.name; }
    public int getType () { return this.type; }
    public boolean isAddAllowed () { return this.addAllowed; }
    public boolean isRemoveAllowed () { return this.removeAllowed; }
    public java.util.List getEntries () { return this.entries; }

    public void setName (String s) { this.name = s; }
    public void setType (int i) { this.type = i; }
    public void setAddAllowed (boolean b) { this.addAllowed = b; }
    public void setRemoveAllowed (boolean b) { this.removeAllowed = b; }
    public void setEntries (java.util.List l) { this.entries = l; }

    //
    // METHODS

    private boolean hasPermission 
        (final String fieldName, final char permissionChar)
    {
        log.debug("hasPermission() fieldName >"+fieldName+"<");

        java.util.Iterator it = this.entries.iterator();
        while (it.hasNext())
        {
            FilterEntry entry = (FilterEntry)it.next();

            if (entry.getFieldRegex() == null)
            {
                log.warn
                    ("hasPermission() Filter named '"+getName()+
                     "' has 'null' regex. Skipped");

                continue;
            }

            log.debug
                ("hasPermission() entry.fieldRegex >"+
                 entry.getFieldRegex()+"<");

            if (fieldName.matches(entry.getFieldRegex()))
            {
                final boolean result = 
                    (entry.getPermissions().indexOf(permissionChar) > -1);

                log.debug("hasPermission() returning "+result+" result");

                return result;
            }
        }

        log.debug("hasPermission() returning false");

        return false;
    }

    private String getPermissions (final String fieldName)
    {
        java.util.Iterator it = this.entries.iterator();
        while (it.hasNext())
        {
            final FilterEntry entry = (FilterEntry)it.next();

            if (entry.getFieldRegex() == null) continue;
                // already skipped in hasPermission()

            if (fieldName.matches(entry.getFieldRegex())) 
                return entry.getPermissions().trim();
        }

        return null;
    }

    /*
     * Returns true if the filter has been given instructions about
     * the field named as parameter
     */
    private boolean knowsAbout (String fieldName)
    {
        return 
            (getPermissions(fieldName) != null);
    }

    public boolean allowWrite (String fieldName)
    {
        if (this.type == TYPE_OPEN && ! knowsAbout(fieldName))
            return true;

        return hasPermission(fieldName, 'w');
    }

    public boolean allowRead (String fieldName)
    {
        if (this.type == TYPE_OPEN && ! knowsAbout(fieldName))
            return true;

        return hasPermission(fieldName, 'r');
    }

    public boolean allowRemove (String fieldName)
    {
        if ( ! removeAllowed) return false;

        if (this.type == TYPE_OPEN && ! knowsAbout(fieldName))
        {
            return true;
        }

        return hasPermission(fieldName, 'w');
    }

    /**
     * This method takes a workitem and produces another where all
     * hidden fields have been removed.
     * It ensures that the worklist or whatever listens for the workitem
     * won't see the hidden value.
     */
    public InFlowWorkItem constrain (final InFlowWorkItem workItem)
    {

        log.debug("constrain() filter '"+this.name+"'");

        InFlowWorkItem result = new InFlowWorkItem();

        result.setId(workItem.getId());
        result.setParticipantName(workItem.getParticipantName());
        result.setLastModified(workItem.getLastModified());

        result.setAttributes
            (new StringMapAttribute(workItem.getAttributes().size()));

        result.setHistory(workItem.getHistory());


        java.util.Iterator it = workItem.getAttributes().keySet().iterator();
        while (it.hasNext())
        {
            String attributeName = ((StringAttribute)it.next()).toString();

            log.debug
                ("constrain() considering attribute named >"+attributeName+"<");

            if (this.type == TYPE_OPEN)
            {
                if (hasPermission(attributeName, 'r') ||
                    ( ! "".equals(getPermissions(attributeName))))
                {
                    result.getAttributes().put
                        (attributeName, 
                         (Attribute)workItem.getAttribute(attributeName).clone());
                    log.debug
                        ("constrain() (open) added attribute >"+
                         attributeName+"<");
                }
                else
                {
                    log.debug
                        ("constrain() (open) discarding attribute >"+
                         attributeName+"<");
                }
            }
            else // TYPE_CLOSED
            {
                if (hasPermission(attributeName, 'r'))
                {
                    result.getAttributes().put
                        (attributeName, 
                         (Attribute)workItem.getAttribute(attributeName).clone());
                    log.debug
                        ("constrain() (closed) added attribute >"+
                         attributeName+"<");
                }
                else
                {
                    log.debug
                        ("constrain() (closed) discarding attribute >"+
                         attributeName+"<");
                }
            }
        }

        result.setFilter(this);
        return result;
    }

    /**
     * This method enforces the filter policy : unauthorized changes are
     * cancelled, it is used when a workitem comes back from its filtered 
     * participant. A correct modified workitem is returned.
     */
    public InFlowWorkItem enforce 
        (InFlowWorkItem initialWorkItem, 
         InFlowWorkItem modifiedWorkItem)
    {
        log.debug
            ("enforce() filter '"+this.name+
             "' isAddAllowed ? "+this.addAllowed);
        log.debug
            ("enforce() filter '"+this.name+
             "' isRemoveAllowed ? "+this.removeAllowed);

        /*
         * enforce field type ???????????????????????????????????????????
         */

        java.util.Iterator it = 
            modifiedWorkItem.getAttributes().keySet().iterator();
        while (it.hasNext())
        {
            String attributeName = ((StringAttribute)it.next()).toString();

            log.debug
                ("enforce() filter '"+this.name+
                 "' considering att '"+attributeName+"'");

            //
            // new attribute

            boolean isNewAttribute = 
                ! initialWorkItem.containsAttribute(attributeName);
            
            log.debug
                ("enforce() filter '"+this.name+
                 "' isNewAttribute '"+isNewAttribute+"'");

            if (isNewAttribute && isAddAllowed())
            {
                // add

                initialWorkItem.getAttributes().put
                    (attributeName, 
                     modifiedWorkItem.getAttribute(attributeName));

                log.debug
                    ("enforce() Added new attribute '"+attributeName+"'");

                continue;
            }
            else if (isNewAttribute)
            {
                // don't add

                log.debug
                    ("enforce() Dropped new attribute '"+attributeName+"'");

                continue;
            }

            //
            // modified attribute

            if (modifiedWorkItem.getAttribute(attributeName)
                    .equals(initialWorkItem.getAttribute(attributeName)))
            {
                // same value

                log.debug
                    ("enforce() Same value");

                continue;
            }

            //
            // not the same value

            if ( ! allowRead(attributeName))
            {
                // modified, but should have remained hidden

                log.debug
                    ("enforce() Discarding modification on hidden attribute '"+
                     attributeName+"'");

                continue;
            }

            if ( ! allowWrite(attributeName)) 
            {
                log.debug
                    ("enforce() Discarding modification on attribute '"+
                     attributeName+"'");
                continue;
            }

            // modify value

            initialWorkItem.getAttributes().put
                (attributeName, modifiedWorkItem.getAttribute(attributeName));
        }

        //
        // check if an attribute got removed with no authorization
        
        log.debug
            ("enforce() checking removal of attributes (is it allowed ?)");

        if (this.removeAllowed)
        {
            log.debug
                ("enforce() checking removal of attributes (it is allowed)");

            java.util.List fieldsToRemove = 
                new java.util.ArrayList(initialWorkItem.getAttributes().size());

            it = initialWorkItem.getAttributes().keySet().iterator();
            while (it.hasNext())
            {
                String fieldName = ((StringAttribute)it.next()).toString();

                if ( ! modifiedWorkItem.containsAttribute(fieldName))
                {
                    if (allowWrite(fieldName))
                    {
                        fieldsToRemove.add(fieldName);

                        log.debug("enforce() removing field '"+fieldName+"'");
                    }
                }
            }

            it = fieldsToRemove.iterator();
            while (it.hasNext())
                initialWorkItem.removeAttribute((String)it.next());
        }

        return initialWorkItem;
    }

    //
    // STATIC METHODS

    public static Filter getNoFilter ()
    {
        return noFilter;
    }

}
