/*
 * Copyright (c) 2005, John Mettraux, OpenWFE.org
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * . Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.  
 * 
 * . Redistributions in binary form must reproduce the above copyright notice, 
 *   this list of conditions and the following disclaimer in the documentation 
 *   and/or other materials provided with the distribution.
 * 
 * . Neither the name of the "OpenWFE" nor the names of its contributors may be
 *   used to endorse or promote products derived from this software without
 *   specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: AttributeUtils.java 2336 2006-02-17 19:07:52Z jmettraux $
 */

//
// AttributeUtils.java
//
// john.mettraux@openwfe.org
//
// generated with 
// jtmpl 1.1.00 16.08.2003 John Mettraux (jmettraux@openwfe.org)
//

package openwfe.org.engine.workitem;

import java.lang.reflect.Array;
import java.lang.reflect.Method;

import openwfe.org.ReflectionUtils;
import openwfe.org.ApplicationContext;


/**
 * Some methods for turning a MapAttribute into a Map and 'vice versa'.
 *
 * Following an idea of Richard Jennings
 *
 * <p><font size=2>CVS Info :
 * <br>$Author: jmettraux $
 * <br>$Id: AttributeUtils.java 2336 2006-02-17 19:07:52Z jmettraux $ </font>
 *
 * @author john.mettraux@openwfe.org
 */
public abstract class AttributeUtils
{

    private final static org.apache.log4j.Logger log = org.apache.log4j.Logger
        .getLogger(AttributeUtils.class.getName());

    //
    // CONSTANTS

    /*
     * If this bean is present in a StringMapAttribute, it means that this
     * map is a Bean
     */
    private final static String A_BEAN_CLASSNAME
        = "__bean_classname__";

    //
    // OWFE TO JAVA

    /**
     * Turns an OpenWFE attribute into a java instance
     */
    public static Object owfe2java (final Attribute a)
    {
        if (a instanceof AtomicAttribute)
            return ((AtomicAttribute)a).getValue();

        if (a instanceof StringMapAttribute)
        {
            final StringMapAttribute sma = (StringMapAttribute)a;

            final Attribute va = sma.get(A_BEAN_CLASSNAME);

            if ((va != null) && (va instanceof StringAttribute))
                return smap2bean(sma);
        }

        if (a instanceof MapAttribute)
            return map2java((MapAttribute)a);
        if (a instanceof ListAttribute)
            return list2java((ListAttribute)a);
        return "null";
    }

    /*
     * Turns a string map attribute with the '__bean_classname__' attribute
     * set into a bean.
     */
    private static Object smap2bean (final StringMapAttribute sma)
    {
        final String beanClassName = sma.sget(A_BEAN_CLASSNAME);

        final Object bean = ReflectionUtils.buildNewInstance(beanClassName);

        final java.util.Map fields = map2java(sma);

        final java.util.Iterator it = fields.keySet().iterator();
        while (it.hasNext())
        {
            final String fieldName = (String)it.next();

            log.debug("smap2bean() fieldName >"+fieldName+"<");

            if (fieldName.equals(A_BEAN_CLASSNAME)) continue;

            try
            {
                final Class fieldClass = ReflectionUtils
                    .getAttributeClass(bean.getClass(), fieldName);

                log.debug
                    ("smap2bean() fieldClass is >"+fieldClass.getName()+"<");

                Object fieldValue = null;

                if (fieldClass.isArray())
                {
                    log.debug("smap2bean() '"+fieldName+"' is an array.");

                    fieldValue = toArray
                        (fieldClass, (java.util.List)fields.get(fieldName));
                }
                else
                {
                    fieldValue = fields.get(fieldName);
                }

                ReflectionUtils.setAttribute(bean, fieldName, fieldValue);
            }
            catch (final Exception e)
            {
                log.warn
                    ("smap2bean() bean of class '"+beanClassName+
                     "' : failed to set value for field '"+fieldName+
                     "'. Skipped field.");

                log.warn("because of", e);
            }
        }

        return bean;
    }

    /*
     * Turns a list into an array
     */
    private static Object toArray 
        (final Class arrayClass, final java.util.List l)
    throws 
        ClassNotFoundException
    {
        final Class itemClass = ReflectionUtils.getItemClass(arrayClass);

        log.debug("toArray() itemClass is "+itemClass.getName());

        Object result = Array.newInstance(itemClass, l.size());

        for (int i=0; i<l.size(); i++)
            Array.set(result, i, l.get(i));

        return result;
    }

    /**
     * Turns an OpenWFE MapAttribute or StringMapAttribute into 
     * a java HashMap
     */
    public static java.util.Map map2java (final MapAttribute ma)
    {
        java.util.Map result = new java.util.HashMap(ma.size());

        java.util.Iterator it = ma.keySet().iterator();
        while (it.hasNext())
        {
            Attribute k = (Attribute)it.next();
            Attribute v = (Attribute)ma.get(k);

            result.put(owfe2java(k), owfe2java(v));
        }

        return result;
    }

    /**
     * Turns an OpenWFE ListAttribute into a java ArrayList
     */
    public static java.util.List list2java (ListAttribute la)
    {
        java.util.List result = new java.util.ArrayList(la.size());

        java.util.Iterator it = la.iterator();
        while (it.hasNext())
        {
            Attribute v = (Attribute)it.next();
            result.add(owfe2java(v));
        }

        return result;
    }

    /**
     * Turns an OpenWFE Attribute into a list of string (separator defaults
     * to ", *".
     */
    public static java.util.List owfe2list (final Attribute a)
    {
        return owfe2list(a, ", *");
    }

    /**
     * Turns an OpenWFE Attribute into a list of string, you can specify
     * the separator for the final string splitting
     */
    public static java.util.List owfe2list 
        (final Attribute a, final String separator)
    {
        if (a instanceof ListAttribute)
        {
            final ListAttribute la = (ListAttribute)a;

            final java.util.List result = new java.util.ArrayList(la.size());

            final java.util.Iterator it = la.iterator();

            while (it.hasNext())
                result.add((it.next()).toString());

            return result;
        }

        String s = "";
        if (a != null) s = a.toString();

        final String[] ss = s.split(separator);

        final java.util.List result = new java.util.ArrayList(ss.length);

        for (int i=0; i<ss.length; i++) result.add(ss[i]);

        return result;
    }

    //
    // JAVA TO OWFE

    /**
     * Turns an object into an OpenWFE attribute, if the class of the
     * object is 'unknown', a StringAttribute will be issued.
     */
    public static Attribute java2owfe (final Object o)
    {
        if (o == null) return null;

        log.debug("java2owfe() o.class is "+o.getClass().getName());

        if (o instanceof Attribute) return (Attribute)o;

        if (o instanceof String)
            return new StringAttribute((String)o);
        if (o instanceof java.util.List)
            return list2owfe((java.util.List)o);
        if (o instanceof java.util.Map)
            return map2owfe((java.util.Map)o);
        if (o instanceof Integer)
            return new IntegerAttribute((Integer)o);
        if (o instanceof Long)
            return new LongAttribute((Long)o);
        if (o instanceof Double)
            return new DoubleAttribute((Double)o);
        if (o instanceof Boolean)
            return new BooleanAttribute((Boolean)o);
        if (o instanceof org.jdom.Content)
            return new XmlAttribute((org.jdom.Content)o);
        if (o instanceof org.jdom.Document)
            return new XmlAttribute(((org.jdom.Document)o).getRootElement());
        if (o instanceof byte[])
            return Base64Attribute.newBase64Attribute((byte[])o);

        if (o.getClass().isArray())
            return array2owfe(o);

        if (ReflectionUtils.hasNoParamConstructor(o.getClass()))
            return bean2owfe(o);

        //log.debug("java2owfe() default handling...");

        return new StringAttribute(o.toString());
    }

    /*
     * Turns an array into a ListAttribute.
     */
    private static ListAttribute array2owfe (final Object array)
    {
        //log.debug("array2owfe()");

        final ListAttribute result = new ListAttribute(Array.getLength(array));
        //result.setArray(true);

        for (int i=0; i<Array.getLength(array); i++)
            result.add(java2owfe(Array.get(array, i)));

        return result;
    }

    /**
     * Turns a java bean into a StringMapAttribute.
     */
    public static StringMapAttribute bean2owfe (final Object o)
    {
        final StringMapAttribute sma = new StringMapAttribute();

        //
        // class name

        sma.puts(A_BEAN_CLASSNAME, o.getClass().getName());

        //
        // fields

        final java.util.Iterator it = 
            ReflectionUtils.listReadWriteFields(o).iterator();
        while (it.hasNext())
            resolveGet(sma, o, (Method[])it.next());

        return sma;
    }

    /*
     * Applies the get method and turn the result into an OpenWFE attribute.
     */
    private static void resolveGet
        (final StringMapAttribute sma, 
         final Object bean, 
         final Method[] getAndSet)
    {
        try
        {
            final Object value = getAndSet[0].invoke(bean, new Object[] {});
            final String attributeName = getAndSet[0].getName().substring(3);

            final Attribute aValue = java2owfe(value);

            if (aValue != null)
                sma.put(attributeName, aValue);
        }
        catch (final Throwable t)
        {
            log.warn
                ("resolveGet() failed to get value from "+
                 getAndSet[0].getName()+"()", t);
        }
    }

    /**
     * Turns a java List into an OpenWFE ListAttribute
     */
    public static ListAttribute list2owfe (java.util.List l)
    {
        ListAttribute result = new ListAttribute();

        java.util.Iterator it = l.iterator();
        while (it.hasNext())
        {
            result.add(java2owfe(it.next()));
        }

        return result;
    }

    /**
     * Turns a java Map into an OpenWFE MapAttribute.
     * Doesn't produce a StringMapAttribute.
     */
    public static MapAttribute map2owfe (java.util.Map m)
    {
        MapAttribute result = new MapAttribute();

        fillMap(result, m);

        return result;
    }

    /**
     * Turns a map into a StringMapAttribute (may easily be used in 
     * conjunction with workitem.SetAttributes(x))
     */
    public static StringMapAttribute java2attributes (final java.util.Map m)
    {
        final StringMapAttribute result = new StringMapAttribute();

        fillMap(result, m);

        return result;
    }

    private static void fillMap (final MapAttribute ma, final java.util.Map m)
    {
        java.util.Iterator it = m.keySet().iterator();
        while (it.hasNext())
        {
            Object k = it.next();
            Object v = m.get(k);

            if (ma instanceof StringMapAttribute)
                ma.put(new StringAttribute(k.toString()), java2owfe(v));
            else
                ma.put(java2owfe(k), java2owfe(v));
        }
    }

    /**
     * Attempts to turn an attribute into an integer.
     */
    public static int toInt 
        (final ApplicationContext context, 
         Attribute a)
    {
        if (a == null)
        {
            throw new IllegalArgumentException
                ("cannot turn a null attribute into an int");
        }

        log.debug("toInt() a.class is "+a.getClass().getName());

        if (a instanceof XmlAttribute)
        {
            final Attribute a2 = 
                ((XmlAttribute)a).getAttributeContent(context);

            if (a2 != null) a = a2;
        }

        if (a instanceof IntegerAttribute)
            return ((IntegerAttribute)a).intValue();

        return Integer.parseInt(a.toString());
    }

    //
    // OTHER STUFF

}
