/*
 * Copyright (c) 2001-2006, John Mettraux, OpenWFE.org
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * . Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.  
 * 
 * . Redistributions in binary form must reproduce the above copyright notice, 
 *   this list of conditions and the following disclaimer in the documentation 
 *   and/or other materials provided with the distribution.
 * 
 * . Neither the name of the "OpenWFE" nor the names of its contributors may be
 *   used to endorse or promote products derived from this software without
 *   specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: WhenExpression.java 2902 2006-07-01 13:31:15Z jmettraux $
 */

//
// WhenExpression.java
//
// john.mettraux@openwfe.org
//
// generated with 
// jtmpl 1.1.01 2004/05/19 (john.mettraux@openwfe.org)
//

package openwfe.org.engine.expressions;

import openwfe.org.OpenWfeException;
import openwfe.org.ApplicationContext;
import openwfe.org.time.Time;
import openwfe.org.time.Scheduler;
import openwfe.org.time.Schedulable;
import openwfe.org.engine.expool.ExpressionPool;
import openwfe.org.engine.launch.LaunchException;
import openwfe.org.engine.history.History;
import openwfe.org.engine.workitem.InFlowWorkItem;


/**
 * An 'if' expression that blocks until its condition realizes or until
 * it times out.
 * There is no else clause for a 'when' expression.
 *
 * <p><font size=2>CVS Info :
 * <br>$Author: jmettraux $
 * <br>$Id: WhenExpression.java 2902 2006-07-01 13:31:15Z jmettraux $ </font>
 *
 * @author john.mettraux@openwfe.org
 */
public class WhenExpression

    extends CleanCompositeFlowExpression

    implements ExpressionWithTimeOut, Schedulable

{

    private final static org.apache.log4j.Logger log = org.apache.log4j.Logger
        .getLogger(WhenExpression.class.getName());

    //
    // CONSTANTS & co

    //
    // FIELDS

    private InFlowWorkItem workitem = null;
    private boolean conditionTreated = false;

    private Long atId = null;

    //
    // CONSTRUCTORS

    //
    // BEAN METHODS

    /**
     * When this expression gets applied, it 'traps' its workitem
     * until reply...
     */
    public InFlowWorkItem getWorkitem () { return this.workitem; }

    /**
     * Set to true when the condition got treated (before) the second and
     * last reply.
     */
    public boolean isConditionTreated () 
    { 
        //log.debug("isConditionTreated() returning "+this.conditionTreated);

        return this.conditionTreated; 
    }

    public void setWorkitem (final InFlowWorkItem wi) 
    { 
        if (wi == null) return;
        this.workitem = (InFlowWorkItem)wi.clone(); 
    }

    public void setConditionTreated (final boolean b) 
    { 
        //log.debug("setConditionTreated("+b+")");
        this.conditionTreated = b; 
    }

    /**
     * Returns the id of the scheduled 'at' job (used for cancelling it).
     */
    public Long getAtId ()
    {
        return this.atId;
    }

    /*
     * not necessary, at each 'rehydration', the expression is rescheduled
     * and receives a new job id.
     *
    public void setAtId (final Long l)
    {
        this.atId = l;
    }
     */

    //
    // METHODS

    /**
     * This method is used to poll the conditional clause to
     * see wether it's possible to apply the child
     */
    public void trigger (final Object[] params)
    {
        log.debug("trigger()");

        final FlowExpressionId templateId = 
            (FlowExpressionId)getChildren().get(0);

        //log.debug("trigger() =============== template is "+templateId);

        if (this.workitem == null)
        {
            log.debug("trigger() workitem is null. 'when' Skipped.");
            return;
        }

        if (isConditionTreated())
        {
            log.debug("trigger() condition already treated. Returning");
            return;
        }

        tag(this.workitem);

        log.debug("trigger() workitem got tagged");

        try
        {
            /*final FlowExpressionId rootExpressionId =*/ 
            getLauncher().launchSub
                (this.workitem, 
                 getId(), 
                 templateId,
                 new java.util.HashMap(0),  // empty var 'set'
                 true);  // async call (will return immediately)
            //
            // getId() indicates here that the parent id of the newly 
            // launched scope parent is this when expression...
            //
            // The last null stands for variables : none are passed for the
            // subprocess.
        }
        catch (final LaunchException e)
        {
            log.warn
                ("trigger() failed to launch condition checking logic !!!", e);
        }
    }

    /**
     * [re]schedules this expression.
     */
    public Long reschedule (final Scheduler s)
    {
        return getExpressionPool().schedule(this.getId());
    }

    /**
     * Returns the name of this expression as used
     * in the author field of the generated workitems
     */
    protected String getNameInHistory ()
    {
        return this.getClass().getName()+" $Revision: 2902 $";
    }

    //
    // METHODS from FlowExpression

    public void apply (final InFlowWorkItem wi) 
        throws ApplyException
    {
        log.debug("apply()");

        this.setWorkitem(wi);
        this.storeItself();

        this.trigger(null);
    }

    /**
     * This override takes care of removing the conditional subexpression
     * before the flow leaves this when expression.
     */
    public void replyToParent (final InFlowWorkItem wi)
        throws ReplyException
    {
        if (getChildren() != null && getChildren().size() > 0)
        {
            if (log.isDebugEnabled())
            {
                log.debug
                    ("replyToParent() removing template "+
                     this.getChildren().get(0));
            }

            getExpressionPool()
                .removeExpression((FlowExpressionId)this.getChildren().get(0));
        }

        super.replyToParent(wi);
    }

    public void reply (final InFlowWorkItem wi)
        throws ReplyException
    {
        log.debug("reply()");

        if (isConditionTreated())
        {
            log.debug("reply() condition is treated - replying to parent");

            //ValueUtils.cleanBooleanResult(wi);

            replyToParent(wi);
            return;
        }

        if (ValueUtils.lookupBooleanResult(wi))
        {
            //
            // add history to workitem

            this.workitem.addHistoryItem
                (getNameInHistory(),
                 "triggered");

            //
            // log to history service

            historyLog
                (wi,
                 History.EVT_INFO, 
                 null,
                 getNameInHistory());
                    // participantName is set to null

            //
            // do the rest

            if (getChildren().size() < 2)
            {
                log.info
                    ("reply() condition realized but no child to apply");

                replyToParent(wi);
                return;
            }

            //
            // apply child

            log.debug("reply() setting conditionTreated to true");
            setConditionTreated(true);

            try
            {
                this.storeItself();

                getExpressionPool().apply
                    ((FlowExpressionId)getChildren().get(1), this.workitem);
            }
            catch (ApplyException ae)
            {
                throw new ReplyException("reply() failure", ae);
            }
        }
        else
        {
            log.info("reply() conditional clause returned with 'false'");

            //
            // register in scheduler for further evaluation

            this.atId = this.reschedule(null);

            this.storeItself();
        }
    }

    /**
     * Cancels this expression by taking care of unscheduling its 'at job'.
     */
    public InFlowWorkItem cancel () 
        throws ApplyException
    {
        if (this.atId != null)
            getExpressionPool().getScheduler().unscheduleAt(this.atId);

        return super.cancel();
    }

    //
    // METHODS from ExpressionWithTimeOut

    /**
     * This method should return false if the purge daemon should not
     * consider the implemented and instantiated Expression during its run.
     */
    public boolean isTimeOutActivated ()
    {
        return true;
    }

    /**
     * The WhenExpression is subject to timeout, after a certain time, the
     * engine considers the condition will never evaluate to true and discards
     * the when, this method determines the size of this timeout.
     *
     * The priority list is :<br>
     * <ol>
     *   <li>when expression's 'timeout' attribute</li>
     *   <li>when expression's variable scope ('__timeout__')</li>
     *   <li>expressionPool 'expressionTimeout' param</li>
     * </ol>
     */
    public Long determineTimeOut ()
    {
        Long timeOut = TimeoutUtils.determineTimeout(this, this.workitem);

        //
        // determine timeOut at engine (expressionPool) level
        // (lowest priority)

        if (timeOut == null)
            timeOut = getExpressionPool().getExpressionTimeOut();

        // NB : returning a timeout of "-1" means timeout is deactivated 
        //      for the participant

        if (timeOut == null) timeOut = new Long(-1);
            //
            // this transformation shouldn't be necessary
        
        return timeOut;
    }

    /**
     * This method is called by the ExpressionStore when it notices that the
     * WhenExpression expired.
     */
    public void timeOutReply ()
        throws ReplyException
    {
        this.getWorkitem().addHistoryItem
            (getNameInHistory(), 
             "timed out");
        
        //
        // log timeout in history

        historyLog
            (null, EVT_TIMED_OUT, null, getNameInHistory());
                // participantName is set to null

        //
        // resume flow
        
        log.debug("timeOutReply() resuming flow");

        //super.reply(this.getWorkitem());
        reply(this.getWorkitem());
    }

    //
    // STATIC METHODS

}
