/*
 * Copyright (c) 2001-2006, John Mettraux, OpenWFE.org
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * . Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.  
 * 
 * . Redistributions in binary form must reproduce the above copyright notice, 
 *   this list of conditions and the following disclaimer in the documentation 
 *   and/or other materials provided with the distribution.
 * 
 * . Neither the name of the "OpenWFE" nor the names of its contributors may be
 *   used to endorse or promote products derived from this software without
 *   specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: CachedExpressionPool.java 2717 2006-06-01 15:26:06Z jmettraux $
 */

//
// CachedExpressionPool.java
//
// jmettraux@openwfe.org
//
// generated with 
// jtmpl 1.0.04 20.11.2001 John Mettraux (jmettraux@openwfe.org)
//

package openwfe.org.engine.impl.expool;

import openwfe.org.MapUtils;
import openwfe.org.engine.Definitions;
import openwfe.org.ApplicationContext;
import openwfe.org.ServiceException;
import openwfe.org.misc.Cache;
import openwfe.org.engine.expool.PoolException;
import openwfe.org.engine.history.History;
import openwfe.org.engine.workitem.InFlowWorkItem;
import openwfe.org.engine.dispatch.DispatchingException;
import openwfe.org.engine.expressions.FlowExpression;
import openwfe.org.engine.expressions.ApplyException;
import openwfe.org.engine.expressions.ReplyException;
import openwfe.org.engine.expressions.FlowExpressionId;
import openwfe.org.engine.expressions.CompositeFlowExpression;
import openwfe.org.engine.participants.Participant;
import openwfe.org.engine.participants.ParticipantMap;


/**
 * This is the best available expression pool : it has a bounded capacity 
 * (unlike the InMemoryExpressionPool), and the performance price for this is 
 * small.
 *
 * <p><font size=2>CVS Info :
 * <br>$Author: jmettraux $
 * <br>$Date: 2006-06-01 17:26:06 +0200 (Thu, 01 Jun 2006) $
 * <br>$Id: CachedExpressionPool.java 2717 2006-06-01 15:26:06Z jmettraux $ </font>
 *
 * @author john.mettraux@openwfe.org
 */
public class CachedExpressionPool

    extends SimpleExpressionPool

{

    private final static org.apache.log4j.Logger log = org.apache.log4j.Logger
        .getLogger(CachedExpressionPool.class.getName());

    //
    // INNER CLASSES

    //
    // CONSTANTS

    /**
     * Use this parameter name to tell the expression pool how may
     * expressions should be kept cached (ie not requiring a IO operation
     * or whatever costly to fetch back from the ExpressionStore).
     */
    public static String P_CACHE_SIZE = "cacheSize";

    //
    // FIELDS

    private int cacheSize = -1;

    private Cache cache = null;

    //
    // CONSTRUCTORS

    /*
    public CachedExpressionPool ()
    {
        super();
    }
    */

    public void init 
        (final String serviceName, 
         final ApplicationContext context, 
         final java.util.Map serviceParams)
    throws 
        ServiceException
    {
        super.init(serviceName, context, serviceParams);

        //
        // prepare cache

        this.cacheSize = MapUtils.getAsInt(serviceParams, P_CACHE_SIZE, 500);

        this.cache = new Cache(this.cacheSize);

        log.info("init() "+P_CACHE_SIZE+" : "+this.cacheSize);
    }

    //
    // METHODS

    /**
     * Returns the cache used by the pool.
     */
    public Cache getCache ()
    {
        return this.cache;
    }

    /**
     * Usually called by WorkflowInstanceBuilders to add a freshly created
     * expression to the pool.
     */
    public void add (final FlowExpression fe)
        throws PoolException
    {
        this.cache.put(fe.getId(), fe);

        super.add(fe);
    }

    /**
     * Stores the flow expression (as it may have changed).
     */
    public void update (final FlowExpression fe)
        throws PoolException
    {
        //log.debug("update() "+fe.getId());

        this.cache.put(fe.getId(), fe);
        //log.debug("update() put in cache");

        super.update(fe);
        //log.debug("update() super.update()...");
    }

    /**
     * Retrieves an expression from the pool.
     * If the expression cannot be retrieved from the pool, null will be
     * returned.
     */
    public /*synchronized*/ FlowExpression fetch (final FlowExpressionId fei)
    {
        if (log.isDebugEnabled())
            log.debug("fetch() for "+fei);

        if (fei == null) return null;

        FlowExpression result = (FlowExpression)this.cache.get(fei);

        if (result != null) return result;

        if (log.isDebugEnabled())
            log.debug("fetch() not cached : "+fei);

        //log.debug
        //    ("fetch()\n"+
        //     openwfe.org.xml.XmlUtils.toString(this.cache.getStatus()));
        //log.debug
        //    ("fetch() cache size : "+this.cache.size());

        result = super.fetch(fei);

        if (result != null) this.cache.put(result.getId(), result);

        return result;
    }

    /**
     * Removes a flow expression :
     * - removes it from cache;
     * - removes it from store (disk or db)
     *
     * This method is called by the replyToFather method, as the
     * expression tree gets diminished.
     */
    public /*synchronized*/ void removeExpression (final FlowExpression fe)
    {
        //log.debug("removeExpression() is fe null ? "+(fe == null));

        this.cache.remove(fe.getId());

        super.removeExpression(fe);
    }

    //
    // STATUS

    public org.jdom.Element getStatus ()
    {
        org.jdom.Element result = super.getStatus();

        result.addContent(this.cache.getStatus());

        return result;
    }

}
