/*
 * Copyright (c) 2001-2006, John Mettraux, OpenWFE.org
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * . Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.  
 * 
 * . Redistributions in binary form must reproduce the above copyright notice, 
 *   this list of conditions and the following disclaimer in the documentation 
 *   and/or other materials provided with the distribution.
 * 
 * . Neither the name of the "OpenWFE" nor the names of its contributors may be
 *   used to endorse or promote products derived from this software without
 *   specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: SimpleExpressionPool.java 2905 2006-07-02 15:17:05Z jmettraux $
 */

//
// SimpleExpressionPool.java
//
// jmettraux@openwfe.org
//
// generated with 
// jtmpl 1.0.04 20.11.2001 John Mettraux (jmettraux@openwfe.org)
//

package openwfe.org.engine.impl.expool;

import openwfe.org.MapUtils;
import openwfe.org.Application;
import openwfe.org.AbstractService;
import openwfe.org.ServiceException;
import openwfe.org.ApplicationContext;
import openwfe.org.ReflectionUtils;
import openwfe.org.xml.XmlUtils;
import openwfe.org.time.Time;
import openwfe.org.time.Scheduler;
import openwfe.org.time.Schedulable;
import openwfe.org.engine.Definitions;
import openwfe.org.engine.control.auth.ControlPermission;
import openwfe.org.engine.expool.PoolException;
import openwfe.org.engine.expool.ExpressionPool;
import openwfe.org.engine.expool.ExpressionStore;
import openwfe.org.engine.expool.FailureStrategy;
import openwfe.org.engine.history.History;
import openwfe.org.engine.workitem.WorkItem;
import openwfe.org.engine.workitem.InFlowWorkItem;
import openwfe.org.engine.dispatch.DispatchingException;
import openwfe.org.engine.expressions.Environment;
import openwfe.org.engine.expressions.GoneParentId;
import openwfe.org.engine.expressions.TimeoutUtils;
import openwfe.org.engine.expressions.DoExpression;
import openwfe.org.engine.expressions.FlowExpression;
import openwfe.org.engine.expressions.ApplyException;
import openwfe.org.engine.expressions.ReplyException;
import openwfe.org.engine.expressions.FlowExpressionId;
import openwfe.org.engine.expressions.ExpressionWithTimeOut;
import openwfe.org.engine.expressions.state.NormalState;
import openwfe.org.engine.expressions.state.FrozenState;
import openwfe.org.engine.expressions.state.ExpressionState;
import openwfe.org.engine.participants.Participant;
import openwfe.org.engine.participants.ParticipantMap;


/**
 * The simplest implementation of the ExpressionPool.
 * Other implementations of the expool Interface do extend this class.
 * It thus contains the core of what constitutes an expression pool.
 *
 * <p><font size=2>CVS Info :
 * <br>$Author: jmettraux $
 * <br>$Date: 2006-07-02 17:17:05 +0200 (Sun, 02 Jul 2006) $
 * <br>$Id: SimpleExpressionPool.java 2905 2006-07-02 15:17:05Z jmettraux $ </font>
 *
 * @author john.mettraux@openwfe.org
 */
public class SimpleExpressionPool

    extends AbstractService

    implements ExpressionPool

{

    private final static org.apache.log4j.Logger log = org.apache.log4j.Logger
        .getLogger(SimpleExpressionPool.class.getName());

    //
    // INNER CLASSES

    //
    // CONSTANTS

    /*
     * How long should the expool wait before rescheduling previously 
     * persisted (in the store) expressions upon engine restart ?
     */
    private final static String PP_RESCHEDULE_IN
        = "1m";

    /**
     * The default value for this parameter being '0.25', stating that
     * the scheduler will wake up 4 times per second for checking scheduled
     * tasks. Setting this parameter value to '1.0' would make the scheduler 
     * wake up once per second.
     * (@value)
     */
    public final static String P_SCHEDULER_PRECISION
        = "schedulerPrecision";

    /**
     * This parameter 'failureStrategyClass' indicates to the expression 
     * pool how it should behave in case of process failure. This may
     * be very important on production installations. A typical FailureStrategy
     * implementation would store the 'failed' workitems in a directory
     * where they could be rescued.
     */
    public final static String P_FAILURE_STRATEGY_CLASS
        = "failureStrategyClass";

    /**
     * The default failure strategy is implemented in class
     * 'openwfe.org.impl.expool.FailureStrategyNone', which simply
     * discards the 'failed' workitem.
     */
    protected final static String DEFAULT_FAILURE_STRATEGY_CLASSNAME
        = openwfe.org.engine.impl.expool.FailureStrategyNone.class.getName();


    private final int C_LAUNCH = -1;
    private final int C_APPLY = 1;

    //
    // FIELDS

    private ExpressionStore store = null;

    private java.util.TimerTask purgeTask = null;

    private Scheduler scheduler = null;
    private long schedFrequency = Time.parseTimeString("3m");

    private Long expressionTimeOut = null;

    private FailureStrategy failureStrategy = null;

    //
    // CONSTRUCTORS

    public void init 
        (final String serviceName, 
         final ApplicationContext context, 
         final java.util.Map serviceParams)
    throws 
        ServiceException
    {
        super.init(serviceName, context, serviceParams);

        //
        // determine purge frequency and start purge daemon

        String sPurgeFrequency = MapUtils.getAsString
            (getParams(), P_PURGE_FREQUENCY, DEFAULT_PURGE_FREQUENCY);

        final long purgeFrequency = Time.parseTimeString(sPurgeFrequency);
        
        this.purgeTask = new java.util.TimerTask()
        {
            public void run ()
            {
                // 
                // Always catch anything that could go out of the body
                // of a timer task !
                try
                {
                    purge();
                }
                catch (final Throwable t)
                {
                    log.warn("purgeTask : problem", t);
                }
            }
        };

        Application.getTimer().schedule
            (this.purgeTask,
             3 * 60 * 1000,  // start after 3mn
             purgeFrequency);

        log.info("init() started purge daemon every '"+sPurgeFrequency+"'");

        //
        // scheduler

        final float schedPrecision = MapUtils.getAsFloat
            (serviceParams, P_SCHEDULER_PRECISION, 0.25f);

        log.info("init() schedPrecision is "+schedPrecision);

        this.schedFrequency = MapUtils.getAsTime
            (serviceParams, P_WHEN_FREQUENCY, DEFAULT_WHEN_FREQUENCY);

        log.info("init() schedFrequency is "+this.schedFrequency+" ms");

        this.scheduler = new Scheduler
            (getContext().getApplicationName()+"::expool.sched");

        this.scheduler.setPrecision(schedPrecision);

        this.scheduler.start();

        if (MapUtils.getAsBoolean(getParams(), P_RESCHEDULE, true))
        {
            log.info("init() reschedule will occur in "+PP_RESCHEDULE_IN);

            this.scheduler.scheduleIn(PP_RESCHEDULE_IN, this, null);
                // will trigger a reschedule() soon...
        }

        //
        // expression timeout

        this.expressionTimeOut = null;

        final String sExpressionTimeOut = MapUtils.getAsString
            (this.getParams(), P_EXPRESSION_TIMEOUT);

        if (sExpressionTimeOut != null && log.isDebugEnabled())
            log.debug("init() eto >"+sExpressionTimeOut+"<");

        this.expressionTimeOut = 
            TimeoutUtils.determineTimeout(sExpressionTimeOut);

        if (log.isDebugEnabled())
            log.debug("init() eto 0 is "+this.expressionTimeOut);

        if (this.expressionTimeOut == null)
            this.expressionTimeOut = DEFAULT_EXPRESSION_TIMEOUT;

        if (log.isDebugEnabled())
            log.debug("init() eto 1 is "+this.expressionTimeOut);

        if (this.expressionTimeOut.longValue() > -1)
        {
            log.info
                ("init() "+
                 "expool (engine-level) expression timeout set to "+
                 this.expressionTimeOut.longValue()+" ms  ("+
                 Time.toTimeString(this.expressionTimeOut, false)+")");
        }
        else
        {
            log.info
                ("init() "+
                 "expool (engine-level) expression timeout "+
                 "set to 'no timeout'");
        }

        //
        // 'failureStrategy'

        final String s = MapUtils.getAsString
            (serviceParams, 
             P_FAILURE_STRATEGY_CLASS, 
             DEFAULT_FAILURE_STRATEGY_CLASSNAME);

        try
        {
            this.failureStrategy = (FailureStrategy)ReflectionUtils
                .initObjectOfClass(s, serviceParams);
        }
        catch (final Exception e)
        {
            throw new ServiceException
                ("failed to init FailureStrategy '"+s+"'", e);
        }

        //
        // display OpenWFE version in logs
        
        log.info("OpenWFE version : "+Definitions.OPENWFE_VERSION);
    }

    //
    // METHODS

    /**
     * Returns the default expression timeout for this expression pool.
     */
    public Long getExpressionTimeOut ()
    {
        return this.expressionTimeOut;
    }

    /**
     * Returns the object in charge of 'failed' workitems for this
     * expression pool.
     */
    public FailureStrategy getFailureStrategy ()
    {
        return this.failureStrategy;
    }

    /**
     * Stops this expression pool and takes especially care of stopping the 
     * purgeTask and the scheduler.
     */
    public void stop ()
        throws ServiceException
    {
        super.stop();

        this.purgeTask.cancel();
        log.info("stop() PurgeDaemon stopped.");

        this.scheduler.stop();
        log.info("stop() scheduler stopped.");

        log.info("stop() Service '"+getName()+"' stopped.");
    }

    /**
     * Usually called by Launcher implementations to add a freshly created
     * expression to the pool.
     */
    public void add (final FlowExpression fe)
        throws PoolException
    {
        //openwfe.org.Utils.logStackTrace(log, "add()");

        fe.setApplicationContext(getContext());

        getStore().storeExpression(fe);

        if (log.isDebugEnabled())
            log.debug("add() added  "+fe.getId());
    }

    /**
     * Returns the count of expressions stores in this expool.
     */
    public int size ()
    {
        return getStore().size();
    }

    /**
     * Returns an XML representation of the current state of the expression 
     * pool : the forest (a set of expression trees / workflow instances).
     */
    public synchronized org.jdom.Document dump ()
    {
        final org.jdom.Element rootElt = new org.jdom.Element("expool");
        final org.jdom.Document doc = new org.jdom.Document(rootElt);

        final java.util.List content = this.listExpressions();

        rootElt.setAttribute("time", (new java.util.Date()).toString());
        rootElt.setAttribute("size", ""+content.size());

        final java.util.Iterator it = content.iterator();
        while (it.hasNext())
        {
            final FlowExpression fe = (FlowExpression)it.next();

            //if (fe.getParent() != null) continue;

            rootElt.addContent(fe.dump());
        }

        return doc;
    }

    /**
     * Stores the flow expression (as it may have changed).
     */
    public void update (final FlowExpression fe)
        throws PoolException
    {
        getStore().storeExpression(fe);
    }

    /**
     * Retrieves an expression from the pool.
     * If the expression cannot be retrieved from the pool, null will be
     * returned.
     */
    public FlowExpression fetch (final FlowExpressionId fei)
    {
        if (log.isDebugEnabled())
            log.debug("fetch() for "+fei);

        //openwfe.org.Utils.logStackTrace(log);
        if (fei == null) return null;

        FlowExpression result = null;
        try
        {
            result = getStore().loadExpression(fei);

            //log.debug("fetch() expression got loaded");

            //result.setApplicationContext(getContext());
                // 
                // done by the store
        }
        catch (final PoolException e)
        {
            log.info
                ("fetch() Failed to retrieve expression "+fei+
                 " because of "+e);
            //log.debug("fetch() Failed to retrieve expression "+fei, e);
            return null;
        }

        return result;
    }

    /**
     * Fetches the root environement tied to the engine.
     */
    public Environment fetchEngineEnvironment ()
    {
        final Environment e = getStore().loadEngineEnvironment();

        e.setApplicationContext(this.getContext());

        return e;
    }

    /**
     * Given an expression id, returns the environment id of the 
     * flow expression.
     */
    public FlowExpressionId getEnvironmentId (final FlowExpressionId fei)
    {
        final FlowExpression fe = fetch(fei);

        if (fe == null) return null;

        return fe.getEnvironmentId();
    }

    /**
     * given an id, returns the root expression of the flow
     */
    protected FlowExpression fetchRootOfFlow (final FlowExpressionId fei)
    {
        final FlowExpression fe = fetch(fei);
        final FlowExpressionId parentId = fe.getParent();
        if (parentId == null) return fe;
        return fetchRootOfFlow(parentId);
    }

    /**
     * given a workflow instance id, returns the root expression of the flow
     */
    protected FlowExpression fetchRootOfFlow (final String workflowInstanceId)
    {
        //
        // iterate until an expression with the given wfdid is found
        // and then return its root expression

        final java.util.Iterator it = this.contentIterator(null);
        while (it.hasNext())
        {
            final FlowExpression fe = (FlowExpression)it.next();

            if (fe instanceof Environment) continue;

            if (fe.getId().getWorkflowInstanceId().equals(workflowInstanceId))
                return fetchRootOfFlow(fe.getId());
        }

        return null;
    }

    /**
     * A shortcut method for logging messages into the history log.
     */
    protected void log 
        (final FlowExpressionId fei, 
         final WorkItem wi,
         final String eventCode, 
         final String message)
    {
        final History history = Definitions.getHistory(getContext());

        if (log.isDebugEnabled())
            log.debug(""+fei+" :: "+eventCode+" :: "+message);

        if (history == null) return;

        history.log(fei, wi, eventCode, null, message);
            // participantName is set to null
    }

    /*
     * Any launch() or apply() call passes here.
     */
    private void applyOrLaunch
        (final int command, 
         final FlowExpressionId fei, 
         final InFlowWorkItem wi)
    throws 
        ApplyException
    {
        //log.debug("apply() dumpExpool() ::\n"+dumpExpool());

        final FlowExpression fe = fetch(fei);

        if (fe == null)
        {
            log.warn
                ("Failed to find expression  "+fei+"  cannot apply it.");
            throw new ApplyException
                ("Failed to find expression  "+fei+"  cannot apply it.");
        }

        log(fei, wi, History.EVT_DEBUG, "applying");

        fe.tag(wi);
        fe.touchApplyTime();

        //if (log.isDebugEnabled())
        //    log.debug("apply() tagged wi to "+wi.getLastExpressionId());

        if (command == C_LAUNCH)
            fe.launch(wi);
        else
            getState(fe).apply(wi);
    }

    /**
     * Applies an expression. 
     * This method is usually called by expressions on their subexpressions or
     * by a launchListener on a rootExpression of a workflow just instantiated.
     */
    public void apply 
        (final FlowExpressionId fei, final InFlowWorkItem wi)
    throws 
        ApplyException
    {
        this.applyOrLaunch(C_APPLY, fei, wi);
    }

    /**
     * A shortcut to 
     * apply (workflowInstanceId, flowExpressionId, inFlowWorkItem)
     */
    public void apply
        (final FlowExpression fe, final InFlowWorkItem wi)
    throws 
        ApplyException
    {
        this.apply(fe.getId(), wi);
    }

    /**
     * Applies an expression (calls the launch() method of that expression.
     */
    public void launch
        (FlowExpressionId fei, InFlowWorkItem wi)
    throws 
        ApplyException
    {
        this.applyOrLaunch(C_LAUNCH, fei, wi);
    }

    /**
     * A shortcut to 
     * launch (flowExpressionId, inFlowWorkItem).
     */
    public void launch 
        (final FlowExpression fe, final InFlowWorkItem wi)
    throws 
        ApplyException
    {
        this.launch(fe.getId(), wi);
    }

    /**
     * Replies to the parent of the given flow expression, with the given 
     * workitem.
     */
    public void replyToParent
        (final FlowExpression fe, 
         final InFlowWorkItem wi)
    throws 
        ReplyException
    {
        if (log.isDebugEnabled())
        {
            log.debug("replyToParent() fe.id     "+fe.getId());
            log.debug("replyToParent() fe.parent "+fe.getParent());
        }

        //
        // end of a flow

        if (fe.getParent() == null)
        {
            //log.debug("replyToParent() no parent, end of flow at "+fe.getId());

            String message = "";
            if (fe.getId().isInSubFlow()) message = "sub";

            log(fe.getId(), wi, History.EVT_FLOW_END, "");

            this.removeExpression(fe);
            return;
        }

        if (fe.getParent() == GoneParentId.GONE_PARENT_ID)
        {
            //log.debug("replyToParent() gone parent at "+fe.getId());

            log(fe.getId(), wi, History.EVT_GONE_PARENT, "");

            this.removeExpression(fe);
            return;
        }

        //
        // end of a subflow

        if ( ! fe.getId().getWorkflowInstanceId()
                .equals(fe.getParent().getWorkflowInstanceId()))
        {
            log(fe.getId(), wi, History.EVT_FLOW_END, "sub");
        }

        //
        // just replying to the parent
        
        FlowExpressionId parentId = fe.getParent();

        if (log.isDebugEnabled())
        {
            log.debug
                ("replyToParent() \n   "+fe.getId()+
                 "\n   replies to\n   "+parentId);
        }

        if (fe.getId().getEngineId().equals(parentId.getEngineId()))
            reply(parentId, wi);
        else
            replyToRemoteParent(parentId, wi);

        //
        // remove from pool

        this.removeExpression(fe);
    }

    /**
     * This method is called when the parent expression is located in another
     * engine. It's the key to distributed workflows with OpenWFE.
     */
    protected void replyToRemoteParent
        (final FlowExpressionId fei, final InFlowWorkItem wi)
    throws 
        ReplyException
    {
        log(fei, wi, History.EVT_DEBUG, "replying (to remote parent)");

        final ParticipantMap pMap = Definitions.getParticipantMap(getContext());

        final Participant parentEngine = pMap.get(fei.getEngineId());

        if (parentEngine == null)
        {
            //log.warn
            //    ("replyToRemoteParent() no remote engine named '"+
            //     fei.getEngineId()+"'");
            throw new ReplyException
                ("No remote engine named '"+
                 fei.getEngineId()+"'");
        }

        try
        {
            parentEngine.dispatch(getContext(), wi);
        }
        catch (final DispatchingException de)
        {
            throw new ReplyException
                ("Failed to reply to remote engine '"+fei.getEngineId()+"'");
        }
    }

    /**
     * Replies to an expression.
     * This method should only be called by listeners
     */
    public void reply 
        (final FlowExpressionId fei, final InFlowWorkItem wi)
    throws 
        ReplyException
    {
        log(fei, wi, History.EVT_DEBUG, "replying");

        //log.debug
        //    ("reply() __result__ is \n"+
        //     openwfe.org.engine.expressions.ValueUtils.getResult(wi));

        final FlowExpression fe = fetch(fei);

        if (fe == null)
        {
            throw new ReplyException
                ("Cannot reply : expression not found "+fei);
        }

        getState(fe).reply(wi);

        fe.tag(wi);
            //
            // shouldn't we do it before a potential freeze ?
            // i.e. shouldn't this tag() occur before the reply() two
            // lines before ?
            //
            // but well, as of 1.6.3pre3, interverting them breaks
            // deferred choice test (1.4bb)...
    }

    /**
     * Releases a flow expression.
     * This method is called by the replyToFather method, as the
     * expression tree gets diminished.
     */
    public void removeExpression (final FlowExpression fe)
    {
        if (log.isDebugEnabled())
            log.debug("removeExpression() "+fe.getId());
        //log(fe.getId(), null, History.EVT_DEBUG, "removing...");

        //openwfe.org.Utils.logStackTrace(log, "removeExpression()");
            //
            // may be really useful to help locate some bugs

        try
        {
            //
            // remove expression

            if (getStore() != null)
            {
                getStore().unstoreExpression(fe);
            }

            //
            // if it owns an environment, remove the env as well

            if (( ! (fe instanceof Environment)) &&
                (Environment.ownsEnvironment(fe)))
            {
                removeEnvironment(fe);
            }

            //
            // if this is an environment, unbinds its unboundable values

            if (fe instanceof Environment)
            {
                ((Environment)fe).unbind();
            }
        }
        catch (final PoolException pe)
        {
            log.warn
                ("removeExpression() trouble", pe);
        }
    }

    /**
     * Releases a flow expression indicated by its id.
     */
    public void removeExpression (final FlowExpressionId fei)
    {
        if (log.isDebugEnabled())
            log.debug("removeExpression(fei) for "+fei);

        final FlowExpression fe = this.fetch(fei);

        if (fe == null)
        {
            if (log.isDebugEnabled())
            {
                log.debug
                    ("removeExpression(fei) has already been removed : "+fei);
            }
            return;
        }

        this.removeExpression(fe);
    }

    /**
     * Removes the environment of a given expression.
     */
    public void removeEnvironment (final FlowExpression requester)
    {
        if (log.isDebugEnabled())
        {
            log.debug
                ("removeEnvironment() "+
                 "for           "+requester.getId());
            log.debug
                ("removeEnvironment() "+
                 "thus removing "+requester.getEnvironmentId());
        }

        removeExpression(requester.getEnvironmentId());
    }

    /**
     * (leaf cancel) Cancels an expression (simply calls its cancel method and 
     * removes it from the pool).
     * Will return a workitem if the cancelled expression was handling one.
     */
    public InFlowWorkItem childCancel 
        (final FlowExpressionId fei)
    throws 
        ApplyException
    {
        if (log.isDebugEnabled())
            log.debug("childCancel() "+fei);

        final FlowExpression fe = fetch(fei);

        if (fe == null) // still null ?
        {
            log.debug("childCancel() expression not found, not cancelling it.");
            return null;
        }

        final InFlowWorkItem wi = getState(fe).cancel();

        this.removeExpression(fe);

        //log.debug("childCancel() removed "+fe.getId());

        return wi;
    }

    /**
     * (root cancel) Cancels a whole branch, starting with its top expression
     * fei.
     */
    public void cancel (final FlowExpressionId fei)
    {
        if (log.isDebugEnabled())
            log.debug("cancel() "+fei);

        final FlowExpression fe = fetch(fei);

        if (fe == null) return;

        log(fei, null, History.EVT_CANCEL_EXPRESSION, "cancelling exp");

        try
        {
            //
            // cancelling

            final InFlowWorkItem wi = getState(fe).cancel();

            if (wi != null)
                //
                // taking care of resuming the flow
            {
                this.replyToParent(fe, wi);
            }

            this.removeExpression(fe);
        }
        catch (final Throwable t)
        {
            log.warn
                ("cancel() Failed to cancel an expression, "+
                 "removing it from pool anyway.", t);
        }
    }

    /**
     * Cancels a whole process instance.
     *
     * @param fei the FlowExpressionId of any expression belonging to the
     * flow that will get cancelled.
     */
    public void cancelFlow (final FlowExpressionId fei)
    {
        if (log.isDebugEnabled())
            log.debug("cancelFlow() requested by  "+fei);

        final FlowExpression root = fetchRootOfFlow(fei);

        if (root == null)
        {
            log.warn
                ("cancelFlow(fei) "+
                 "did not find root of flow for exp  "+fei+
                 ". Cannot cancel flow");
            return;
        }

        cancel(root.getId());
    }

    /**
     * Cancels a whole process instance.
     *
     * @param workflowInstanceId the unique id of the workflow instance to
     * cancel.
     */
    public void cancelFlow (final String workflowInstanceId)
    {
        if (log.isDebugEnabled())
            log.debug("cancelFlow() for wfid "+workflowInstanceId);

        final FlowExpression root = fetchRootOfFlow(workflowInstanceId);

        if (root == null)
        {
            log.warn
                ("cancelFlow(l) "+
                 "did not find root of flow for wfid "+workflowInstanceId+
                 ". Cannot cancel flow");
            return;
        }

        cancel(root.getId());
    }

    /**
     * Forgets an expression (it stays in the pool, but its reference to its
     * parent expression is replaced by a 'gone-parent' tag, so that its
     * execution continues but its results will get forgotten (never reaching
     * the parent flow)).
     */
    public void forget (final FlowExpressionId fei)
    {
        final FlowExpression fe = fetch(fei);

        if (fe == null) return;

        fe.setParent(GoneParentId.GONE_PARENT_ID);

        fe.storeItself();
    }

    /**
     * This implementation simply delegates the work to the underlying
     * store's contentIterator() method.
     */
    public java.util.Iterator contentIterator (final Class assignClass)
    {
        return getStore().contentIterator(assignClass);
    }

    //
    // purge and when methods

    /**
     * This method checks every expressions stored and reply to those 
     * that are ParticipantExpression instances and whose timeout has come.
     */
    protected synchronized void purge ()
    {
        //log.debug("purge()\n"+dumpExpool());

        final long startTime = System.currentTimeMillis();

        final java.util.Set timedOutExpressions = new java.util.HashSet();

        log.info("purge() starting...");

        java.util.Iterator it = 
            contentIterator(ExpressionWithTimeOut.class);
        while (it.hasNext())
        {
            FlowExpression fe = (FlowExpression)it.next();

            if (fe.getApplyTime() == null) continue;

            ExpressionWithTimeOut toe = (ExpressionWithTimeOut)fe;

            //
            // some expressions may have their timeout disabled

            if ( ! toe.isTimeOutActivated()) continue;

            //
            // gather participant expression who timed out

            if (fe.getTimeSinceApplied() > toe.determineTimeOut().longValue())
            {
                timedOutExpressions.add(toe);
            }
        }

        //
        // trigger their replies (outside of sync scope)

        it = timedOutExpressions.iterator();
        while (it.hasNext())
        {
            Object o = it.next();
            ExpressionWithTimeOut toe = (ExpressionWithTimeOut)o;
            FlowExpression fe = (FlowExpression)o;
            try
            {
                log.info("purge() Timeout reply for "+fe.getId());

                toe.timeOutReply();
            }
            catch (final ReplyException re)
            {
                log.warn("purge() Failed to reply to "+fe.getId(), re);
            }
        }

        long elapsedTime = 
            ((System.currentTimeMillis() - startTime) / 1000L);

        log.info("purge() Purge ends. It took "+elapsedTime+"s.");
    }

    private DoExpression lookupDoExpression 
        (final FlowExpression caller, final String doName)
    throws 
        ApplyException
    {
        String varName = DoExpression.V_DO;
        if (doName != null) 
            varName += doName;
        else
            varName += ".*";

        final FlowExpressionId doei = 
            (FlowExpressionId)lookupVariable(caller, varName);

        final DoExpression doe = (DoExpression)this.fetch(doei);

        if (doe == null)
        {
            throw new ApplyException
                ("Cannot undo : didn't find 'do' named '"+varName+"'");
        }
        
        return doe;
    }

    /**
     * Looks up the containing 'do' or the 'do' with the given doName and
     * triggers its undo() method.
     */
    public void undo (final FlowExpression caller, final String doName)
        throws ApplyException
    {
        lookupDoExpression(caller, doName).undo();
    }

    /**
     * Looks up the containing 'do' or the 'do' with the given doName and
     * triggers its redo() method.
     */
    public void redo (final FlowExpression caller, final String doName)
        throws ApplyException
    {
        lookupDoExpression(caller, doName).redo();
    }

    //
    // CONTROL METHODS

    private boolean isSecurityManagerPresent ()
    {
        try
        {
            return (System.getProperty("java.security.manager") != null);
        }
        catch (final java.security.AccessControlException ace)
        {
            // obviously, it's present...
        }

        return true;
    }

    private void securityCheck (final String actionName)
    {
        if (isSecurityManagerPresent())
        {
            java.security.AccessController.checkPermission
                (ControlPermission.newControlPermission(getName(), actionName));
        }
    }

    public java.util.List listExpressions ()
    {
        securityCheck("read");

        final java.util.List result = new java.util.ArrayList(2048);

        final java.util.Iterator it = contentIterator(null);

        while (it.hasNext()) result.add(it.next());

        if (log.isDebugEnabled())
        {
            log.debug
                ("listExpressions() returning "+result.size()+
                 " expressions");
        }

        return result;
    }

    public void unfreezeExpression (final FlowExpressionId fei)
        throws PoolException
    {
        securityCheck("freeze");

        final FlowExpression fe = fetch(fei);

        try
        {
            getState(fe).exitState();
        }
        catch (final ApplyException ae)
        {
            throw new PoolException("Unfreeze failure", ae);
        }

        if (log.isDebugEnabled())
            log.debug("unfreezeExpression() unfroze "+fei);

        getHistory().log(fe.getId(), null, FrozenState.EVT_UNFREEZE, null, "");
    }

    /**
     * Freezes an entire flow.
     */
    public void freezeFlow (final String workflowInstanceId)
        throws PoolException
    {
        securityCheck("freeze");

        log.warn("freezeFlow() not yet implemented.");

        throw new PoolException("freezeFlow() not yet implemented.");
    }

    /**
     * Attaches a frozen state to the given expression.
     */
    public void freezeExpression (final FlowExpressionId fei)
        throws PoolException
    {
        securityCheck("freeze");

        final FlowExpression fe = fetch(fei);

        if (fe.getState() != null)
        {
            throw new PoolException
                ("Expression already in state '"+fe.getState().getName()+"'");
        }

        fe.setState(new FrozenState());
        fe.storeItself();

        if (log.isDebugEnabled())
            log.debug("freezeExpression() froze "+fei);

        getHistory().log(fe.getId(), null, FrozenState.EVT_FREEZE, null, "");
    }

    //
    // METHODS about variables

    /**
     * Returns the environment tied to a given expression.
     */
    protected Environment lookupEnvironment (final FlowExpression requester)
    {
        if (log.isDebugEnabled())
        {
            log.debug
                ("lookupEnvironment() requester :       "+requester.getId());
        }
        /*
        log.debug
            ("lookupEnvironment() requester.envId : "+
             requester.getEnvironmentId());
        */

        if (requester.getEnvironmentId() == null)
        {
            //log.debug("lookupEnvironment() returning engine env...");

            return fetchEngineEnvironment();
        }

        return (Environment)this.fetch(requester.getEnvironmentId());
    }

    /**
     * Returns the environment with no top environment (except for the
     * engine environment, which it does not reference directly).
     */
    public Environment fetchRootEnvironment (final FlowExpression requester)
    {
        if (requester instanceof Environment)
        {
            if (requester.getEnvironmentId() == null)
                return (Environment)requester;

            final Environment parentEnv = 
                (Environment)fetch(requester.getEnvironmentId());

            return fetchRootEnvironment(parentEnv);
        }

        return fetchRootEnvironment(lookupEnvironment(requester));
    }

    /**
     * Sets a variable.
     * If the variableName is prefixed, this method will take care of 
     * storing the variable in the appropriate [upper] expression.
     */
    public void setVariable
        (final FlowExpression requester, 
         final String variableName, 
         final Object value)
    {
        if (log.isDebugEnabled())
        {
            log.debug("setVariable() '"+variableName+"' -> "+value);

            if (requester != null)
            {
                log.debug
                    ("setVariable() '"+variableName+"' for "+requester.getId());
            }
        }

        if (variableName.startsWith("///"))
        {
            fetchEngineEnvironment().put(variableName.substring(3), value);
            return;
        }

        if (variableName.startsWith("//"))
        {
            fetchEngineEnvironment().put(variableName.substring(2), value);
            return;
        }

        if (variableName.startsWith("/"))
        {
            final Environment rootEnv = fetchRootEnvironment(requester);

            rootEnv.put(variableName.substring(1), value);

            if (log.isDebugEnabled())
                log.debug("setVariable() put into "+rootEnv.getId());

            return;
        }

        Environment env = null;

        if (requester instanceof Environment)
            env = (Environment)requester;
        else
            env = lookupEnvironment(requester);

        if (log.isDebugEnabled())
            log.debug("setVariable() in local env  "+env.getId());

        env.put(variableName, value);
    }

    /**
     * Looks up the value of a variable. Will lookup until it reaches the
     * engine environment.
     */
    public Object lookupVariable 
        (final FlowExpression requester, final String variableName)
    {
        if (log.isDebugEnabled())
        {
            log.debug("lookupVariable() requester is  "+requester.getId());
            log.debug("lookupVariable() varname is   >"+variableName+"<");
        }

        if (variableName.startsWith("///"))
        {
            return fetchEngineEnvironment().get(variableName.substring(3));
        }

        if (variableName.startsWith("//"))
        {
            return fetchEngineEnvironment().get(variableName.substring(2));
        }

        if (variableName.startsWith("/"))
        {
            return fetchRootEnvironment(requester)
                .get(variableName.substring(1));
        }

        //return lookupEnvironment(requester).lookup(variableName);

        final Environment e = lookupEnvironment(requester);

        if (e == null)
        {
            log.error
                ("lookupVariable() "+
                 "didn't find environment for  "+requester.getId());
            return null;
        }

        return e.lookup(variableName);
    }

    /**
     * Looks up the environment that contains the given variable name.
     */
    public Environment lookupContainingEnvironment 
        (final FlowExpression requester, final String variableName)
    {
        if (variableName.startsWith("//"))
            return fetchEngineEnvironment();

        if (variableName.startsWith("/"))
            return fetchRootEnvironment(requester);

        final Environment env = lookupEnvironment(requester);

        return env.lookupContainingEnvironment(variableName);
    }

    /**
     * Looks up the value of a variable, but will only look in the local
     * environment, will look in parent environments.
     */
    public Object lookupLocalVariable
        (FlowExpression requester, String variableName)
    {
        final Environment localEnv = lookupEnvironment(requester);

        return localEnv.get(variableName);
    }

    //
    // Schedule METHODS

    /**
     * When the engine restarts, this method roams though the saved expressions
     * and reschedules the one that need it.
     * The expressions get rescheduled for after the complete engine startup.
     */
    protected void reschedule ()
    {
        log.info("reschedule() ...");

        log.debug("reschedule() from pool...");

        java.util.Iterator it = this.contentIterator(Schedulable.class);
        while (it.hasNext())
        {
            final FlowExpression fe = (FlowExpression)it.next();

            // the rawExpression of a Schedulable expression might have
            // been taken into account : let's discard it :
            //
            if ( ! (fe instanceof Schedulable)) continue;

            reschedule(fe);
        }

        log.debug("reschedule() from engine-env...");

        it = fetchEngineEnvironment().getVariables().values().iterator();
        while (it.hasNext())
        {
            final Object o = it.next();

            if ( ! (o instanceof Schedulable)) continue;

            final FlowExpression fe = (FlowExpression)o;
                
            reschedule(fe);
        }

        log.info("reschedule() done.");
    }

    /**
     * Reschedules a particular expression (this method is called by
     * reschedule().
     */
    protected void reschedule (final FlowExpression fe)
    {
        //log.debug("reschedule() fe.class is "+fe.getClass().getName());
            // should always be Schedulable...

        //this.scheduler.scheduleIn
        //    (RESCHEDULE_DELAY, this, new Object[] { o });

        fe.setApplicationContext(getContext());

        ((Schedulable)fe).reschedule(this.getScheduler());

        if (log.isInfoEnabled())
            log.info("reschedule() rescheduled  "+fe.getId());
    }

    /**
     * This method is not needed (it will thus always return -1).
     */
    public Long reschedule (final Scheduler s)
    {
        // nada

        return new Long(-1);
    }

    /**
     * Making the scheduler available for extending children.
     */
    public Scheduler getScheduler ()
    {
        return this.scheduler;
    }

    /**
     * The scheduler uses this method to remind the expression pool about
     * a Schedulable expression instance.
     */
    public void trigger (final Object[] params)
    {
        if (params == null)
        {
            log.debug("trigger() rescheduling...");

            reschedule();
            return;
        }

        final Object o = params[0];

        Schedulable schedulableExpression = null;

        if (log.isDebugEnabled())
            log.debug("trigger() triggering  "+o);

        if (o instanceof Schedulable)
        {
            schedulableExpression = 
                (Schedulable)o;
        }
        else if (o instanceof FlowExpressionId)
        {
            schedulableExpression = 
                (Schedulable)this.fetch((FlowExpressionId)o);
        }
        else
        {
            log.warn
                ("trigger() cannot reschedule instance of "+
                 o.getClass().getName());
            return;
        }

        if (schedulableExpression == null)
        {
            log.warn
                ("trigger() "+
                 "did not find schedulable expression. Forgetting it");
            return;
        }

        if (log.isDebugEnabled())
        {
            log.debug
                ("trigger() for  "+
                 ((FlowExpression)schedulableExpression).getId());
        }

        schedulableExpression.trigger(null);
    }

    /**
     * This method is used by Schedulable flow expressions when reregistering
     * themselves for a further schedule (for instance, an irresolved 'when'
     * will reregister for further evaluation).
     *
     * @return the atId of the sheduled job (useful for cancelling it)
     */
    public Long schedule (final FlowExpressionId fei)
    {
        if (log.isDebugEnabled())
            log.debug("schedule()  "+fei);

        return this.scheduler.scheduleIn
            (getSchedFrequency(), this, new Object[] { fei.copy() });
    }

    /**
     * Schedules a (Schedulable) flow expression for a given 'at' point in
     * time.
     * This is used by the SleepExpression.
     *
     * @return the atId of the sheduled job (useful for cancelling it)
     */
    public Long scheduleAt (long at, FlowExpressionId fei)
    {
        if (log.isDebugEnabled())
            log.debug("scheduleAt() at "+new java.util.Date(at)+"  "+fei);

        return this.scheduler.scheduleAt
            (at, this, new Object[] { fei.copy() });
    }

    /**
     * Returning how long (ms) a when will wait before getting rechecked.
     */
    public long getSchedFrequency ()
    {
        return this.schedFrequency;
    }

    //
    // other METHODS

    /**
     * Returns a string representation of the expression pool.
     */
    public String dumpExpool ()
    {
        final StringBuffer sb = new StringBuffer();

        final java.util.Iterator it = listExpressions().iterator();
        while (it.hasNext())
        {
            final FlowExpression e = (FlowExpression)it.next();

            sb
                .append("- ")
                .append(e.getId().toString())
                .append("\n");

            String sParent = "";
            if (e.getParent() != null) sParent = e.getParent().toString();

            sb
                .append("     p-> ")
                .append(sParent)
                .append("\n");
        }

        return sb.toString();
    }

    /**
     * Returns the state of the flow expression, if it's not state, the
     * NormalState singleton will be returned.
     * This method has been made public and static as it's also used 
     * by the poolGrapher.
     */
    public static ExpressionState getState (final FlowExpression fe)
    {
        ExpressionState state = fe.getState();

        if (state == null)
            state = new NormalState();

        state.setExpression(fe);

        return state;
    }

    /**
     * A shortcut to get the history service.
     */
    protected History getHistory ()
    {
        return Definitions.getHistory(getContext());
    }

    //
    // STATUS

    public org.jdom.Element getStatus ()
    {
        org.jdom.Element result = new org.jdom.Element(getName());

        result.addContent(XmlUtils.getClassElt(this));
        result.addContent(XmlUtils.getRevisionElt("$Id: SimpleExpressionPool.java 2905 2006-07-02 15:17:05Z jmettraux $"));

        //
        // pooled expression count
        
        org.jdom.Element pecElt = new org.jdom.Element("pooledExpressionCount");
        pecElt.addContent(""+getStore().size());

        result.addContent(pecElt);

        return result;
    }

    /**
     * A convenience method for looking up the expression store in the
     * application context
     */
    protected ExpressionStore getStore ()
    {
        if (this.store != null) return this.store;

        this.store = Definitions.getExpressionStore(getContext());

        //log.debug("getStore() first call, pointing to '"+this.store+"'");

        return this.store;
    }

}
