/*
 * Copyright (c) 2005, John Mettraux, OpenWFE.org
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * . Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.  
 * 
 * . Redistributions in binary form must reproduce the above copyright notice, 
 *   this list of conditions and the following disclaimer in the documentation 
 *   and/or other materials provided with the distribution.
 * 
 * . Neither the name of the "OpenWFE" nor the names of its contributors may be
 *   used to endorse or promote products derived from this software without
 *   specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: StringList.java 2808 2006-06-12 21:31:46Z jmettraux $
 */

//
// StringList.java
//
// john.mettraux@openwfe.org
//
// generated with 
// jtmpl 1.1.01 2004/05/19 (john.mettraux@openwfe.org)
//

package openwfe.org.engine.impl.functions;

import openwfe.org.engine.workitem.InFlowWorkItem;
import openwfe.org.engine.expressions.FlowExpression;


/**
 * A string list is used by functions when receiving a string representing
 * a list
 *
 * <p><font size=2>CVS Info :
 * <br>$Author: jmettraux $
 * <br>$Id: StringList.java 2808 2006-06-12 21:31:46Z jmettraux $ </font>
 *
 * @author john.mettraux@openwfe.org
 */
public class StringList
{

    private final static org.apache.log4j.Logger log = org.apache.log4j.Logger
        .getLogger(StringList.class.getName());

    //
    // CONSTANTS & co

    /**
     * This constant is used by car() and cdr() to determine how to split
     * incoming string into lists of strings.<br>
     * When this list separator is set in the flow variables, it used in
     * preference to the default list separator which is ",".
     */
    public final static String BF_LIST_SEPARATOR
        = "__list_separator__";

    private final static String DEFAULT_LIST_SEPARATOR = ",";

    //
    // FIELDS

    private String listSeparator = null;
    private java.util.List list = null;

    //
    // CONSTRUCTORS

    /**
     * Builds a StringList, assuming that the list is the first arg, and that
     * there is potentially a listSep as second arg.
     */
    public StringList
        (final FlowExpression fe, 
         final InFlowWorkItem wi, 
         final String[] args)
    {
        this(fe, wi, args, 0, 1);
    }

    /**
     * Builds a string list for any function
     *
     * @param listIndex indicates which arg (index) is the list
     * @param separatorIndex indicates to this method at which point of the arg
     * array, the separator is supposed to be
     */
    public StringList
        (final FlowExpression fe, 
         final InFlowWorkItem wi, 
         final String[] args,
         int listIndex,
         int separatorIndex)
    {
        this.listSeparator = determineListSeparator(fe, args, separatorIndex);

        if (log.isDebugEnabled())
            log.debug("StringList() separator is >"+this.listSeparator+"<");

        String[] ss = new String[] { };

        if (args != null && args.length > 0)
            ss = args[listIndex].trim().split(this.listSeparator);

        //log.debug("StringList() list length is "+ss.length);

        this.list = new java.util.ArrayList(ss.length);

        for (int i=0; i<ss.length; i++) this.list.add(ss[i].trim());
    }

    //
    // METHODS

    /**
     * Changes the list separator used by this list
     * (will thus have an impact on any [partial] toString call).
     */
    public void setListSeparator (final String sep)
    {
        this.listSeparator = sep;
    }

    public String getListSeparator ()
    {
        return this.listSeparator;
    }

    /**
     * Reorders randomly the list (in place)
     */
    public void shuffle ()
    {
        java.util.Collections.shuffle(this.list);
    }

    /**
     * Reverses the list (in place)
     */
    public void reverse ()
    {
        java.util.Collections.reverse(this.list);
    }

    /**
     * Sorts the list (in place)
     */
    public void sort ()
    {
        java.util.Collections.sort(this.list);
    }

    /**
     * Returns the first element of this list
     */
    public String car ()
    {
        if (this.list.size() < 1) return "";

        return this.list.get(0).toString();
    }

    /**
     * Returns the list without its first element
     */
    public String cdr ()
    {
        if (this.list.size() <= 1) return "";

        return toString(this.list.subList(1, this.list.size()));
    }

    /**
     * Returns a new list (as a String) consisting of the given elt followed
     * by the elt in this list.
     */
    public String cons (final String elt)
    {
        if (this.list.size() < 1) return elt;

        final StringBuffer sb = new StringBuffer();

        sb.append(elt);
        sb.append(this.listSeparator);
        sb.append(toString(this.list));

        return sb.toString();
    }

    /**
     * Returns the element at the given index
     */
    public String elt (final int index)
    {
        return this.list.get(index).toString();
    }

    /**
     * Returns the size of this list.
     */
    public int size ()
    {
        return this.list.size();
    }

    /**
     * Renders the list as a string
     */
    public String toString ()
    {
        return toString(this.listSeparator, this.list);
    }

    /**
     * Renders a sublist with the same listSep as this StringList
     */
    public String toString (final java.util.List sublist)
    {
        return toString(this.listSeparator, sublist);
    }

    //
    // STATIC METHODS

    /*
     * It's static because other lists and listSep could  be given...
     */
    private static String toString 
        (final String listSep, final java.util.List list)
    {
        final StringBuffer sb = new StringBuffer();

        final java.util.Iterator it = list.iterator();
        while (it.hasNext())
        {
            sb.append(it.next().toString());

            if (it.hasNext()) sb.append(listSep);
        }

        return sb.toString();
    }

    /**
     * Determines the list separator (given mainly an array of args)
     *
     * @param sepIndex at which index of the arg array should the separator
     * be found
     */
    public static String determineListSeparator
        (final FlowExpression fe, final String[] args, int sepIndex)
    {
        String listsep = null;

        if (args.length > sepIndex) 
        {
            listsep = args[sepIndex];
        }

        if (listsep == null) 
        {
            listsep = (String)fe.getExpressionPool()
                .lookupVariable(fe, BF_LIST_SEPARATOR);

            /* 
             * some debug & test code
             *
            Object o = fe.getExpressionPool()
                .lookupVariable(fe, BF_LIST_SEPARATOR);

            log.debug("determineListSeparator() it's >"+o.toString()+"<");

            listsep = (String)o;
             *
             */
        }

        if (listsep == null) 
        {
            listsep = DEFAULT_LIST_SEPARATOR;
        }

        if (log.isDebugEnabled())
            log.debug("determineListSeparator() it's >"+listsep+"<");

        return listsep;
    }

}
